/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.api.param;

import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableMap;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.annotation.dsl.xml.ParameterDsl;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;

import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * The definition of the invocation to a stored procedure
 *
 * @since 1.0
 */
public class StoredProcedureCall extends ParameterizedStatementDefinition<StoredProcedureCall> {

  /**
   * A {@link Map} which keys are the name of a parameter to be set on the JDBC prepared statement which is both input and output.
   * <p>
   * Each parameter should be referenced in the sql text using a semicolon prefix (E.g: {@code where id = :myParamName)}).
   * <p>
   * The map's values will contain the actual assignation for each parameter.
   */
  @Parameter
  @Optional
  @NullSafe
  @DisplayName("Input - Output Parameters")
  @ParameterDsl(allowReferences = false)
  protected Map<String, Object> inOutParameters = new LinkedHashMap<>();

  /**
   * A list of output parameters to be set on the JDBC prepared statement. Each parameter should be referenced in the sql text
   * using a semicolon prefix (E.g: {@code call multiply(:value, :result)})
   */
  @Parameter
  @Optional
  @NullSafe
  @DisplayName("Output Parameters")
  @ParameterDsl(allowReferences = false)
  private List<OutputParameter> outputParameters = new LinkedList<>();

  /**
   * Creates a new instance of StoredProcedureCall.
   */
  public StoredProcedureCall() {

  }

  /**
   * Creates a new instance of StoredProcedureCalll with the given arguments.
   * 
   * @param sql              The text of the SQL query to be executed
   * @param parameterTypes   Allows to optionally specify the type of one or more of the parameters in the query. If provided,
   *                         you're not even required to reference all of the parameters, but you cannot reference a parameter not
   *                         present in the input values
   * @param inputParameters  A {@link Map} which keys are the name of an input parameter to be set on the JDBC prepared statement.
   *                         Each parameter should be referenced in the sql text using a semicolon prefix (E.g:
   *                         {@code where id = :myParamName)}).
   *                         <p>
   *                         The map's values will contain the actual assignation for each parameter.
   * @param inOutParameters  * {@link LinkedHashMapMap} which keys are the name of a parameter to be set on the JDBC prepared
   *                         statement which is both input and output.
   *                         <p>
   *                         Each parameter should be referenced in the sql text using a semicolon prefix (E.g:
   *                         {@code where id = :myParamName)}).
   *                         <p>
   *                         The map's values will contain the actual assignation for each parameter.
   * @param outputParameters A list of output parameters to be set on the JDBC prepared statement. Each parameter should be
   *                         referenced in the sql text using a semicolon prefix (E.g: {@code call multiply(:value, :result)})
   * @param queryTimeout     Indicates the minimum amount of time before the JDBC driver attempts to cancel a running statement.
   *                         No timeout is used by default.
   * @param queryTimeoutUnit A {@link TimeUnit} which qualifies the {@link #queryTimeout}
   * @param fetchSize        Indicates how many rows to fetch from the database when rows are read from a resultSet. This property
   *                         is required when streaming is {@code true};
   * @param maxRows          Sets the limit for the maximum number of rows that any ResultSet object generated by this message
   *                         processor can contain for. If the limit is exceeded, the excess rows are silently dropped.
   */
  public StoredProcedureCall(String sql, List<ParameterType> parameterTypes, Map<String, Object> inputParameters,
                             Map<String, Object> inOutParameters, List<OutputParameter> outputParameters,
                             int queryTimeout, TimeUnit queryTimeoutUnit, Integer fetchSize, Integer maxRows) {
    super(sql, parameterTypes, inputParameters, queryTimeout, queryTimeoutUnit, fetchSize, maxRows);
    this.inOutParameters = inOutParameters;
    this.outputParameters = outputParameters;
  }


  @Override
  protected StoredProcedureCall copy() {
    StoredProcedureCall copy = super.copy();
    copy.outputParameters = new LinkedList<>(outputParameters);
    copy.inOutParameters = new LinkedHashMap<>(inOutParameters);

    return copy;
  }

  public java.util.Optional<OutputParameter> getOutputParameter(String name) {
    return outputParameters.stream().filter(p -> p.getKey().equals(name)).findFirst();
  }

  public java.util.Optional<Reference<Object>> getInOutParameter(String name) {
    return findParameter(inOutParameters, name);
  }

  public List<ParameterType> getOutputParameters() {
    return unmodifiableList(outputParameters);
  }

  public Map<String, Object> getInOutParameters() {
    return unmodifiableMap(inOutParameters);
  }

}
