/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.operation;

import org.mule.db.commons.AbstractDbConnector;
import org.mule.db.commons.api.StatementResult;
import org.mule.db.commons.api.exception.connection.BadSqlSyntaxException;
import org.mule.db.commons.api.param.ParameterType;
import org.mule.db.commons.api.param.ParameterizedStatementDefinition;
import org.mule.db.commons.api.param.QueryDefinition;
import org.mule.db.commons.api.param.QuerySettings;
import org.mule.db.commons.internal.domain.autogeneratedkey.AutoGenerateKeysAttributes;
import org.mule.db.commons.internal.domain.autogeneratedkey.AutoGenerateKeysStrategy;
import org.mule.db.commons.internal.domain.autogeneratedkey.ColumnNameAutoGenerateKeysStrategy;
import org.mule.db.commons.internal.domain.autogeneratedkey.ColumnIndexAutoGenerateKeysStrategy;
import org.mule.db.commons.internal.domain.autogeneratedkey.DefaultAutoGenerateKeysStrategy;
import org.mule.db.commons.internal.domain.autogeneratedkey.NoAutoGenerateKeysStrategy;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.executor.QueryExecutor;
import org.mule.db.commons.internal.domain.executor.UpdateExecutor;
import org.mule.db.commons.internal.domain.param.QueryParam;
import org.mule.db.commons.internal.domain.query.Query;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.db.commons.internal.domain.query.QueryType;
import org.mule.db.commons.internal.domain.statement.ConfigurableStatementFactory;
import org.mule.db.commons.internal.domain.statement.QueryStatementFactory;
import org.mule.db.commons.internal.resolver.query.ParameterizedQueryResolver;
import org.mule.db.commons.internal.resolver.query.QueryResolver;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.slf4j.Logger;

import java.sql.SQLException;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import static java.lang.Math.min;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Base class with common functionality for Database operations
 *
 * @since 1.0
 */
public abstract class BaseDbOperations {

  public static final int DEFAULT_FETCH_SIZE = 10;
  public static final String QUERY_GROUP = "Query";
  public static final String QUERY_SETTINGS = "Query Settings";

  protected final QueryResolver<ParameterizedStatementDefinition> queryResolver;
  protected final ConfigurableStatementFactory statementFactory;
  protected Function<ConfigurableStatementFactory, QueryExecutor> updateExecutor;
  private static final Logger LOGGER = getLogger(BaseDbOperations.class);

  protected BaseDbOperations(QueryResolver<ParameterizedStatementDefinition> queryResolver,
                             ConfigurableStatementFactory statementFactory,
                             Function<ConfigurableStatementFactory, QueryExecutor> updateExecutor) {
    this.queryResolver = queryResolver;
    this.statementFactory = statementFactory;
    this.updateExecutor = updateExecutor;
  }

  protected static QueryResolver<ParameterizedStatementDefinition> getDefaultQueryResolver() {
    return new ParameterizedQueryResolver<>();
  }

  protected static ConfigurableStatementFactory getDefaultStatementFactory() {
    return new QueryStatementFactory();
  }

  protected static Function<ConfigurableStatementFactory, QueryExecutor> getDefaultUpdateExecutor() {
    return UpdateExecutor::new;
  }

  protected ConfigurableStatementFactory getStatementFactory(QuerySettings settings) {
    final int maxRows = settings.getMaxRows() != null ? settings.getMaxRows() : 0;
    int fetchSize = getFetchSize(settings);

    if (maxRows > 0) {
      fetchSize = min(fetchSize, maxRows);
    }
    if (maxRows >= 0) {
      statementFactory.setMaxRows(maxRows);
    }

    statementFactory.setFetchSize(fetchSize);
    statementFactory.setQueryTimeout(timeoutToSeconds(settings.getQueryTimeout(), settings.getQueryTimeoutUnit()));

    return statementFactory;
  }

  private static int timeoutToSeconds(int queryTimeout, TimeUnit sourceUnit) {
    long timeoutInSeconds = sourceUnit.toSeconds(queryTimeout);

    // Rounding up is more sensible for timeouts than rounding up.
    // In particular, as zero has a special meaning (no timeout) rounding to zero when
    // the user specified a very small value is wrong because the user intended to have
    // a very strict timeout and will get no timeout at all!

    if (sourceUnit.convert(timeoutInSeconds, TimeUnit.SECONDS) < queryTimeout) {
      LOGGER.warn("Timeout are rounded up to the next second.");
      timeoutInSeconds++;
    }

    return Math.toIntExact(timeoutInSeconds);
  }

  protected StatementResult executeUpdate(QueryDefinition query,
                                          AutoGenerateKeysAttributes autoGenerateKeysAttributes,
                                          DbConnection connection,
                                          Query resolvedQuery)
      throws SQLException {
    ConfigurableStatementFactory localStatementFactory = getStatementFactory(query);

    return (StatementResult) updateExecutor.apply(localStatementFactory)
        .execute(connection, resolvedQuery, getAutoGeneratedKeysStrategy(autoGenerateKeysAttributes));
  }

  // TODO see if an abstract resolveQuery method can be created, capable of managing generic QueryResolvers
  protected Query resolveQuery(ParameterizedStatementDefinition<?> query,
                               AbstractDbConnector connector,
                               DbConnection connection,
                               StreamingHelper streamingHelper,
                               QueryType... validTypes) {

    final Query resolvedQuery =
        queryResolver.resolve(query, connector, connection, streamingHelper);
    validateQueryType(resolvedQuery.getQueryTemplate(), asList(validTypes));
    validateNoParameterTypeIsUnused(resolvedQuery, query.getParameterTypes());
    return resolvedQuery;
  }

  protected void validateQueryType(QueryTemplate queryTemplate, List<QueryType> validTypes) {
    if (validTypes == null || !validTypes.contains(queryTemplate.getType())) {
      String typeList = validTypes.stream().map(QueryType::name).collect(joining(", "));
      throw new BadSqlSyntaxException(format("Query type must be one of [%s] but query '%s' is of type '%s'",
                                             typeList, queryTemplate.getSqlText(),
                                             queryTemplate.getType()));
    }
  }

  protected void validateNoParameterTypeIsUnused(Query query, List<ParameterType> parameterTypes) {
    Set<String> params = query.getQueryTemplate().getParams().stream().map(QueryParam::getName).collect(toSet());
    Set<String> unusedTypes =
        parameterTypes.stream().map(ParameterType::getKey).filter(type -> !params.contains(type)).collect(toSet());

    if (!unusedTypes.isEmpty()) {
      String msg = unusedTypes.stream()
          .map(s -> new StringBuilder().append("'").append(s).append("'").toString())
          .collect(joining(", "));
      throw new IllegalArgumentException(format("Query defines parameters [%s] but they aren't present in the query", msg));
    }
  }

  protected AutoGenerateKeysStrategy getAutoGeneratedKeysStrategy(AutoGenerateKeysAttributes keyAttributes) {

    if (keyAttributes == null) {
      return new NoAutoGenerateKeysStrategy();
    }

    if (keyAttributes.isAutoGenerateKeys()) {

      final List<Integer> columnIndexes = keyAttributes.getAutoGeneratedKeysColumnIndexes();
      final List<String> columnNames = keyAttributes.getAutoGeneratedKeysColumnNames();

      if (!isEmpty(columnIndexes)) {
        int[] indexes = new int[columnIndexes.size()];
        int i = 0;
        for (int index : columnIndexes) {
          indexes[i++] = index;
        }
        return new ColumnIndexAutoGenerateKeysStrategy(indexes);
      } else if (!isEmpty(columnNames)) {
        return new ColumnNameAutoGenerateKeysStrategy(columnNames.stream().toArray(String[]::new));
      } else {
        return new DefaultAutoGenerateKeysStrategy();
      }
    } else {
      return new NoAutoGenerateKeysStrategy();
    }
  }

  protected int getFetchSize(QuerySettings settings) {
    Integer fetchSize = settings.getFetchSize();
    return fetchSize != null ? fetchSize : DEFAULT_FETCH_SIZE;
  }

}
