/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.type;

import static java.sql.Types.DATE;
import static java.sql.Types.DECIMAL;
import static java.sql.Types.NUMERIC;
import static java.sql.Types.TIMESTAMP;
import static org.mule.db.commons.internal.domain.type.UnknownDbType.createArraysAndStructs;

import org.mule.db.commons.internal.domain.connection.DbConnection;

import java.math.BigDecimal;
import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;

/**
 * Defines a data type that was resolved for a database instance
 */
public class ResolvedDbType extends AbstractDbType {

  private static final int ORACLE_TIMESTAMPLTZ_ID = -102;
  private static final String ORACLE_JDBC_PREFIX = "oracle";
  private boolean statementIsComplete;

  public ResolvedDbType(int id, String name) {
    super(id, name);
  }

  @Override
  public void setParameterValue(PreparedStatement statement, int index, Object value, DbConnection connection)
      throws SQLException {
    if (value == null) {
      statement.setNull(index, id);
      return;
    }

    value = createArraysAndStructs(value, connection);
    statementIsComplete = false;
    // If special treatment is needed for a specific vendor, you can add it in this section.
    if (isValidDriverName(statement)
        && statement.getConnection().getMetaData().getDriverName().toLowerCase().contains(ORACLE_JDBC_PREFIX.toLowerCase())) {
      oracleResolveParameterValue(statement, index, value);
    }
    // This block corresponds to the section common to all providers, so any modification will affect them
    // Make sure to include tests that cover the scenario you are including for each provider.
    if (!statementIsComplete) {
      if (DECIMAL == id || NUMERIC == id) {
        if (value instanceof BigDecimal) {
          statement.setObject(index, value, id, ((BigDecimal) value).scale());
        } else if (value instanceof Float || value instanceof Double) {
          BigDecimal bigDecimal = new BigDecimal(value.toString());
          statement.setObject(index, bigDecimal, id, bigDecimal.scale());
        } else if (value instanceof String) {
          BigDecimal bigDecimal = new BigDecimal((String) value);
          statement.setObject(index, bigDecimal, id, bigDecimal.scale());
        } else {
          statement.setObject(index, value, id);
        }
      } else {
        statement.setObject(index, value, id);
      }
    }
  }

  @Override
  public Object getParameterValue(CallableStatement statement, int index) throws SQLException {
    return statement.getObject(index);
  }

  private void oracleResolveParameterValue(PreparedStatement statement, int index, Object value) throws SQLException {
    if (id == ORACLE_TIMESTAMPLTZ_ID || id == DATE || id == TIMESTAMP) {
      statement.setObject(index, value);
      this.statementIsComplete = true;
    }
  }

  private boolean isValidDriverName(PreparedStatement statement) throws SQLException {
    return (statement != null && statement.getConnection() != null && statement.getConnection().getMetaData() != null
        && statement.getConnection().getMetaData().getDriverName() != null);
  }
}
