/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.util;

import org.mule.db.commons.api.exception.connection.BadSqlSyntaxException;
import org.mule.runtime.api.util.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;

import java.sql.SQLException;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility class for stored procedure operations.
 *
 * @since 1.4.1
 */
public class StoredProcedureUtils {

  private static final String STORED_PROCEDURE_REGEX =
      "(?msi)(\\{\\s*)?call\\s+([\\[\\]\\w]+\\.)?([\\[\\]\\w]+\\.)?([\\[\\]\\w\\$]+)\\s*\\(.*";
  private static final String FUNCTION_REGEX =
      "(?msi)(\\{\\?\\s*)?=\\s*call\\s+([\\[\\]\\w]+\\.)?([\\[\\]\\w]+\\.)?([\\[\\]\\w]+)\\s*\\(.*";

  private static final Pattern storedProcedurePattern = Pattern.compile(STORED_PROCEDURE_REGEX);
  private static final Pattern functionPattern = Pattern.compile(FUNCTION_REGEX);

  private static final Logger LOGGER = LoggerFactory.getLogger(StoredProcedureUtils.class);

  /**
   * Gets the name of the stored procedure of the given SQL Query.
   *
   * @param sqlText the SQL Query text
   * @return the name of the stored procedure
   * @throws SQLException if it is no possible to get name of the stored procedure from the given SQL Query or the SQL Query
   *                      syntax is not valid
   */
  public static Pair<String, Boolean> analyzeStoredOperation(String sqlText) throws BadSqlSyntaxException {
    Pair<Matcher, Boolean> result = analyzeSqlText(sqlText, "name");

    return new Pair<>(result.getFirst().group(4), result.getSecond());
  }

  /**
   * Gets the owner of the stored procedure of the given SQL Query. Normally the owner will be a schema, although in case of
   * Oracle it can be a package.
   *
   * @param sqlText the SQL Query text
   * @return an {@link Optional} with the Schema of the stored procedure
   * @throws SQLException if the SQL Query syntax is not valid
   */
  public static Optional<String> getStoreProcedureOwner(String sqlText) throws BadSqlSyntaxException {
    Matcher matcher = analyzeSqlText(sqlText, "schema").getFirst();

    String firstPart = matcher.group(2);
    String secondPart = matcher.group(3);

    if (!isBlank(firstPart) && !isBlank(secondPart)) {
      String packageName = firstPart.substring(0, firstPart.length() - 1);
      return Optional.of(packageName);
    } else if (!isBlank(firstPart)) {
      String packageName = firstPart.substring(0, firstPart.length() - 1);
      return Optional.of(packageName);
    } else if (!isBlank(secondPart)) {
      String packageName = secondPart.substring(0, secondPart.length() - 1);
      return Optional.of(packageName);
    } else {
      return Optional.empty();
    }
  }

  /**
   * Gets the owner of the owner of the stored procedure of the given SQL Query. This has sense only for Oracle, where Stored
   * procedures can be defined within packages which in turn will be defined under a schema.
   *
   * @param sqlText the SQL Query text
   * @return an {@link Optional} with the package of the stored procedure
   * @throws SQLException if the SQL Query syntax is not valid
   */
  public static Optional<String> getStoredProcedureParentOwner(String sqlText) throws BadSqlSyntaxException {
    Matcher matcher = analyzeSqlText(sqlText, "package").getFirst();

    String firstPart = matcher.group(2);
    String secondPart = matcher.group(3);

    if (!isBlank(firstPart) && !isBlank(secondPart)) {
      String packageName = secondPart.substring(0, secondPart.length() - 1);
      return Optional.of(packageName);
    } else {
      return Optional.empty();
    }
  }

  private static Pair<Matcher, Boolean> analyzeSqlText(String sqlText, String stage) throws BadSqlSyntaxException {
    boolean isFunction = false;
    Matcher matcher = storedProcedurePattern.matcher(sqlText);

    if (!matcher.matches()) {
      String exception =
          format("Unable to detect stored procedure '%s' from '%s'. Looking for function '%s'.", stage, sqlText, stage);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(exception);
      }

      matcher = functionPattern.matcher(sqlText);

      if (!matcher.matches()) {
        exception = format("Unable to detect function '%s' from '%s'.", stage, sqlText);

        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug(exception);
        }

        throw new BadSqlSyntaxException(exception);
      }

      isFunction = true;
    }

    return new Pair<>(matcher, isFunction);
  }

}
