/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.api;

import java.io.Serializable;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.RandomAccess;

/**
 * Represents an array or a struct that will be sent to a database.
 */
public class DbAggregate extends AbstractList<Object> implements Serializable, RandomAccess {

  private static final long serialVersionUID = 1L;

  /**
   * This can either be the type's name or the item type name depending on the target database.
   */
  private final String typeName;

  private final List<Object> elements;

  private final Kind kind;

  public enum Kind {
    /**
     * An SQL array.
     *
     * Standard JDBC only supports this.
     */
    ARRAY,
    /**
     * An SQL struct.
     */
    STRUCT
  }

  public DbAggregate(Kind kind, String typeName, List<Object> elements) {
    this.kind = kind;
    this.typeName = typeName;
    this.elements = Collections.unmodifiableList(new ArrayList<>(elements));
  }

  public Kind getKind() {
    return kind;
  }

  /**
   * This can either be the type's name or the item type name.
   *
   * For bases that support JDBC's createArrayOf this is the type of the elements in the array, for bases that don't (i.e. Oracle)
   * is the type of the collection.
   */
  public String getTypeName() {
    return typeName;
  }

  public List<Object> getElements() {
    return elements;
  }

  @Override
  public Object get(int index) {
    return elements.get(index);
  }

  @Override
  public int size() {
    return elements.size();
  }

  // ArrayList's toArray() implementation is more efficient than AbstractList's.
  @Override
  public Object[] toArray() {
    return elements.toArray();
  }

  // ArrayList's toArray() implementation is more efficient than AbstractList's.
  @Override
  public <T> T[] toArray(T[] a) {
    return elements.toArray(a);
  }
}
