/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.operation;

import org.mule.db.commons.AbstractDbConnector;
import org.mule.db.commons.api.param.ParameterizedStatementDefinition;
import org.mule.db.commons.api.param.QueryDefinition;
import org.mule.db.commons.api.param.QuerySettings;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.executor.QueryExecutor;
import org.mule.db.commons.internal.domain.query.Query;
import org.mule.db.commons.internal.domain.statement.ConfigurableStatementFactory;
import org.mule.db.commons.internal.resolver.query.QueryResolver;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.sql.SQLException;
import java.util.Optional;
import java.util.function.Function;

import static java.util.Optional.of;
import static org.mule.db.commons.internal.domain.query.QueryType.DDL;

/**
 * Operations to manipulate data definitions in a relational Database
 * 
 * @since 1.0
 */
public class DdlOperations extends BaseDbOperations {

  private DdlOperations(QueryResolver<ParameterizedStatementDefinition> queryResolver,
                        ConfigurableStatementFactory statementFactory,
                        Function<ConfigurableStatementFactory, QueryExecutor> updateExecutor) {
    super(queryResolver, statementFactory, updateExecutor);
  }

  /**
   * Enables execution of DDL queries against a database.
   *
   * @param sql        The text of the SQL query to be executed
   * @param settings   Parameters to configure the query
   * @param connector  the acting connector
   * @param connection the acting connection
   * @return the number of affected rows
   */
  public int executeDdl(String sql,
                        QuerySettings settings,
                        AbstractDbConnector connector,
                        DbConnection connection,
                        StreamingHelper streamingHelper)
      throws SQLException {

    QueryDefinition query = new QueryDefinition();
    query.setSql(sql);
    query.copyInto(settings);

    final Query resolvedQuery = resolveQuery(query, connector, connection, streamingHelper, DDL);
    return executeUpdate(query, null, connection, resolvedQuery).getAffectedRows();
  }

  public static class Builder {

    private Optional<QueryResolver<ParameterizedStatementDefinition>> queryResolverOptional = Optional.empty();
    private Optional<ConfigurableStatementFactory> statementFactoryOptional = Optional.empty();
    private Optional<Function<ConfigurableStatementFactory, QueryExecutor>> updateExecutorOptional = Optional.empty();

    /**
     * Defines the kind of QueryResolver to resolve the query.
     * 
     * @param queryResolver An instance of a class that implements QueryResolver
     * @return The builder instance.
     */
    public DdlOperations.Builder withQueryResolver(QueryResolver<ParameterizedStatementDefinition> queryResolver) {
      this.queryResolverOptional = Optional.of(queryResolver);
      return this;
    }

    public DdlOperations.Builder withStatementFactory(ConfigurableStatementFactory statementFactory) {
      this.statementFactoryOptional = Optional.of(statementFactory);
      return this;
    }

    public DdlOperations.Builder withUpdateExecutor(Function<ConfigurableStatementFactory, QueryExecutor> updateExecutor) {
      this.updateExecutorOptional = of(updateExecutor);
      return this;
    }

    /**
     * Builds a DdlInstance using the provided / default arguments.
     * 
     * @return DdlOperations Instance.
     */
    public DdlOperations build() {
      return new DdlOperations(queryResolverOptional.orElse(getDefaultQueryResolver()),
                               statementFactoryOptional.orElse(getDefaultStatementFactory()),
                               updateExecutorOptional.orElse(getDefaultUpdateExecutor()));
    }
  }

}
