/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.exception;

import org.mule.db.commons.api.exception.connection.BadSqlSyntaxException;
import org.mule.db.commons.api.exception.connection.QueryExecutionException;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.exception.ExceptionHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.SQLException;
import java.sql.SQLSyntaxErrorException;
import java.util.Arrays;

/**
 * Translates {@link SQLException} into connector specific ones.
 *
 * @since 1.0
 */
// TODO - MULE-13798: Improve DB Error types by using SQLException hierarchy and SQLState
public class DbExceptionHandler extends ExceptionHandler {

  private static final Logger LOGGER = LoggerFactory.getLogger(DbExceptionHandler.class);

  @Override
  public Exception enrichException(Exception e) {
    if (e instanceof ModuleException) {
      return e;
    }

    return getCauseOfType(e, SQLSyntaxErrorException.class)
        .map(cause -> (Exception) new BadSqlSyntaxException(e.getMessage(), e))
        .orElseGet(() -> getCauseOfType(e, SQLException.class)
            .map(sqlException -> {
              String sqlState = sqlException.getSQLState();
              String message = sqlException.getMessage();

              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("SQLState is: {}, ErrorCode is: {}, Cause is: {}", sqlState, sqlException.getErrorCode(),
                             sqlException.getCause());
              }

              if (e instanceof ConnectionException || isConnectionExceptionByState(sqlState)
                  || isConnectionExceptionByMessage(message)) {
                if (LOGGER.isDebugEnabled()) {
                  LOGGER.debug("Exception is related to the Connection. Throwing ConnectionException. Exception: {}",
                               message);
                }

                return new ConnectionException(message, sqlException);
              }

              if (isBadSyntaxExceptionByState(sqlState)) {
                if (LOGGER.isDebugEnabled()) {
                  LOGGER.debug("Exception is related to the sql' syntax. Throwing BadSqlSyntaxException. Exception: {}",
                               message);
                }
                return new BadSqlSyntaxException(message, sqlException);
              }

              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Defaulting to QueryExecutionException. Exception: {}", message);
              }

              return new QueryExecutionException(message, sqlException);
            })
            .orElse(e));
  }

  private boolean isConnectionExceptionByState(String sqlState) {
    switch (sqlState == null ? "" : sqlState) {
      case "08000":

      case "08001":

      case "08S01":

      case "08002":

      case "08003":

      case "08004":

      case "08006":
        return true;

      default:
        return false;
    }
  }

  private boolean isConnectionExceptionByMessage(String message) {
    message = message == null ? "" : message;

    String[] connectionMessages = new String[] {"The connection is closed", "Connection timed out"};

    return Arrays.stream(connectionMessages).anyMatch(message::contains);
  }

  private boolean isBadSyntaxExceptionByState(String sqlState) {
    return "S0001".equals(sqlState);
  }

}
