/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.oauth.http;

import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.mock;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAUTH_CLIENT;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAuthClientStory.OAUTH_DANCER;
import static org.mule.tck.probe.PollingProber.check;

import io.qameta.allure.Story;

import org.mule.oauth.client.api.http.HttpClientFactory;
import org.mule.oauth.client.internal.http.DefaultHttpClientFactory;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;

import io.qameta.allure.Feature;
import org.junit.Before;
import org.junit.Test;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

@Feature(OAUTH_CLIENT)
@Story(OAUTH_DANCER)
public class DefaultHttpClientFactoryTestCase {

  private static final int PROBER_TIMEOUT = 5000;
  private static final int PROBER_FREQ = 500;

  private AtomicBoolean clientStarted;
  private Latch startLatch;
  private List<Boolean> startedChecks;

  private TlsContextFactory tlsContextFactory;
  private ProxyConfig proxyConfig;
  private HttpClientFactory oAuthHttpClientFactory;
  private HttpClient httpClient;

  @Before
  public void setup() {
    tlsContextFactory = mock(TlsContextFactory.class);
    proxyConfig = mock(ProxyConfig.class);

    org.mule.runtime.http.api.client.HttpClientFactory httpClientFactory =
        mock(org.mule.runtime.http.api.client.HttpClientFactory.class);
    httpClient = mock(HttpClient.class);
    when(httpClientFactory.create(any())).thenReturn(httpClient);
    HttpService httpService = mock(HttpService.class);
    when(httpService.getClientFactory()).thenReturn(httpClientFactory);

    oAuthHttpClientFactory = new DefaultHttpClientFactory(httpService);
    clientStarted = new AtomicBoolean(false);
    startLatch = new Latch();
    startedChecks = new ArrayList<>();
  }

  @Test
  public void createHttpClientWithSameParameterReturnCachedHttpClient() {
    HttpClient client = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(client, is(notNullValue()));

    HttpClient cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(cachedHttpClient, is(client));
  }

  @Test
  public void createHttpClientWithDifferentParametersReturnCachedHttpClient() {
    HttpClient client = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(client, is(notNullValue()));

    HttpClient cachedHttpClient = oAuthHttpClientFactory.create(null, null);
    assertThat(cachedHttpClient, is(not(client)));

    cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, null);
    assertThat(cachedHttpClient, is(not(client)));

    cachedHttpClient = oAuthHttpClientFactory.create(null, proxyConfig);
    assertThat(cachedHttpClient, is(not(client)));

    cachedHttpClient = oAuthHttpClientFactory.create(mock(TlsContextFactory.class), proxyConfig);
    assertThat(cachedHttpClient, is(not(client)));
  }

  @Test
  public void stoppingCachedHttpClientShouldReturnANewHttpClient() {
    HttpClient client = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(client, is(notNullValue()));

    HttpClient cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(cachedHttpClient, is(client));
    client.start();
    client.stop();

    cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(cachedHttpClient, is(not(client)));
  }

  @Test
  public void callingStartInTheClientMustWaitForItToBeStarted() throws InterruptedException {
    doAnswer(invocation -> {
      startLatch.await(3000, TimeUnit.MILLISECONDS);
      clientStarted.set(true);
      return null;
    }).when(httpClient).start();
    Thread callOne = new Thread(() -> getAndStartClient());
    Thread callTwo = new Thread(() -> getAndStartClient());
    callOne.start();
    callTwo.start();
    check(PROBER_TIMEOUT, PROBER_FREQ, () -> startedChecks.size() == 2);
    assertThat(startedChecks.get(0), is(true));
    assertThat(startedChecks.get(1), is(true));

  }

  private void getAndStartClient() {
    HttpClient client = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    client.start();
    startLatch.release();
    startedChecks.add(clientStarted.get());
  }

}
