/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.oauth.authorizationcode;

import static org.mule.oauth.client.api.builder.ClientCredentialsLocation.BASIC_AUTH_HEADER;
import static org.mule.oauth.client.api.builder.ClientCredentialsLocation.BODY;
import static org.mule.oauth.client.api.builder.ClientCredentialsLocation.QUERY_PARAMS;
import static org.mule.oauth.client.api.state.ResourceOwnerOAuthContext.DEFAULT_RESOURCE_OWNER_ID;
import static org.mule.oauth.client.internal.OAuthConstants.CODE_PARAMETER;
import static org.mule.oauth.client.internal.OAuthConstants.REDIRECT_URI_PARAMETER;
import static org.mule.oauth.client.internal.OAuthConstants.STATE_PARAMETER;
import static org.mule.oauth.client.internal.state.StateEncoder.ON_COMPLETE_REDIRECT_TO_PARAM_NAME_ASSIGN;
import static org.mule.oauth.client.internal.state.StateEncoder.RESOURCE_OWNER_PARAM_NAME_ASSIGN;
import static org.mule.runtime.api.metadata.TypedValue.of;
import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.BAD_REQUEST;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.MOVED_TEMPORARILY;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.LOCATION;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAuthClientStory.AUTHORIZATION_CODE;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAuthClientStory.BODY_PARAMS;

import static java.lang.String.format;
import static java.net.URLEncoder.encode;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.singleton;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static org.apache.commons.io.IOUtils.toInputStream;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.hasKey;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.text.IsEqualIgnoringCase.equalToIgnoringCase;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentCaptor.forClass;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.oauth.client.api.AuthorizationCodeOAuthDancer;
import org.mule.oauth.client.api.builder.OAuthAuthorizationCodeDancerBuilder;
import org.mule.oauth.client.api.listener.AuthorizationCodeListener;
import org.mule.oauth.client.api.state.ResourceOwnerOAuthContext;
import org.mule.oauth.client.api.state.ResourceOwnerOAuthContextWithRefreshState;
import org.mule.oauth.client.internal.builder.DefaultOAuthAuthorizationCodeDancerBuilder;
import org.mule.runtime.api.el.MuleExpressionLanguage;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.runtime.test.oauth.AbstractOAuthTestCase;
import org.mule.runtime.test.oauth.state.CustomResourceOwnerOAuthContext;
import org.mule.tck.SimpleUnitTestSupportSchedulerService;

import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.mockito.ArgumentCaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@RunWith(Parameterized.class)
@Story(AUTHORIZATION_CODE)
public class AuthorizationCodeTokenTestCase extends AbstractOAuthTestCase {

  private static final Logger LOGGER = LoggerFactory.getLogger(AuthorizationCodeTokenTestCase.class);

  @Parameters
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {(Supplier<ResourceOwnerOAuthContext>) (() -> {
          final CustomResourceOwnerOAuthContext context =
              new CustomResourceOwnerOAuthContext(new ReentrantLock(), DEFAULT_RESOURCE_OWNER_ID);
          context.setRefreshToken("refreshToken");
          return context;
        })},
        {(Supplier<ResourceOwnerOAuthContext>) (() -> {
          final ResourceOwnerOAuthContextWithRefreshState context =
              new ResourceOwnerOAuthContextWithRefreshState(DEFAULT_RESOURCE_OWNER_ID);
          context.setRefreshToken("refreshToken");
          return context;
        })}
    });
  }

  private final ArgumentCaptor<RequestHandler> localCallbackCaptor = forClass(RequestHandler.class);
  private final ArgumentCaptor<RequestHandler> authCallbackCaptor = forClass(RequestHandler.class);
  private AuthorizationCodeOAuthDancer minimalDancer;

  private final Supplier<ResourceOwnerOAuthContext> contextSupplier;

  public AuthorizationCodeTokenTestCase(Supplier<ResourceOwnerOAuthContext> contextSupplier) {
    this.contextSupplier = contextSupplier;
  }

  @Before
  public void before() {
    when(httpServer.addRequestHandler(eq(singleton(GET.name())), eq("/localCallback"), localCallbackCaptor.capture()))
        .thenReturn(mock(RequestHandlerManager.class));
    when(httpServer.addRequestHandler(eq(singleton(GET.name())), eq("/auth"), authCallbackCaptor.capture()))
        .thenReturn(mock(RequestHandlerManager.class));
  }

  @After
  public void after() throws MuleException {
    if (minimalDancer != null) {
      stopIfNeeded(minimalDancer);
      disposeIfNeeded(minimalDancer, LOGGER);
    }
  }

  private void assertAuthCodeCredentialsEncodedInHeader(boolean useDeprecatedMethod) throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.clientCredentials("Aladdin", "open sesame");
    if (useDeprecatedMethod) {
      builder.encodeClientCredentialsInBody(false);
    } else {
      builder.withClientCredentialsIn(BASIC_AUTH_HEADER);
    }

    minimalDancer = startDancer(builder);
    localCallbackCaptor.getValue().handleRequest(buildLocalCallbackRequestContext(), mock(HttpResponseReadyCallback.class));

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getHeaderValue(AUTHORIZATION), is("Basic QWxhZGRpbjpvcGVuIHNlc2FtZQ=="));

    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_id")));
    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_secret")));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("code=authCode"));
    assertThat(requestBody, containsString("grant_type=authorization_code"));
    assertThat(requestBody, not(containsString("client_secret=open+sesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Test
  public void requestWithNullAuthorizationCode() throws Exception {
    URI testUri = new URI("http://host/testUri");

    HttpResponse response = runCallback(requestMock -> {
      when(requestMock.getQueryParams()).thenReturn(emptyMultiMap());
      when(requestMock.getUri()).thenReturn(testUri);
    }, localCallbackCaptor);

    assertThat(response.getStatusCode(), is(BAD_REQUEST.getStatusCode()));
    assertThat(IOUtils.toString(response.getEntity().getContent(), UTF_8),
               is("Failure retrieving access token.\n OAuth Server uri from callback: " + testUri));
  }

  @Test
  public void requestWithNullAuthorizationCodeAndRedirectOnComplete() throws Exception {
    String redirectUrl = "http://host:12/path";

    HttpResponse response = runCallback(requestMock -> {
      MultiMap<String, String> queryParams = new MultiMap<>();
      queryParams.put(STATE_PARAMETER, ON_COMPLETE_REDIRECT_TO_PARAM_NAME_ASSIGN + redirectUrl);
      when(requestMock.getQueryParams()).thenReturn(queryParams);
    }, localCallbackCaptor);

    assertThat(response.getStatusCode(), is(MOVED_TEMPORARILY.getStatusCode()));
    assertThat(response.getHeaderValue(LOCATION), is(redirectUrl + "?authorizationStatus=100"));
  }

  @Test
  public void exceptionHandlingRequest() throws Exception {
    String testError = "Testing error";

    HttpResponse response = runCallback(requestMock -> when(requestMock.getQueryParams()).then(invocationOnMock -> {
      throw new Exception(testError);
    }), localCallbackCaptor);

    assertThat(response.getStatusCode(), is(INTERNAL_SERVER_ERROR.getStatusCode()));
    assertThat(IOUtils.toString(response.getEntity().getContent(), UTF_8), is(testError));
  }

  @Test
  public void localAuthorizationRequest() throws Exception {
    runLocalAuthorizationCallback();
  }

  @Test
  public void localAuthorizationRequestWithResourceOwnerId() throws Exception {
    String testResourceOwnerId = "testResourceOwnerId";
    OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.localAuthorizationUrlResourceOwnerId(testResourceOwnerId);
    HttpResponse response = runLocalAuthorizationCallback(builder);

    assertThat(response.getHeaderValue(LOCATION),
               containsString(format("&%s=", STATE_PARAMETER)
                   + encode(RESOURCE_OWNER_PARAM_NAME_ASSIGN + testResourceOwnerId, UTF_8)));
  }

  @Test
  public void localAuthorizationRequestWithOriginalStateAndResourceOwnerId() throws Exception {
    String testOriginalState = "testOriginalState";
    String testResourceOwnerId = "testResourceOwnerId";
    OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.state(testOriginalState);
    builder.localAuthorizationUrlResourceOwnerId(testResourceOwnerId);
    HttpResponse response = runLocalAuthorizationCallback(builder);

    assertThat(response.getHeaderValue(LOCATION),
               containsString(format("&%s=%s", STATE_PARAMETER, testOriginalState)
                   + encode(RESOURCE_OWNER_PARAM_NAME_ASSIGN + testResourceOwnerId, UTF_8)));
  }

  @Test
  public void localAuthorizationRequestWithAndResourceOwnerIdStateAsExpression() throws Exception {
    MuleExpressionLanguage muleExpressionLanguage = mock(MuleExpressionLanguage.class);
    String testOriginalStateExpression = "#[attributes.queryParams.state]";
    String resolvedOriginalState = "resolvedOriginalState";
    doReturn(of(resolvedOriginalState)).when(muleExpressionLanguage).evaluate(eq(testOriginalStateExpression), any(),
                                                                              any());
    String testResourceOwnerIdExpression = "#[attributes.queryParams.resourceOwnerId]";
    String resolvedResourceOwnerId = "resolvedOriginalState";
    doReturn(of(resolvedResourceOwnerId)).when(muleExpressionLanguage).evaluate(eq(testResourceOwnerIdExpression), any(),
                                                                                any());
    doReturn(true).when(muleExpressionLanguage).isExpression(any());
    OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder(muleExpressionLanguage);
    builder.state(testOriginalStateExpression);
    builder.localAuthorizationUrlResourceOwnerId(testResourceOwnerIdExpression);
    HttpResponse response = runLocalAuthorizationCallback(builder);

    assertThat(response.getHeaderValue(LOCATION),
               containsString(format("&%s=%s", STATE_PARAMETER, resolvedOriginalState)
                   + encode(RESOURCE_OWNER_PARAM_NAME_ASSIGN + resolvedResourceOwnerId, UTF_8)));
  }

  @Test
  public void localAuthorizationRequestWithRedirect() throws Exception {
    String redirectUrl = "http://host:12/path";

    MultiMap<String, String> queryParams = new MultiMap<>();
    queryParams.put("onCompleteRedirectTo", redirectUrl);
    HttpResponse response = runLocalAuthorizationCallback(baseAuthCodeDancerbuilder(), queryParams);

    assertThat(response.getHeaderValue(LOCATION),
               containsString(format("&%s=", STATE_PARAMETER)
                   + encode(ON_COMPLETE_REDIRECT_TO_PARAM_NAME_ASSIGN + redirectUrl, UTF_8)));
  }

  private void runLocalAuthorizationCallback() throws Exception {
    runLocalAuthorizationCallback(baseAuthCodeDancerbuilder());
  }

  private HttpResponse runLocalAuthorizationCallback(OAuthAuthorizationCodeDancerBuilder builder) throws Exception {
    return runLocalAuthorizationCallback(builder, emptyMultiMap());
  }

  private HttpResponse runLocalAuthorizationCallback(OAuthAuthorizationCodeDancerBuilder builder,
                                                     MultiMap<String, String> queryParams)
      throws Exception {
    String testBody = "Testing body";

    HttpResponse response = runCallback(builder, requestMock -> {
      HttpEntity entity = mock(HttpEntity.class);
      when(entity.getContent()).thenReturn(toInputStream(testBody, UTF_8));
      when(requestMock.getEntity()).thenReturn(entity);
      when(requestMock.getHeaders()).thenReturn(emptyMultiMap());
      when(requestMock.getQueryParams()).thenReturn(queryParams);
    }, authCallbackCaptor);

    assertThat(response.getStatusCode(), is(MOVED_TEMPORARILY.getStatusCode()));
    assertThat(IOUtils.toString(response.getEntity().getContent(), UTF_8), is(testBody));

    return response;
  }

  private HttpResponse runCallback(Consumer<HttpRequest> requestMockConfigurer,
                                   ArgumentCaptor<RequestHandler> requestHandlerCaptor)
      throws Exception {
    return runCallback(baseAuthCodeDancerbuilder(), requestMockConfigurer, requestHandlerCaptor);
  }

  private HttpResponse runCallback(OAuthAuthorizationCodeDancerBuilder builder,
                                   Consumer<HttpRequest> requestMockConfigurer,
                                   ArgumentCaptor<RequestHandler> requestHandlerCaptor)
      throws Exception {
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.externalCallbackUrl("http://host/externalCallback");
    builder.withClientCredentialsIn(BASIC_AUTH_HEADER);

    minimalDancer = startDancer(builder);

    HttpRequest request = mock(HttpRequest.class);
    requestMockConfigurer.accept(request);
    HttpRequestContext requestContext = mock(HttpRequestContext.class);
    when(requestContext.getRequest()).thenReturn(request);
    HttpResponseReadyCallback responseCallback = mock(HttpResponseReadyCallback.class);

    requestHandlerCaptor.getValue().handleRequest(requestContext, responseCallback);

    ArgumentCaptor<HttpResponse> responseCaptor = forClass(HttpResponse.class);
    ArgumentCaptor<ResponseStatusCallback> responseStatusCaptor = forClass(ResponseStatusCallback.class);
    verify(responseCallback).responseReady(responseCaptor.capture(), responseStatusCaptor.capture());
    responseStatusCaptor.getValue().responseSendFailure(new RuntimeException());
    responseStatusCaptor.getValue().responseSendSuccessfully();

    return responseCaptor.getValue();
  }

  @Test
  public void authCodeCredentialsEncodedInHeader() throws Exception {
    assertAuthCodeCredentialsEncodedInHeader(false);
  }

  @Test
  public void authCodeCredentialsEncodedInHeaderCompatibility() throws Exception {
    assertAuthCodeCredentialsEncodedInHeader(true);
  }

  @Test
  @Issue("MULE-20019")
  @Description("Tests the listeners added without an identifying resource owner are correctly called.")
  public void authCodeListenersWithNoResourceOwnerCalled() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.clientCredentials("Aladdin", "open sesame");

    minimalDancer = startDancer(builder);

    Latch onAuthorizationCompletedLatch = new Latch();
    Latch onTokenRefreshedLatch = new Latch();
    Latch onTokenInvalidatedLatch = new Latch();

    minimalDancer.addListener(new AuthorizationCodeListener() {

      @Override
      public void onAuthorizationCompleted(ResourceOwnerOAuthContext context) {
        onAuthorizationCompletedLatch.release();
      }

      @Override
      public void onTokenRefreshed(ResourceOwnerOAuthContext context) {
        onTokenRefreshedLatch.release();
      }

      @Override
      public void onTokenInvalidated() {
        onTokenInvalidatedLatch.release();
      }
    });

    localCallbackCaptor.getValue().handleRequest(buildLocalCallbackRequestContext(), mock(HttpResponseReadyCallback.class));
    assertTrue(onAuthorizationCompletedLatch.await(RECEIVE_TIMEOUT, MILLISECONDS));

    minimalDancer.refreshToken(null);
    assertTrue(onTokenRefreshedLatch.await(RECEIVE_TIMEOUT, MILLISECONDS));

    minimalDancer.invalidateContext(null);
    assertTrue(onTokenInvalidatedLatch.await(RECEIVE_TIMEOUT, MILLISECONDS));
  }

  @Test
  @Issue("MULE-20019")
  @Description("Tests the listeners added with an identifying resource owner are correctly called.")
  public void authCodeListenersWithResourceOwnerCalled() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.clientCredentials("Aladdin", "open sesame");

    minimalDancer = startDancer(builder);

    CountDownLatch onAuthorizationCompletedLatch = new CountDownLatch(2);
    CountDownLatch onTokenRefreshedLatch = new CountDownLatch(2);
    CountDownLatch onTokenInvalidatedLatch = new CountDownLatch(2);

    minimalDancer.addListener(DEFAULT_RESOURCE_OWNER_ID, new AuthorizationCodeListener() {

      @Override
      public void onAuthorizationCompleted(ResourceOwnerOAuthContext context) {
        onAuthorizationCompletedLatch.countDown();
      }

      @Override
      public void onTokenRefreshed(ResourceOwnerOAuthContext context) {
        onTokenRefreshedLatch.countDown();
      }

      @Override
      public void onTokenInvalidated() {
        onTokenInvalidatedLatch.countDown();
      }
    });

    minimalDancer.addListener(DEFAULT_RESOURCE_OWNER_ID, new AuthorizationCodeListener() {

      @Override
      public void onAuthorizationCompleted(ResourceOwnerOAuthContext context) {
        onAuthorizationCompletedLatch.countDown();
      }

      @Override
      public void onTokenRefreshed(ResourceOwnerOAuthContext context) {
        onTokenRefreshedLatch.countDown();
      }

      @Override
      public void onTokenInvalidated() {
        onTokenInvalidatedLatch.countDown();
      }
    });

    localCallbackCaptor.getValue().handleRequest(buildLocalCallbackRequestContext(), mock(HttpResponseReadyCallback.class));
    assertTrue(onAuthorizationCompletedLatch.await(RECEIVE_TIMEOUT, MILLISECONDS));

    minimalDancer.refreshToken(DEFAULT_RESOURCE_OWNER_ID);
    assertTrue(onTokenRefreshedLatch.await(RECEIVE_TIMEOUT, MILLISECONDS));

    minimalDancer.invalidateContext(DEFAULT_RESOURCE_OWNER_ID);
    assertTrue(onTokenInvalidatedLatch.await(RECEIVE_TIMEOUT, MILLISECONDS));
  }

  @Test
  public void authCodeCredentialsInBodyByDefault() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.clientCredentials("Aladdin", "open sesame");

    minimalDancer = startDancer(builder);
    localCallbackCaptor.getValue().handleRequest(buildLocalCallbackRequestContext(), mock(HttpResponseReadyCallback.class));

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getHeaderNames(), not(hasItem(equalToIgnoringCase(AUTHORIZATION))));

    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_id")));
    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_secret")));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("grant_type=authorization_code"));
    assertThat(requestBody, containsString("client_secret=open+sesame"));
    assertThat(requestBody, containsString("client_id=Aladdin"));
    assertThat(requestBody, containsString("code=authCode"));
  }

  private void assertAuthCodeCredentialsInBody(boolean useDeprecatedMethod) throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.clientCredentials("Aladdin", "open sesame");
    if (useDeprecatedMethod) {
      builder.encodeClientCredentialsInBody(true);
    } else {
      builder.withClientCredentialsIn(BODY);
    }

    minimalDancer = startDancer(builder);
    localCallbackCaptor.getValue().handleRequest(buildLocalCallbackRequestContext(), mock(HttpResponseReadyCallback.class));

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getHeaderNames(), not(hasItem(equalToIgnoringCase(AUTHORIZATION))));

    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_id")));
    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_secret")));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("grant_type=authorization_code"));
    assertThat(requestBody, containsString("client_secret=open+sesame"));
    assertThat(requestBody, containsString("client_id=Aladdin"));
    assertThat(requestBody, containsString("code=authCode"));
  }

  @Test
  public void authCodeCredentialsInBodyCompatibility() throws Exception {
    assertAuthCodeCredentialsInBody(true);
  }

  @Test
  public void authCodeCredentialsInBody() throws Exception {
    assertAuthCodeCredentialsInBody(false);
  }

  protected HttpRequestContext buildLocalCallbackRequestContext() {
    HttpRequest request = mock(HttpRequest.class);
    MultiMap<String, String> queryParams = new MultiMap<>();
    queryParams.put(CODE_PARAMETER, "authCode");
    when(request.getQueryParams()).thenReturn(queryParams);

    HttpRequestContext requestContext = mock(HttpRequestContext.class);
    when(requestContext.getRequest()).thenReturn(request);
    return requestContext;
  }

  private void assertAuthCodeCredentialsEncodedInHeaderRefresh(boolean useDeprecatedMethod) throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "open sesame");
    if (useDeprecatedMethod) {
      builder.encodeClientCredentialsInBody(false);
    } else {
      builder.withClientCredentialsIn(BASIC_AUTH_HEADER);
    }

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getHeaderValue(AUTHORIZATION), is("Basic QWxhZGRpbjpvcGVuIHNlc2FtZQ=="));

    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_id")));
    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_secret")));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("grant_type=refresh_token"));
    assertThat(requestBody, not(containsString("client_secret=open+sesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Test
  public void authCodeCredentialsEncodedInHeaderRefreshCompatibility() throws Exception {
    assertAuthCodeCredentialsEncodedInHeaderRefresh(true);
  }

  @Test
  public void authCodeCredentialsEncodedInHeaderRefresh() throws Exception {
    assertAuthCodeCredentialsEncodedInHeaderRefresh(false);
  }

  @Test
  public void authCodeCredentialsInBodyRefreshByDefault() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "open sesame");

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getHeaderNames(), not(hasItem(equalToIgnoringCase(AUTHORIZATION))));

    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_id")));
    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_secret")));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("grant_type=refresh_token"));
    assertThat(requestBody, containsString("client_secret=open+sesame"));
    assertThat(requestBody, containsString("client_id=Aladdin"));
  }

  private void assertAuthCodeCredentialsInBodyRefresh(boolean useDeprecatedMethod) throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "open sesame");

    if (useDeprecatedMethod) {
      builder.encodeClientCredentialsInBody(true);
    } else {
      builder.withClientCredentialsIn(BODY);
    }

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getHeaderNames(), not(hasItem(equalToIgnoringCase(AUTHORIZATION))));

    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_id")));
    assertThat(requestCaptor.getValue().getQueryParams(), not(hasKey("client_secret")));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("grant_type=refresh_token"));
    assertThat(requestBody, containsString("client_secret=open+sesame"));
    assertThat(requestBody, containsString("client_id=Aladdin"));
  }

  @Test
  public void authCodeCredentialsInBodyRefreshCompatibility() throws Exception {
    assertAuthCodeCredentialsInBodyRefresh(true);
  }

  @Test
  public void authCodeCredentialsInBodyRefresh() throws Exception {
    assertAuthCodeCredentialsInBodyRefresh(false);
  }

  @Test
  public void authCodeRefreshTokenWithQueryParamsIncludingRedirectUriParameter() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "openSesame");
    builder.externalCallbackUrl("http://localhost:8081/callback");

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null, true);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getQueryParams().get("client_id"), is("Aladdin"));
    assertThat(requestCaptor.getValue().getQueryParams().get("client_secret"), is("openSesame"));
    assertThat(requestCaptor.getValue().getQueryParams().get("grant_type"), is("refresh_token"));
    assertThat(requestCaptor.getValue().getQueryParams().get("refresh_token"), is("refreshToken"));
    assertThat(requestCaptor.getValue().getQueryParams().get(REDIRECT_URI_PARAMETER), is("http://localhost:8081/callback"));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, not(containsString("grant_type=refresh_token")));
    assertThat(requestBody, not(containsString("refresh_token=")));
    assertThat(requestBody, not(containsString("client_secret=openSesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Test
  public void authCodeRefreshTokenWithQueryParamsAndAdditionalParameters() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "openSesame");
    builder.externalCallbackUrl("http://localhost:8081/callback");
    MultiMap<String, String> additionalRefreshTokenParameters = new MultiMap<>();
    additionalRefreshTokenParameters.put("additional_parameter", "additionalParam");
    builder.addAdditionalRefreshTokenRequestParameters(additionalRefreshTokenParameters);

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null, true);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getQueryParams().get("client_id"), is("Aladdin"));
    assertThat(requestCaptor.getValue().getQueryParams().get("client_secret"), is("openSesame"));
    assertThat(requestCaptor.getValue().getQueryParams().get("grant_type"), is("refresh_token"));
    assertThat(requestCaptor.getValue().getQueryParams().get("refresh_token"), is("refreshToken"));
    assertThat(requestCaptor.getValue().getQueryParams().get("additional_parameter"), is("additionalParam"));
    assertThat(requestCaptor.getValue().getQueryParams().get(REDIRECT_URI_PARAMETER), is("http://localhost:8081/callback"));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, not(containsString("grant_type=refresh_token")));
    assertThat(requestBody, not(containsString("refresh_token=")));
    assertThat(requestBody, not(containsString("client_secret=openSesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Test
  public void authCodeRefreshTokenWithQueryParamsAndAdditionalHeaders() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "openSesame");
    builder.externalCallbackUrl("http://localhost:8081/callback");
    MultiMap<String, String> additionalRefreshTokenHeaders = new MultiMap<>();
    additionalRefreshTokenHeaders.put("additional_header", "header");
    builder.addAdditionalRefreshTokenRequestHeaders(additionalRefreshTokenHeaders);

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null, true);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getQueryParams().get("client_id"), is("Aladdin"));
    assertThat(requestCaptor.getValue().getQueryParams().get("client_secret"), is("openSesame"));
    assertThat(requestCaptor.getValue().getQueryParams().get("grant_type"), is("refresh_token"));
    assertThat(requestCaptor.getValue().getQueryParams().get("refresh_token"), is("refreshToken"));
    assertThat(requestCaptor.getValue().getQueryParams().get(REDIRECT_URI_PARAMETER), is("http://localhost:8081/callback"));

    assertThat(requestCaptor.getValue().getHeaderValue("additional_header"), is("header"));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, not(containsString("grant_type=refresh_token")));
    assertThat(requestBody, not(containsString("refresh_token=")));
    assertThat(requestBody, not(containsString("client_secret=openSesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Test
  @Feature(BODY_PARAMS)
  public void authCodeRefreshTokenWithCustomBodyParam() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.clientCredentials("Aladdin", "open sesame");

    Map<String, String> bodyParams = new LinkedHashMap<>();
    bodyParams.put("ice", "cream");
    bodyParams.put("pan", "cake");
    builder.customBodyParameters(bodyParams);

    minimalDancer = startDancer(builder);
    localCallbackCaptor.getValue().handleRequest(buildLocalCallbackRequestContext(), mock(HttpResponseReadyCallback.class));

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("ice=cream"));
    assertThat(requestBody, containsString("pan=cake"));
  }

  @Test
  public void authCodeRefreshTokenWithQueryParamsAndSeveralAdditionalHeadersAndParameters() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "openSesame");
    builder.externalCallbackUrl("http://localhost:8081/callback");
    MultiMap<String, String> additionalRefreshTokenHeaders = new MultiMap<>();
    additionalRefreshTokenHeaders.put("additional_header", "header");
    additionalRefreshTokenHeaders.put("additional_header", "header2");
    builder.addAdditionalRefreshTokenRequestHeaders(additionalRefreshTokenHeaders);
    MultiMap<String, String> additionalRefreshTokenParameters = new MultiMap<>();
    additionalRefreshTokenParameters.put("additional_parameter", "additionalParam");
    additionalRefreshTokenParameters.put("additional_parameter", "additionalParam2");
    builder.addAdditionalRefreshTokenRequestParameters(additionalRefreshTokenParameters);

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null, true);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getQueryParams().get("client_id"), is("Aladdin"));
    assertThat(requestCaptor.getValue().getQueryParams().get("client_secret"), is("openSesame"));
    assertThat(requestCaptor.getValue().getQueryParams().get("grant_type"), is("refresh_token"));
    assertThat(requestCaptor.getValue().getQueryParams().get("refresh_token"), is("refreshToken"));
    assertThat(requestCaptor.getValue().getQueryParams().get(REDIRECT_URI_PARAMETER), is("http://localhost:8081/callback"));
    assertThat(requestCaptor.getValue().getQueryParams().getAll("additional_parameter").size(), is(2));
    assertThat(requestCaptor.getValue().getQueryParams().getAll("additional_parameter").contains("additionalParam"), is(true));
    assertThat(requestCaptor.getValue().getQueryParams().getAll("additional_parameter").contains("additionalParam2"), is(true));

    assertThat(requestCaptor.getValue().getHeaders().getAll("additional_header").size(), is(2));
    assertThat(requestCaptor.getValue().getHeaders().getAll("additional_header").contains("header"), is(true));
    assertThat(requestCaptor.getValue().getHeaders().getAll("additional_header").contains("header2"), is(true));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, not(containsString("grant_type=refresh_token")));
    assertThat(requestBody, not(containsString("refresh_token=")));
    assertThat(requestBody, not(containsString("client_secret=openSesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Test
  public void authCodeRefreshTokenWithQueryParamsAndNoRedirectUriParameter() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.clientCredentials("Aladdin", "openSesame");
    builder.includeRedirectUriInRefreshTokenRequest(false);

    minimalDancer = startDancer(builder);
    minimalDancer.refreshToken(null, true);

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getQueryParams().get("client_id"), is("Aladdin"));
    assertThat(requestCaptor.getValue().getQueryParams().get("client_secret"), is("openSesame"));
    assertThat(requestCaptor.getValue().getQueryParams().get("grant_type"), is("refresh_token"));
    assertThat(requestCaptor.getValue().getQueryParams().get("refresh_token"), is("refreshToken"));
    assertThat(requestCaptor.getValue().getQueryParams().get(REDIRECT_URI_PARAMETER), nullValue());

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, not(containsString("grant_type=refresh_token")));
    assertThat(requestBody, not(containsString("refresh_token=")));
    assertThat(requestBody, not(containsString("client_secret=openSesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Test
  public void authCodeCredentialsAsQueryParams() throws Exception {
    final OAuthAuthorizationCodeDancerBuilder builder = baseAuthCodeDancerbuilder();
    builder.tokenUrl("http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(httpServer, "/localCallback");
    builder.localAuthorizationUrlPath("/auth");
    builder.clientCredentials("Aladdin", "openSesame");
    builder.withClientCredentialsIn(QUERY_PARAMS);

    minimalDancer = startDancer(builder);
    localCallbackCaptor.getValue().handleRequest(buildLocalCallbackRequestContext(), mock(HttpResponseReadyCallback.class));

    ArgumentCaptor<HttpRequest> requestCaptor = forClass(HttpRequest.class);
    verify(httpClient).sendAsync(requestCaptor.capture(), any(HttpRequestOptions.class));

    assertThat(requestCaptor.getValue().getQueryParams().get("client_id"), is("Aladdin"));
    assertThat(requestCaptor.getValue().getQueryParams().get("client_secret"), is("openSesame"));

    assertThat(requestCaptor.getValue().getHeaderNames(), not(hasItem(equalToIgnoringCase(AUTHORIZATION))));

    String requestBody = IOUtils.toString(requestCaptor.getValue().getEntity().getContent(), UTF_8);
    assertThat(requestBody, containsString("code=authCode"));
    assertThat(requestBody, containsString("grant_type=authorization_code"));
    assertThat(requestBody, not(containsString("client_secret=openSesame")));
    assertThat(requestBody, not(containsString("client_id=Aladdin")));
  }

  @Override
  protected OAuthAuthorizationCodeDancerBuilder baseAuthCodeDancerbuilder() {
    return baseAuthCodeDancerbuilder(mock(MuleExpressionLanguage.class));
  }

  protected OAuthAuthorizationCodeDancerBuilder baseAuthCodeDancerbuilder(MuleExpressionLanguage muleExpressionLanguage) {
    ResourceOwnerOAuthContext context = contextSupplier.get();
    Map<String, ResourceOwnerOAuthContext> tokensMap = new HashMap<>();
    tokensMap.put(DEFAULT_RESOURCE_OWNER_ID, context);

    final OAuthAuthorizationCodeDancerBuilder builder =
        new DefaultOAuthAuthorizationCodeDancerBuilder(new SimpleUnitTestSupportSchedulerService(), lockFactory, tokensMap,
                                                       httpService, oAuthHttpClientFactory, muleExpressionLanguage);

    builder.clientCredentials("clientId", "clientSecret");
    return builder;
  }
}
