/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.oauth.internal.builder;

import static java.util.Collections.emptyMap;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.oauth.api.builder.ClientCredentialsLocation.BODY;
import static org.mule.runtime.oauth.internal.util.MultiMapUtils.putAll;

import org.mule.runtime.api.el.MuleExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.oauth.api.AuthorizationCodeOAuthDancer;
import org.mule.runtime.oauth.api.AuthorizationCodeRequest;
import org.mule.runtime.oauth.api.builder.AuthorizationCodeDanceCallbackContext;
import org.mule.runtime.oauth.api.builder.OAuthAuthorizationCodeDancerBuilder;
import org.mule.runtime.oauth.api.http.HttpClientFactory;
import org.mule.runtime.oauth.api.listener.AuthorizationCodeListener;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext;
import org.mule.runtime.oauth.internal.DefaultAuthorizationCodeOAuthDancer;
import org.mule.runtime.oauth.internal.config.AuthDancerCredentialConfig;
import org.mule.runtime.oauth.internal.config.DefaultAuthorizationCodeOAuthDancerConfig;
import org.mule.runtime.oauth.internal.config.TokenParserConfig;

import java.net.URL;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Builder that allows to configure the attributes for the authorization-code grant type.
 *
 * @since 1.0
 */
public class DefaultOAuthAuthorizationCodeDancerBuilder extends AbstractOAuthDancerBuilder<AuthorizationCodeOAuthDancer>
    implements OAuthAuthorizationCodeDancerBuilder {

  private final HttpService httpService;

  private Supplier<HttpServer> localCallbackServerFactory;
  private String localCallbackUrlPath;
  private String localAuthorizationUrlPath;
  private String localAuthorizationUrlResourceOwnerId;
  private String externalCallbackUrl;

  private String state;
  private String authorizationUrl;
  private boolean includeRedirectUriInRefreshTokenRequest = true;
  private final List<AuthorizationCodeListener> listeners = new LinkedList<>();
  private MultiMap<String, String> additionalRefreshTokenRequestParameters = new MultiMap<>();
  private MultiMap<String, String> additionalRefreshTokenHeaders = new MultiMap<>();

  private Supplier<Map<String, String>> customParameters = () -> emptyMap();
  private Supplier<Map<String, String>> customHeaders = () -> emptyMap();

  private Function<AuthorizationCodeRequest, AuthorizationCodeDanceCallbackContext> beforeDanceCallback = r -> k -> empty();
  private BiConsumer<AuthorizationCodeDanceCallbackContext, ResourceOwnerOAuthContext> afterDanceCallback = (vars, ctx) -> {
  };

  public DefaultOAuthAuthorizationCodeDancerBuilder(SchedulerService schedulerService, LockFactory lockProvider,
                                                    Map<String, ResourceOwnerOAuthContext> tokensStore,
                                                    HttpService httpService,
                                                    HttpClientFactory httpClientFactory,
                                                    MuleExpressionLanguage expressionEvaluator) {
    super(schedulerService, lockProvider, tokensStore, httpClientFactory, expressionEvaluator);
    this.httpService = httpService;
    clientCredentialsLocation = BODY;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder localCallback(URL localCallbackUrl) {
    localCallbackServerFactory = () -> {
      final HttpServerConfiguration.Builder serverConfigBuilder = new HttpServerConfiguration.Builder();
      serverConfigBuilder.setHost(localCallbackUrl.getHost()).setPort(localCallbackUrl.getPort())
          .setName(localCallbackUrl.toString());
      try {
        return httpService.getServerFactory().create(serverConfigBuilder.build());
      } catch (ServerCreationException e) {
        throw new MuleRuntimeException(e);
      }
    };
    localCallbackUrlPath = localCallbackUrl.getPath();

    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder localCallback(URL localCallbackUrl, TlsContextFactory tlsContextFactory) {
    localCallbackServerFactory = () -> {
      final HttpServerConfiguration.Builder serverConfigBuilder = new HttpServerConfiguration.Builder();
      serverConfigBuilder.setHost(localCallbackUrl.getHost()).setPort(localCallbackUrl.getPort())
          .setName(localCallbackUrl.toString());
      serverConfigBuilder.setTlsContextFactory(tlsContextFactory);
      try {
        return httpService.getServerFactory().create(serverConfigBuilder.build());
      } catch (ServerCreationException e) {
        throw new MuleRuntimeException(e);
      }
    };
    localCallbackUrlPath = localCallbackUrl.getPath();

    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder localCallback(HttpServer server, String localCallbackConfigPath) {
    localCallbackServerFactory = () -> server;

    localCallbackUrlPath = localCallbackConfigPath;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder localAuthorizationUrlPath(String path) {
    this.localAuthorizationUrlPath = path;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder localAuthorizationUrlResourceOwnerId(String localAuthorizationUrlResourceOwnerIdExpr) {
    this.localAuthorizationUrlResourceOwnerId = localAuthorizationUrlResourceOwnerIdExpr;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder customParameters(Map<String, String> customParameters) {
    requireNonNull(customParameters, "customParameters cannot be null");
    return customParameters(() -> customParameters);
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder customParameters(Supplier<Map<String, String>> customParameters) {
    requireNonNull(customParameters, "customParameters cannot be null");
    this.customParameters = customParameters;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder customHeaders(Map<String, String> customHeaders) {
    requireNonNull(customHeaders, "customHeaders cannot be null");
    return customHeaders(() -> customHeaders);
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder customHeaders(Supplier<Map<String, String>> customHeaders) {
    requireNonNull(customHeaders, "customHeaders cannot be null");
    this.customHeaders = customHeaders;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder state(String stateExpr) {
    this.state = stateExpr;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder authorizationUrl(String authorizationUrl) {
    this.authorizationUrl = authorizationUrl;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder externalCallbackUrl(String externalCallbackUrl) {
    this.externalCallbackUrl = externalCallbackUrl;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder beforeDanceCallback(Function<AuthorizationCodeRequest, AuthorizationCodeDanceCallbackContext> beforeDanceCallback) {
    requireNonNull(beforeDanceCallback, "beforeDanceCallback cannot be null");
    this.beforeDanceCallback = beforeDanceCallback;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder afterDanceCallback(BiConsumer<AuthorizationCodeDanceCallbackContext, ResourceOwnerOAuthContext> afterDanceCallback) {
    requireNonNull(afterDanceCallback, "afterDanceCallback cannot be null");
    this.afterDanceCallback = afterDanceCallback;
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder encodeClientCredentialsInBody(boolean encodeClientCredentialsInBody) {
    return (OAuthAuthorizationCodeDancerBuilder) super.encodeClientCredentialsInBody(encodeClientCredentialsInBody);
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder addListener(AuthorizationCodeListener listener) {
    requireNonNull(listener, "Cannot add a null listener");
    listeners.add(listener);
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder addAdditionalRefreshTokenRequestParameters(MultiMap<String, String> additionalParameters) {
    requireNonNull(additionalParameters, "Cannot add null additional refresh token request parameters");
    putAll(additionalParameters, this.additionalRefreshTokenRequestParameters);
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder addAdditionalRefreshTokenRequestHeaders(MultiMap<String, String> additionalHeaders) {
    requireNonNull(additionalHeaders, "Cannot add null additional refresh token headers");
    putAll(additionalHeaders, this.additionalRefreshTokenHeaders);
    return this;
  }

  @Override
  public OAuthAuthorizationCodeDancerBuilder includeRedirectUriInRefreshTokenRequest(boolean includeRedirectUriInRefreshTokenRequest) {
    this.includeRedirectUriInRefreshTokenRequest = includeRedirectUriInRefreshTokenRequest;
    return this;
  }

  @Override
  public AuthorizationCodeOAuthDancer build() {
    checkArgument(isNotBlank(clientId), "clientId cannot be blank");
    checkArgument(isNotBlank(clientSecret), "clientSecret cannot be blank");
    checkArgument(isNotBlank(tokenUrl), "tokenUrl cannot be blank");
    checkArgument(isNotBlank(authorizationUrl), "authorizationUrl cannot be blank");

    Optional<HttpServer> httpServer = localCallbackServerFactory != null ? of(localCallbackServerFactory.get()) : empty();

    DefaultAuthorizationCodeOAuthDancerConfig config = new DefaultAuthorizationCodeOAuthDancerConfig();
    config.setName(name);
    config.setHttpServer(httpServer);
    config.setLocalCallbackUrlPath(localCallbackUrlPath);
    config.setLocalAuthorizationUrlPath(localAuthorizationUrlPath);
    config.setCredentialConfig(AuthDancerCredentialConfig.builder()
        .withClientId(clientId)
        .withClientSecret(clientSecret)
        .withClientCredentialsLocation(clientCredentialsLocation)
        .build());
    config.setTokenUrl(tokenUrl);
    config.setScopes(scopes);
    config.setExternalCallbackUrl(externalCallbackUrl);
    config.setEncoding(encoding);
    config.setLocalAuthorizationUrlResourceOwnerId(localAuthorizationUrlResourceOwnerId);
    config.setState(state);
    config.setAuthorizationUrl(authorizationUrl);
    config.setTokenParserConfig(TokenParserConfig.builder()
        .withResponseAccessTokenExpr(responseAccessTokenExpr)
        .withResponseRefreshTokenExpr(responseRefreshTokenExpr)
        .withResponseExpiresInExpr(responseExpiresInExpr)
        .build());
    config.setCustomParameters(customParameters);
    config.setCustomHeaders(customHeaders);
    config.setCustomParametersExtractorsExprs(customParametersExtractorsExprs);
    config.setResourceOwnerIdTransformer(resourceOwnerIdTransformer);
    config.setSchedulerService(schedulerService);
    config.setLockProvider(lockProvider);
    config.setTokensStore(tokensStore);
    config.setHttpClient(httpClientFactory.get());
    config.setExpressionEvaluator(expressionEvaluator);
    config.setBeforeDanceCallback(beforeDanceCallback);
    config.setAfterDanceCallback(afterDanceCallback);
    config.setListeners(listeners);
    config.setAdditionalRefreshTokenRequestParameters(additionalRefreshTokenRequestParameters);
    config.setAdditionalRefreshTokenHeaders(additionalRefreshTokenHeaders);
    config.setIncludeRedirectUriInRefreshTokenRequest(includeRedirectUriInRefreshTokenRequest);

    return new DefaultAuthorizationCodeOAuthDancer(config);
  }

}
