/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.oauth.internal;

import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.sameInstance;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsMapContaining.hasEntry;
import static org.hamcrest.collection.IsMapContaining.hasKey;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext.DEFAULT_RESOURCE_OWNER_ID;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAuthClientStory.OAUTH_DANCER;

import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.el.MuleExpressionLanguage;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.oauth.api.AuthorizationCodeOAuthDancer;
import org.mule.runtime.oauth.api.ClientCredentialsOAuthDancer;
import org.mule.runtime.oauth.api.builder.OAuthAuthorizationCodeDancerBuilder;
import org.mule.runtime.oauth.api.builder.OAuthClientCredentialsDancerBuilder;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext;
import org.mule.runtime.oauth.internal.builder.DefaultOAuthAuthorizationCodeDancerBuilder;
import org.mule.runtime.oauth.internal.builder.DefaultOAuthClientCredentialsDancerBuilder;
import org.mule.runtime.test.oauth.AbstractOAuthTestCase;
import org.mule.tck.SimpleUnitTestSupportSchedulerService;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;

@Story(OAUTH_DANCER)
public class OAuthContextTestCase extends AbstractOAuthTestCase {

  private Map<String, ResourceOwnerOAuthContext> tokensStore;

  @Before
  public void before() throws ConnectionException, IOException, TimeoutException {
    tokensStore = new HashMap<>();
  }

  @Test
  public void clientCredentialsFirstGetContext() throws Exception {
    ClientCredentialsOAuthDancer clientCredentialsDancer =
        baseClientCredentialsDancerBuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    assertThat(tokensStore, not(hasKey(DEFAULT_RESOURCE_OWNER_ID)));
    ResourceOwnerOAuthContext contextFromDancer = clientCredentialsDancer.getContext();
    assertThat(contextFromDancer, not(nullValue()));
    assertThat(tokensStore, hasEntry(DEFAULT_RESOURCE_OWNER_ID, contextFromDancer));
  }

  @Test
  public void clientCredentialsFollowingGetContext() throws Exception {
    ClientCredentialsOAuthDancer clientCredentialsDancer =
        baseClientCredentialsDancerBuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    ResourceOwnerOAuthContext contextFromDancer = clientCredentialsDancer.getContext();

    assertThat(clientCredentialsDancer.getContext(), sameInstance(contextFromDancer));
    assertThat(tokensStore.entrySet(), hasSize(1));
  }

  @Test
  public void clientCredentialsInvalidateContext() throws Exception {
    ClientCredentialsOAuthDancer clientCredentialsDancer =
        baseClientCredentialsDancerBuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    assertThat(tokensStore, not(hasKey(DEFAULT_RESOURCE_OWNER_ID)));
    clientCredentialsDancer.getContext();
    assertThat(tokensStore, hasKey(DEFAULT_RESOURCE_OWNER_ID));

    clientCredentialsDancer.invalidateContext();
    assertThat(tokensStore, not(hasKey(DEFAULT_RESOURCE_OWNER_ID)));
  }

  @Test
  public void authCodeFirstGetContext() throws Exception {
    AuthorizationCodeOAuthDancer authCodeDancer =
        baseAuthCodeDancerbuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    assertThat(tokensStore, not(hasKey("user1")));
    ResourceOwnerOAuthContext contextFromDancer = authCodeDancer.getContextForResourceOwner("user1");
    assertThat(contextFromDancer, not(nullValue()));
    assertThat(tokensStore, hasEntry("user1", contextFromDancer));
  }

  @Test
  public void authCodeFollowingGetContext() throws Exception {
    AuthorizationCodeOAuthDancer authCodeDancer =
        baseAuthCodeDancerbuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    ResourceOwnerOAuthContext contextFromDancer = authCodeDancer.getContextForResourceOwner("user1");

    assertThat(authCodeDancer.getContextForResourceOwner("user1"), sameInstance(contextFromDancer));
    assertThat(tokensStore.entrySet(), hasSize(1));
  }

  @Test
  public void authCodeFollowingGetContextDiffernetUser() throws Exception {
    AuthorizationCodeOAuthDancer authCodeDancer =
        baseAuthCodeDancerbuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    ResourceOwnerOAuthContext contextFromDancer1 = authCodeDancer.getContextForResourceOwner("user1");
    ResourceOwnerOAuthContext contextFromDancer2 = authCodeDancer.getContextForResourceOwner("user2");

    assertThat(authCodeDancer.getContextForResourceOwner("user1"), sameInstance(contextFromDancer1));
    assertThat(authCodeDancer.getContextForResourceOwner("user2"), sameInstance(contextFromDancer2));
    assertThat(contextFromDancer1, not(sameInstance(contextFromDancer2)));
    assertThat(tokensStore.entrySet(), hasSize(2));
  }

  @Test
  public void authCodeInvalidateContext() throws Exception {
    AuthorizationCodeOAuthDancer authCodeDancer =
        baseAuthCodeDancerbuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    assertThat(tokensStore, not(hasKey("user1")));
    authCodeDancer.getContextForResourceOwner("user1");
    assertThat(tokensStore, hasKey("user1"));

    authCodeDancer.invalidateContext("user1");
    assertThat(tokensStore, not(hasKey("user1")));
  }

  @Test
  public void authCodeInvalidateContextDoesNotAffectOtherUsers() throws Exception {
    AuthorizationCodeOAuthDancer authCodeDancer =
        baseAuthCodeDancerbuilder().tokenUrl(mock(HttpClient.class), "http://host/token").build();

    authCodeDancer.getContextForResourceOwner("user1");
    authCodeDancer.getContextForResourceOwner("user2");

    assertThat(tokensStore.entrySet(), hasSize(2));
    assertThat(tokensStore, hasKey("user1"));
    assertThat(tokensStore, hasKey("user2"));

    authCodeDancer.invalidateContext("user1");
    assertThat(tokensStore, not(hasKey("user1")));
    assertThat(tokensStore, hasKey("user2"));
  }

  @Override
  protected OAuthClientCredentialsDancerBuilder baseClientCredentialsDancerBuilder() {
    final DefaultOAuthClientCredentialsDancerBuilder builder =
        new DefaultOAuthClientCredentialsDancerBuilder(new SimpleUnitTestSupportSchedulerService(), lockFactory, tokensStore,
                                                       oAuthHttpClientFactory, mock(MuleExpressionLanguage.class));

    builder.clientCredentials("clientId", "clientSecret");
    builder.tokenUrl(mock(HttpClient.class), "http://host/token");
    return builder;
  }

  @Override
  protected OAuthAuthorizationCodeDancerBuilder baseAuthCodeDancerbuilder() {
    DefaultOAuthAuthorizationCodeDancerBuilder builder =
        new DefaultOAuthAuthorizationCodeDancerBuilder(new SimpleUnitTestSupportSchedulerService(), lockFactory, tokensStore,
                                                       httpService, oAuthHttpClientFactory, mock(MuleExpressionLanguage.class));

    builder.clientCredentials("clientId", "clientSecret");
    builder.tokenUrl(mock(HttpClient.class), "http://host/token");
    builder.authorizationUrl("http://host/auth");
    builder.localCallback(mock(HttpServer.class), "localCallback");
    return builder;
  }

}
