/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.oauth.internal;

import static java.lang.Thread.currentThread;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext.DEFAULT_RESOURCE_OWNER_ID;
import static org.mule.runtime.oauth.internal.OAuthConstants.GRANT_TYPE_CLIENT_CREDENTIALS;
import static org.mule.runtime.oauth.internal.OAuthConstants.GRANT_TYPE_PARAMETER;
import static org.mule.runtime.oauth.internal.OAuthConstants.SCOPE_PARAMETER;
import static org.mule.runtime.oauth.internal.util.ClassLoaderUtils.setContextClassLoader;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.LifecycleException;
import org.mule.runtime.oauth.api.ClientCredentialsOAuthDancer;
import org.mule.runtime.oauth.api.exception.RequestAuthenticationException;
import org.mule.runtime.oauth.api.exception.TokenNotFoundException;
import org.mule.runtime.oauth.api.exception.TokenUrlResponseException;
import org.mule.runtime.oauth.api.listener.ClientCredentialsListener;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContextWithRefreshState;
import org.mule.runtime.oauth.internal.config.DefaultClientCredentialsOAuthDancerConfig;
import org.slf4j.Logger;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;

/**
 * Provides OAuth dance support for client-credentials grant-type.
 *
 * @since 1.0
 */
public class DefaultClientCredentialsOAuthDancer extends AbstractOAuthDancer<DefaultClientCredentialsOAuthDancerConfig>
    implements ClientCredentialsOAuthDancer {

  private static final Logger LOGGER = getLogger(DefaultClientCredentialsOAuthDancer.class);

  private boolean accessTokenRefreshedOnStart = false;

  public DefaultClientCredentialsOAuthDancer(DefaultClientCredentialsOAuthDancerConfig config) {
    super(config);
  }

  @Override
  public void start() throws MuleException {
    super.start();
    try {
      refreshToken().get();
      accessTokenRefreshedOnStart = true;
    } catch (ExecutionException e) {
      if (!(e.getCause() instanceof TokenUrlResponseException) && !(e.getCause() instanceof TokenNotFoundException)) {
        super.stop();
        throw new LifecycleException(e.getCause(), this);
      }
      // else nothing to do, accessTokenRefreshedOnStart remains false and this is called later
    } catch (InterruptedException e) {
      super.stop();
      currentThread().interrupt();
      throw new LifecycleException(e, this);
    }
  }

  @Override
  public CompletableFuture<String> accessToken() throws RequestAuthenticationException {
    if (!accessTokenRefreshedOnStart) {
      accessTokenRefreshedOnStart = true;
      return refreshToken().thenApply(v -> getContext().getAccessToken());
    }

    final String accessToken = getContext().getAccessToken();
    if (accessToken == null) {
      LOGGER.info("Previously stored token has been invalidated. Refreshing...");
      return doRefreshTokenRequest(false).thenApply(v -> getContext().getAccessToken());
    }

    // TODO MULE-11858 proactively refresh if the token has already expired based on its 'expiresIn' parameter
    return completedFuture(accessToken);
  }

  @Override
  public CompletableFuture<Void> refreshToken() {
    return doRefreshTokenRequest(true);
  }

  private CompletableFuture<Void> doRefreshTokenRequest(boolean notifyListeners) {
    return doRefreshToken(() -> getContext(),
                          ctx -> doRefreshTokenRequest(notifyListeners, (ResourceOwnerOAuthContextWithRefreshState) ctx));
  }

  private CompletableFuture<Void> doRefreshTokenRequest(boolean notifyListeners,
                                                        ResourceOwnerOAuthContextWithRefreshState defaultUserState) {
    final Map<String, String> formData = new HashMap<>();

    formData.put(GRANT_TYPE_PARAMETER, GRANT_TYPE_CLIENT_CREDENTIALS);
    if (config.getScopes() != null) {
      formData.put(SCOPE_PARAMETER, config.getScopes());
    }
    String authorization = handleClientCredentials(formData);

    return invokeTokenUrl(config.getTokenUrl(), formData, config.getCustomParameters(), config.getCustomHeaders(), authorization,
                          false, config.getEncoding())
                              .thenAccept(tokenResponse -> {
                                final Thread thread = currentThread();
                                final ClassLoader currentClassLoader = thread.getContextClassLoader();
                                final ClassLoader contextClassLoader = DefaultClientCredentialsOAuthDancer.class.getClassLoader();
                                setContextClassLoader(thread, currentClassLoader, contextClassLoader);

                                try {
                                  if (LOGGER.isDebugEnabled()) {
                                    LOGGER
                                        .debug("Retrieved access token, refresh token and expires from token url are: %s, %s, %s",
                                               tokenResponse.getAccessToken(), tokenResponse.getRefreshToken(),
                                               tokenResponse.getExpiresIn());
                                  }

                                  defaultUserState.setAccessToken(tokenResponse.getAccessToken());
                                  defaultUserState.setExpiresIn(tokenResponse.getExpiresIn());
                                  for (Entry<String, Object> customResponseParameterEntry : tokenResponse
                                      .getCustomResponseParameters().entrySet()) {
                                    defaultUserState.getTokenResponseParameters().put(customResponseParameterEntry.getKey(),
                                                                                      customResponseParameterEntry.getValue());
                                  }

                                  updateOAuthContextAfterTokenResponse(defaultUserState);
                                  if (notifyListeners) {
                                    forEachListener(l -> l.onTokenRefreshed(defaultUserState));
                                  }
                                } finally {
                                  setContextClassLoader(thread, contextClassLoader, currentClassLoader);
                                }
                              })
                              .exceptionally(tokenUrlExceptionHandler(defaultUserState));
  }

  @Override
  public void addListener(ClientCredentialsListener listener) {
    doAddListener(listener);
  }

  @Override
  public void removeListener(ClientCredentialsListener listener) {
    doRemoveListener(listener);
  }

  @Override
  public void invalidateContext() {
    invalidateContext(DEFAULT_RESOURCE_OWNER_ID);
  }

  @Override
  public ResourceOwnerOAuthContext getContext() {
    return getContextForResourceOwner(DEFAULT_RESOURCE_OWNER_ID);
  }

  private void forEachListener(Consumer<ClientCredentialsListener> action) {
    onEachListener(listener -> action.accept((ClientCredentialsListener) listener));
  }
}
