/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.oauth.http;

import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.mock;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAUTH_CLIENT;
import static org.mule.runtime.test.AllureConstants.OAuthClientFeature.OAuthClientStory.OAUTH_DANCER;

import io.qameta.allure.Story;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.oauth.api.http.HttpClientFactory;
import org.mule.runtime.oauth.internal.http.DefaultHttpClientFactory;

import io.qameta.allure.Feature;
import org.junit.Before;
import org.junit.Test;

@Feature(OAUTH_CLIENT)
@Story(OAUTH_DANCER)
public class DefaultHttpClientFactoryTestCase {

  private TlsContextFactory tlsContextFactory;
  private ProxyConfig proxyConfig;
  private HttpClientFactory oAuthHttpClientFactory;

  @Before
  public void setup() {
    tlsContextFactory = mock(TlsContextFactory.class);
    proxyConfig = mock(ProxyConfig.class);

    org.mule.runtime.http.api.client.HttpClientFactory httpClientFactory =
        mock(org.mule.runtime.http.api.client.HttpClientFactory.class);
    HttpClient httpClient = mock(HttpClient.class);
    when(httpClientFactory.create(any())).thenReturn(httpClient);
    HttpService httpService = mock(HttpService.class);
    when(httpService.getClientFactory()).thenReturn(httpClientFactory);

    oAuthHttpClientFactory = new DefaultHttpClientFactory(httpService);
  }

  @Test
  public void createHttpClientWithSameParameterReturnCachedHttpClient() {
    HttpClient client = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(client, is(notNullValue()));

    HttpClient cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(cachedHttpClient, is(client));
  }

  @Test
  public void createHttpClientWithDifferentParametersReturnCachedHttpClient() {
    HttpClient client = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(client, is(notNullValue()));

    HttpClient cachedHttpClient = oAuthHttpClientFactory.create(null, null);
    assertThat(cachedHttpClient, is(not(client)));

    cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, null);
    assertThat(cachedHttpClient, is(not(client)));

    cachedHttpClient = oAuthHttpClientFactory.create(null, proxyConfig);
    assertThat(cachedHttpClient, is(not(client)));

    cachedHttpClient = oAuthHttpClientFactory.create(mock(TlsContextFactory.class), proxyConfig);
    assertThat(cachedHttpClient, is(not(client)));
  }

  @Test
  public void stoppingCachedHttpClientShouldReturnANewHttpClient() {
    HttpClient client = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(client, is(notNullValue()));

    HttpClient cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(cachedHttpClient, is(client));
    client.start();
    client.stop();

    cachedHttpClient = oAuthHttpClientFactory.create(tlsContextFactory, proxyConfig);
    assertThat(cachedHttpClient, is(not(client)));
  }

}
