/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.apikit.model;

import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.Namespace;
import org.jdom2.filter.Filters;
import org.jdom2.xpath.XPathExpression;

import java.util.ArrayList;
import java.util.List;

import static org.jdom2.xpath.XPathFactory.instance;

public class ErrorHandler extends Configuration {

  private static final Namespace muleCoreNamespace = Namespace
      .getNamespace("http://www.mulesoft.org/schema/mule/core");

  private static final XPathExpression<Element> configExpression =
      instance().compile("//*/*[local-name()='error-handler']", Filters.element(muleCoreNamespace));

  /**
   * OnError flows.
   */
  protected final List<OnErrorFlow> onErrorFlows = new ArrayList<>();

  /**
   * Global Error Handler Configuration.
   */
  protected final Configuration globalErrorHandler;

  public ErrorHandler(String name) {
    super(name);
    this.globalErrorHandler = null;
  }

  public ErrorHandler(String name, Configuration globalErrorHandler) {
    super(name);
    this.globalErrorHandler = globalErrorHandler;
  }

  public void addOnErrorFlow(OnErrorFlow flow) {
    this.onErrorFlows.add(flow);
  }

  @Override
  public void transformToXml(Element element) {
    Element errorHandler = new Element("error-handler", muleCoreNamespace);
    errorHandler.setAttribute("name", name);
    if (this.globalErrorHandler != null) {
      errorHandler.setAttribute("ref", globalErrorHandler.getName());
    }
    onErrorFlows.forEach(muleElement -> muleElement.transformToXml(errorHandler));
    element.addContent(errorHandler);
  }

  @Override
  public boolean exists(Document document) {
    List<Element> elementList = configExpression.evaluate(document);
    for (Element element : elementList) {
      Attribute name = element.getAttribute("name");
      if (name != null && getName().equals(name.getValue())) {
        return true;
      }
    }
    return false;
  }
}
