%dw 2.0
import filterHasType, getOperationsBindings, getOperationBindingType, getOperationBindingsType, getObjectById from scripts::modules::ApiGraphModule
import fail from dw::Runtime

/**
*  In case the API contains Solace bindings, checks if these bindings are supported
*/
fun validateAPIDoesNotContainSolaceUnsupportedScenarios(api, publishOps, subscribeOps): Boolean = do {
    var solaceOperationsQueues = getSolaceOperationsQueues(api)
    ---
    !hasSolaceBindings(api) or
    (
        validateAPIWithSolaceBindingsIncludesExactlyOneDestinationInDestinationsObject(api) and
        validateAPIWithSolaceBindingsIncludesChannelNameInQueue(solaceOperationsQueues) and
        validateAPIWithSolaceBindingsDoesNotContainMultipleOperationsWithDestinationUsingTheSameQueueName(api, publishOps) and
        validateAPIWithSolaceBindingsDoesNotContainMultipleOperationsWithDestinationUsingTheSameQueueName(api, subscribeOps)
    )
}
/**
* Finds if the destination queues used by all @code{ publish | subscribe} use unique destination queues.
* Graph nodes have only child references; no parent references. So we have to trace top to bottom.
* The list of destination queues are traced as follows
* - get operationBindings objects belonging to operations
* - get operationBinding objects belonging to operationBindings objects
* - get destinations of type queues belonging to operation binding objects
* - get queues belonging to to destinations
*/
fun validateAPIWithSolaceBindingsDoesNotContainMultipleOperationsWithDestinationUsingTheSameQueueName(api, operations): Boolean = do {
    var operationBindingsIds = getBindingsIdsFromOperations(operations)
    var operationBindingIds = getOperationBindingIdsFromOperationBindingsIds(api, operationBindingsIds)
    var operationDestinationIds = getOperationDestinationIdsFromOperationBindingIds(api, operationBindingIds)
    var operationQueueIds = getOperationQueueIdsFromOperationDestinationIds(api, operationDestinationIds)
    var operationQueues = getOperationQueuesFromOperationQueueIds(api, operationQueueIds)
    var operationsDestinationsQueueNames = operationQueues filter $."core:name"? map ((item, index) -> item."core:name")
    var noDuplicateQueueNames = sizeOf(operationsDestinationsQueueNames) == sizeOf(operationsDestinationsQueueNames distinctBy $)
    ---
    if (noDuplicateQueueNames)
        true
    else
        fail("The AsyncAPI specification contains Solace bindings with multiple operations of same type using the same Queue name in their destination")
}

/**
* Get operationBindings ids referred in the operations
*/
fun getBindingsIdsFromOperations(operations) = operations map ((item, index) -> item."apiBinding:binding"."@id")

/**
* Get operationBinding ids referred in the given list of operation bindings ids
*/
fun getOperationBindingIdsFromOperationBindingsIds(api, operationBindingsIds) = do {
    var apiOperationBindingsTypes = getOperationBindingsType(api)
    ---
    flatten(flatten(operationBindingsIds map ((item, index) -> getObjectById(apiOperationBindingsTypes, item))) map ((item, index) -> item."apiBinding:bindings"."@id"))
}

/**
* Get the list of destination ids referred in the given list of operation binding ids
*/
fun getOperationDestinationIdsFromOperationBindingIds(api, operationBindingIds) = do {
    var apiOperationBindingTypes = getOperationBindingType(api)
    ---
    flatten(flatten(operationBindingIds map ((item, index) -> getObjectById(apiOperationBindingTypes, item))) map ((item, index) -> item."apiBinding:destinations")) map ((item, index) -> item."@id")
}

/**
* Get the operation queue ids referred in the list of destination ids
*/
fun getOperationQueueIdsFromOperationDestinationIds(api, operationDestinationIds) = do {
    var apiSolaceOperationDestinationTypes = getSolaceOperationDestinationQueueTypes(api)
    ---
    flatten(operationDestinationIds map((item, index) -> getObjectById(apiSolaceOperationDestinationTypes, item)) map ((item, index) -> item."apiBinding:queue"."@id"))
}

/**
* Get the list of queues from the list of operation queue ids
*/
fun getOperationQueuesFromOperationQueueIds(api, operationQueueIds) = do {
    var apiOperationsQueues = getSolaceOperationsQueues(api)
    ---
    flatten(operationQueueIds map ((item, index) -> getObjectById(apiOperationsQueues, item)))
}

/**
* Validates that Solace destination bindings are not including multiple destinations
**/
fun validateAPIWithSolaceBindingsIncludesExactlyOneDestinationInDestinationsObject(api): Boolean = do {
    var solaceOperationsDestinations = getSolaceOperationsDestinations(api)
    var solaceOperationsDestinationsWithoutOnlyOneDestination = solaceOperationsDestinations filter(!(sizeOf($."apiBinding:destinations") == 1))
    ---
    if (!isEmpty(solaceOperationsDestinationsWithoutOnlyOneDestination))
        fail("The AsyncAPI specification contains Solace bindings with multiple destinations in the channels. Only one destination per channel is supported for Solace")
    else
        true
}


/**
* Validates that Solace queue bindings are not missing the queue name
**/
fun validateAPIWithSolaceBindingsIncludesChannelNameInQueue(solaceOperationsQueues): Boolean =
    if (!isEmpty(solaceOperationsQueues filter $."core:name"? and !isEmpty($."core:name")))
        true
    else
        fail("The AsyncAPI specification contains Solace bindings with at least one destination missing the queue name")

/**
* Validates that Solace queue bindings are not repeating the queue name
**/
fun validateAPIWithSolaceBindingsDoesNotContainMultipleChannelsWithDestinationUsingTheSameQueueName(solaceOperationDestinations): Boolean = do {
    var solaceOperationsDestinationsQueueNames = solaceOperationDestinations filter $."core:name"? map ((item, index) -> item."core:name")
    var distinctVal = solaceOperationsDestinationsQueueNames distinctBy $
    var duplicates = (distinctVal map(key,value) -> {
        count : if(sizeOf((solaceOperationsDestinationsQueueNames map $ == key) filter $ ) > 1) key else null
    }) filter $.count !=null
    ---
    if (isEmpty(duplicates.count))
        true
    else
        fail("The AsyncAPI specification contains Solace bindings with multiple channels using the same Queue name in their destination")
}

fun hasSolaceBindings(api) =
    !isEmpty(getSolaceBindings(getOperationsBindings(api)))

fun getSolaceDestinations(api) =
    api."@graph" filterHasType ["apiBinding:SolaceOperationQueue", "apiBinding:SolaceOperationQueue010", "apiBinding:SolaceOperationQueue030"]

fun getSolaceBindings(operationBindings) =
    operationBindings filter $."apiBinding:type"? and $."apiBinding:type" == "solace"

fun getSolaceOperationsQueues(api) =
    api."@graph" filterHasType ["apiBinding:SolaceOperationQueue", "apiBinding:SolaceOperationQueue010", "apiBinding:SolaceOperationQueue030"]

fun getSolaceOperationBindings(api) =
    api."@graph" filterHasType ["apiBinding:SolaceOperationBinding", "apiBinding:SolaceOperationBinding010", "apiBinding:SolaceOperationBinding030"]

fun getSolaceOperationsDestinations(api) = do {
    var solaceOperationBindings = getSolaceOperationBindings(api)
    ---
    solaceOperationBindings filter $."apiBinding:destinations"?
}

fun getSolaceOperationDestinationQueueTypes(api) =
api."@graph" filterHasType ["apiBinding:SolaceOperationDestination", "apiBinding:SolaceOperationDestination010"] filter $."apiBinding:destinationType" == "queue"