/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xcap.diff.dom;

import org.mobicents.protocols.xcap.diff.BuildPatchException;
import org.mobicents.protocols.xcap.diff.XcapDiffFactory;
import org.mobicents.protocols.xcap.diff.XcapDiffPatchBuilder;
import org.mobicents.protocols.xcap.diff.dom.utils.DOMXmlUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * DOM impl for {@link XcapDiffPatchBuilder}.
 * 
 * @author baranowb
 * @author martins
 * 
 */
public class DOMXcapDiffPatchBuilder implements
		XcapDiffPatchBuilder<Document, Element, Document, Node, Node> {

	private final DOMAttributePatchComponentBuilder attributePatchComponentBuilder = new DOMAttributePatchComponentBuilder(
			this);
	private final DOMElementPatchComponentBuilder elementPatchComponentBuilder = new DOMElementPatchComponentBuilder(
			this);
	private final DOMDocumentPatchComponentBuilder documentPatchComponentBuilder = new DOMDocumentPatchComponentBuilder(
			this);

	private final DOMXcapDiffFactory xcapDiffFactory;

	private static final String XCAP_DIFF_ROOT_ELEMENT_NAME = "xcap-diff";
	private static final String XCAP_ROOT_ATTR_NAME = "xcap-root";
	private static final String XMLNS_ATTR_NAME = "xmlns";

	public DOMXcapDiffPatchBuilder(DOMXcapDiffFactory xcapDiffFactory) {
		this.xcapDiffFactory = xcapDiffFactory;
	}

	/**
	 * 
	 * @return
	 */
	public DOMXcapDiffFactory getXcapDiffFactory() {
		return xcapDiffFactory;
	}

	@Override
	public Document buildPatch(String xcapRoot, Element[] components)
			throws BuildPatchException {

		if (xcapRoot == null) {
			throw new NullPointerException("null xcap root");
		}

		if (components == null) {
			throw new NullPointerException("null components");
		}

		// create doc
		Document document = null;
		try {
			document = DOMXmlUtils.createWellFormedDocumentFragment(
					XCAP_DIFF_ROOT_ELEMENT_NAME,
					XcapDiffFactory.XCAP_DIFF_NAMESPACE_URI);
		} catch (Throwable e) {
			throw new BuildPatchException(
					"failed to create xcap diff DOM document", e);
		}
		final Element rootElement = document.getDocumentElement();
		// set namespace attr
		rootElement.setAttribute(XMLNS_ATTR_NAME,
				XcapDiffFactory.XCAP_DIFF_NAMESPACE_URI);
		// set xcap root attr
		rootElement.setAttribute(XCAP_ROOT_ATTR_NAME, xcapRoot);
		// add patch components
		Node importedNode = null;
		for (Element component : components) {
			importedNode = document.importNode(component, true);
			rootElement.appendChild(importedNode);
		}
		return document;
	}

	@Override
	public DOMAttributePatchComponentBuilder getAttributePatchComponentBuilder() {
		return attributePatchComponentBuilder;
	}

	@Override
	public DOMElementPatchComponentBuilder getElementPatchComponentBuilder() {
		return elementPatchComponentBuilder;
	}

	@Override
	public DOMDocumentPatchComponentBuilder getDocumentPatchComponentBuilder() {
		return documentPatchComponentBuilder;
	}

}
