/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xcap.diff.dom;

import java.util.Map;

import org.mobicents.protocols.xcap.diff.BuildPatchException;
import org.mobicents.protocols.xcap.diff.XcapDiffFactory;
import org.mobicents.protocols.xcap.diff.component.ElementPatchComponentBuilder;
import org.mobicents.protocols.xcap.diff.dom.utils.DOMXmlUtils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * DOM impl for {@link ElementPatchComponentBuilder}.
 * 
 * @author baranowb
 * @author martins
 * 
 */
public class DOMElementPatchComponentBuilder implements
		ElementPatchComponentBuilder<Element, Node> {

	private static final String ELEMENT_NAME = "element";
	private static final String SEL_ATTR_NAME = "sel";
	private static final String EXISTS_ATTR_NAME = "exists";
	private static final String NOT_EXISTS_ATTR_VALUE = "0";

	private static final String DEFAULT_ELEMENT_PREFIX = "xd";

	private final DOMXcapDiffPatchBuilder xcapDiffPatchBuilder;

	public DOMElementPatchComponentBuilder(
			DOMXcapDiffPatchBuilder xcapDiffPatchBuilder) {
		this.xcapDiffPatchBuilder = xcapDiffPatchBuilder;
	}

	/**
	 * 
	 * @return
	 */
	public DOMXcapDiffPatchBuilder getXcapDiffPatchBuilder() {
		return xcapDiffPatchBuilder;
	}

	@Override
	public Element buildPatchComponent(String sel, boolean exists,
			Map<String, String> namespaceBindings) throws BuildPatchException {

		Element patchComponent = DOMXmlUtils.createElement(ELEMENT_NAME,
				DEFAULT_ELEMENT_PREFIX,
				XcapDiffFactory.XCAP_DIFF_NAMESPACE_URI, namespaceBindings);
		patchComponent.setAttribute(SEL_ATTR_NAME, sel);
		if (!exists) {
			patchComponent
					.setAttribute(EXISTS_ATTR_NAME, NOT_EXISTS_ATTR_VALUE);
		}
		return patchComponent;
	}

	@Override
	public Element buildPatchComponent(String sel, Node element,
			Map<String, String> namespaceBindings) throws BuildPatchException {

		// build patch component
		final Element patchComponent = buildPatchComponent(sel, true,
				namespaceBindings);
		// import element
		Node importedElement = patchComponent.getOwnerDocument().importNode(
				element, true);
		patchComponent.appendChild(importedElement);
		return patchComponent;
	}

}
