/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xml.patch;

import java.util.Map;

import org.mobicents.protocols.xcap.diff.BuildPatchException;

/**
 * 
 * Builds XML Patch Ops (RFC 5261) patching instructions.
 * 
 * @author baranowb
 * @author martins
 * @param <D>
 *            the document type, defines what is the concrete type of XML
 *            documents used
 * @param <P>
 *            the patching instruction type
 * @param <E>
 *            the element type
 * @param <N>
 *            the node type
 */
public interface XmlPatchOperationsBuilder<D,P, E, N> {

	public static final String XML_PATCH_OPS_NAMESPACE = "urn:ietf:params:xml:schema:patch-ops";

	/**
	 * possible <add/> "pos" attribute values
	 */
	public enum Pos {
		prepend, before, after
	}

	/**
	 * possible <remove/> "ws" attribute values
	 */
	public enum Ws {
		before, after, both
	}

	/**
	 * Builds the patching instruction for the creation of an attribute.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the added attribute.
	 * @param attrName
	 *            the attribute name
	 * @param attrValue
	 *            the attribute value
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in type and sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P addAttribute(String sel, String attrName, String attrValue,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds the patching instruction for the creation of an element.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the added element.
	 * @param element
	 *            the element
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in the element
	 *            and sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P addElement(String sel, E element,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds the patching instruction for the creation of a node.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the added node.
	 * @param pos
	 *            the pos attribute
	 * @param node
	 *            the node
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in the node and
	 *            sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P addNode(String sel, Pos pos, N node,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds the patching instruction for the creation of a prefix namespace
	 * declaration.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the added prefix namespace
	 *            declaration.
	 * @param namespacePrefix
	 *            the namespace prefix
	 * @param namespaceValue
	 *            the namespace value
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P addPrefixNamespaceDeclaration(String sel, String namespacePrefix,
			String namespaceValue, Map<String, String> namespaceBindings)
			throws BuildPatchException;

	/**
	 * Builds the patching instruction for the replacement of an attribute.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the replaced attribute.
	 * @param attrValue
	 *            the attribute value
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P replaceAttribute(String sel, String attributeValue,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds the patching instruction for the replacement of an element.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the replaced element.
	 * @param element
	 *            the element
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in the element
	 *            and sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P replaceElement(String sel, E element,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds the patching instruction for the replacement of a node.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the replaced node.
	 * @param node
	 *            the node
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in the node and
	 *            sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P replaceNode(String sel, N node,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds the patching instruction for the replacement of a prefix namespace
	 * declaration.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the replaced prefix
	 *            namespace declaration.
	 * @param namespaceValue
	 *            the namespace value
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P replacePrefixNamespaceDeclaration(String sel,
			String namespaceValue, Map<String, String> namespaceBindings)
			throws BuildPatchException;

	/**
	 * Builds the patching instruction for the removal of an attribute.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the removed attribute.
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P removeAttribute(String sel, Map<String, String> namespaceBindings)
			throws BuildPatchException;

	/**
	 * Builds the patching instruction for the removal of an element.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the removed element.
	 * @param ws
	 *            the ws.
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P removeElement(String sel, Ws ws,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds the patching instruction for the removal of node.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the removed node.
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P removeNode(String sel, Map<String, String> namespaceBindings)
			throws BuildPatchException;

	/**
	 * Builds the patching instruction for the removal of a prefix namespace
	 * declaration.
	 * 
	 * @param sel
	 *            the xpath selector, which points to the removed prefix
	 *            namespace declaration.
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P removePrefixNamespaceDeclaration(String sel,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	
	/**
	 * Builds a document patching instructions, by comparing two versions of a
	 * document. It requires documents to exist, that is this method should not
	 * be called for operations which add/remove document.

	 * @param originalDocument
	 * @param patchedDocument
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public P[] buildPatchInstructions(D originalDocument, D patchedDocument)
			throws BuildPatchException;
}
