/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xcap.diff.component;

import org.mobicents.protocols.xcap.diff.BuildPatchException;
import org.mobicents.protocols.xml.patch.XmlPatchOperationsBuilder;

/**
 * 
 * Builds patch components indicating changes in a document.
 * 
 * @author baranowb
 * @author martins
 * 
 * @param <C>
 *            the component type, defines what is the concrete type of each
 *            patch component, to be aggregated in a XCAP DIFF patch
 * @param <D>
 *            the document type, defines what is the concrete type of XML
 *            documents used
 * @param <E>
 *            the element type, defines what is the concrete type of XML
 *            elements used
 * 
 * @param <N>
 *            the xml patch ops node type, defines what is the concrete type of
 *            XML patch ops node params used by
 *            {@link XmlPatchOperationsBuilder}
 */
public interface DocumentPatchComponentBuilder<C, D, E, N> {

	/**
	 * Builds a document patch component, from an array of patching
	 * instructions.
	 * 
	 * @param sel
	 *            the document selector, relative to the xcap root of the final
	 *            patch.
	 * @param previousETag
	 *            the previous etag of the document.
	 * @param newETag
	 *            the new etag of the document.
	 * @param patchingInstructions
	 *            instructions to be used as body of patch component returned.
	 *            Instruction can be <b>add</b> or <b>remove</b> for instance.
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public C buildPatchComponent(String sel, String previousETag,
			String newETag, C[] patchingInstructions)
			throws BuildPatchException;

	/**
	 * Retrieves the patching instruction which indicates that the document body
	 * did not change.
	 * 
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public C getBodyNotChangedPatchingInstruction() throws BuildPatchException;

	/**
	 * Retrieves the XML Patch Ops (RFC 5261) builder, used to build patching
	 * instructions.
	 * 
	 * @return
	 */
	public XmlPatchOperationsBuilder<D,C, E, N> getXmlPatchOperationsBuilder();

}
