/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xcap.diff.component;

import java.util.Map;

import org.mobicents.protocols.xcap.diff.BuildPatchException;

/**
 * Builds attribute patch components. Generated content is valid for xcap-diff
 * document, ie, xcap-diff document could look like follows(bold content is
 * generated by instance of this interface):
 * 
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *    &lt;xcap-diff xmlns="urn:ietf:params:xml:ns:xcap-diff"
 *               xcap-root="http://xcap.example.com/"&gt;
 * 
 *     <b>&lt;attribute sel="tests/users/sip:joe@example.com/index/~~/doc/@id"&gt;bar&lt;/attribute&gt;</b>
 * 
 *    &lt;/xcap-diff&gt;
 * </pre>
 * 
 * @author baranowb
 * @author martins
 * 
 * @param <C>
 *            the component type, defines what is the concrete type of each
 *            patch component, to be aggregated in a XCAP DIFF patch
 */
public interface AttributePatchComponentBuilder<C> {

	/**
	 * Builds a patch component which indicates an update of an attribute value.
	 * 
	 * @param sel
	 *            the selector which points to the attribute in the owner
	 *            document, relative to the final patch xcap root.
	 * @param attributeValue
	 *            the updated value of the attribute
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public C buildPatchComponent(String sel, String attributeValue,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds a patch component which indicates the non existence of an attribute.
	 * 
	 * @param sel
	 *            the selector which points to the attribute in the owner
	 *            document, relative to the final patch xcap root.
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public C buildPatchComponent(String sel, Map<String, String> namespaceBindings) throws BuildPatchException;

}
