/*

This software is OSI Certified Open Source Software.
OSI Certified is a certification mark of the Open Source Initiative.

The license (Mozilla version 1.0) can be read at the MMBase site.
See http://www.MMBase.org/license

*/
package org.mmbase.util.xml;

import java.util.*;
import java.net.URL;
import java.io.IOException;
import org.mmbase.util.*;
import org.mmbase.util.logging.*;
import org.mmbase.core.event.*;
import org.w3c.dom.Element;
/**
 * This class reads configuration files for utilities, that are
 * placed in /config/utils/.
 *
 * A typical way to use it may be like so:
 <pre>
    private UtilReader.PropertiesMap utilProperties = new UtilReader("myutil.xml", new Runnable() { public void run() { init();}}).getProperties();
    private void init() {
      // use utilProperties
    }
    {
      init();
    }
 </pre>
 * This produces a 'watched map' utilProperties. Every time the
 * underlying config file(s) are changed 'init' is called. Init is
 * called on instantation of the surrounding class too. The map is
 * unmodifiable, and only mirrors the resource(s) "utils/myutil.xml".
 *
 * @since MMBase-1.6.4
 * @author Rob Vermeulen
 * @author Michiel Meeuwissen
 * @version $Id$
 */
public class UtilReader {

    private static final Logger log = Logging.getLoggerInstance(UtilReader.class);

    public static final String CONFIG_UTILS = "utils";

    /** Public ID of the Utilities config DTD version 1.0 */
    public static final String PUBLIC_ID_UTIL_1_0 = "-//MMBase//DTD util config 1.0//EN";
    /** DTD resource filename of the Utilities config DTD version 1.0 */
    public static final String DTD_UTIL_1_0 = "util_1_0.dtd";

    /** Public ID of the most recent Utilities config DTD */
    public static final String PUBLIC_ID_UTIL = PUBLIC_ID_UTIL_1_0;
    /** DTD respource filename of the most recent Utilities config DTD */
    public static final String DTD_UTIL = DTD_UTIL_1_0;

    /**
     * Register the Public Ids for DTDs used by UtilReader
     * This method is called by EntityResolver.
     */
    public static void registerPublicIDs() {
        EntityResolver.registerPublicID(PUBLIC_ID_UTIL_1_0, DTD_UTIL_1_0, UtilReader.class);
    }

    private static final Map<String, UtilReader> utilReaders = new HashMap<String, UtilReader>();     // file-name -> utilreader

    /**
     * Returns a UtilReader for the given fileName. When you use this, the UtilReader instance will be cached.
     *
     * @since MMBase-1.8
     */

    public static UtilReader get(String fileName) {
        UtilReader utilReader = utilReaders.get(fileName);
        if (utilReader == null) {
            synchronized(utilReaders) {
                utilReader = new UtilReader(fileName);
                utilReaders.put(fileName, utilReader);
            }
        }
        return utilReader;
    }

    static {
        // doesnt startup, probably because of cyclic referecnes, if this happens in DocumentReader itself.
        registerPublicIDs();
        DocumentReader.utilProperties = UtilReader.get("documentreader.xml").getProperties();


    }

    private class UtilFileWatcher extends ResourceWatcher {
        private ResourceWatcher wrappedWatcher;
        public UtilFileWatcher(ResourceWatcher f) {
            super(); // true: keep reading.
            wrappedWatcher = f;
        }

        @Override
        public void onChange(String f) {
            readProperties(f);
            if (wrappedWatcher != null) {
                wrappedWatcher.onChange(f);
            }
        }
    }

    private final Map<String, String> properties = new HashMap<String, String>();
    private final Map<String,Collection<Map.Entry<String, String>>> maps = new HashMap<String, Collection<Map.Entry<String,String>>>();
    private final String file;
    private ResourceWatcher watcher;



    /**
     * Instantiates a UtilReader for a given configuration file in <config>/utils. If the configuration file is used on more spots, then you may consider
     * using the static method {@link #get(String)} in stead.
     *
     * @param fileName The name of the property file (e.g. httppost.xml).
     */
    public UtilReader(String fileName) {
        file = CONFIG_UTILS + "/" + fileName;
        readProperties(file);
        startWatcher(null);

    }
    /**
     * Produces a UtilReader for the given resource name.
     * @param fileName a Resource name relative to config/utils
     * @param w A unstarted ResourceWatcher without files. (It will be only be called from the
     *          filewatcher in this reader). It defines what must happen if something changes in the util's
     *          configuration. Since you probably don't need the resource name for that any more, you
     *          can also simply use {@link #UtilReader(String, Runnable)}
     * @since MMBase-1.8
     */
    public UtilReader(String fileName, ResourceWatcher w) {
        file =  CONFIG_UTILS + "/" + fileName;
        readProperties(file);
        startWatcher(w);

    }
    /**
     * Produces a UtilReader for the given resource name.
     * @param resourceName a Resource name relative to config/utils
     * @param onChange     A Runnable defining what must happen if something changes.
     * @since MMBase-1.8
     */
    public UtilReader(String resourceName, final Runnable onChange) {
        this(resourceName, new ResourceWatcher(ResourceLoader.getConfigurationRoot(), false) {
            @Override
                public void onChange(String name) {
                    onChange.run();
                }
            }
            );
    }

    /**
     * @since MMBase-2.0
     */
    protected final void startWatcher(final ResourceWatcher w) {
        EventManager.getInstance().addEventListener(new SystemEventListener() {
                @Override
                public void notify(SystemEvent se) {
                    if (se instanceof EventManager.Ready) {
                        watcher = new UtilFileWatcher(w);
                        watcher.add(file);
                        watcher.start();
                        EventManager.getInstance().removeEventListener(this);
                    }
                }
                @Override
                public int getWeight() {
                    return 0;
                }
            });
    }

    @Override
    public void finalize() throws Throwable {
        super.finalize();
        if (watcher != null) {
            watcher.exit();
        }
    }

    /**
     * Get the properties of this utility.
     */
    public PropertiesMap<String> getProperties() {
        return new PropertiesMap<String>(properties);
    }

    /**
     * Get the properties of this utility.
     * @since MMBase-1.8.6
     */
    public PropertiesMap<Collection<Map.Entry<String, String>>> getMaps() {
        return new PropertiesMap<Collection<Map.Entry<String, String>>>(maps);
    }


    /**
     * Reports whether the configured resource (in the constructor) is actually backed. If not,
     * getProperties will certainly return an empty Map.
     * @since MMBase-1.8.1
     */
    public boolean resourceAvailable() {
        try {
            return ResourceLoader.getConfigurationRoot().getResource(file).openConnection().getDoInput();
        } catch (IOException io) {
            return false;
        }
    }

    /**
     * @since MMBase-1.9.1
     */
    protected Map.Entry<String, String> getEntry(DocumentReader reader, String k, String v) {
        return new Entry<String,String>(k, v);
    }



    protected final void readProperties(String s) {
        properties.clear();
        maps.clear();

        ResourceLoader configLoader = ResourceLoader.getConfigurationRoot();
        List<URL> configList = configLoader.getResourceList(s);
        for (URL url : configList) {
            org.xml.sax.InputSource is;
            try {
                is = ResourceLoader.getInputSource(url);
            } catch (IOException ioe) {
                // input source does not exist
                log.debug(ioe.getMessage() + " for " + url);
                continue;
            }
            if (is != null) {
                log.debug("Reading " + url);
                DocumentReader reader = new DocumentReader(is, false, true, UtilReader.class);
                Element e = reader.getElementByPath("util.properties");
                if (e != null) {
                    for (Element p : DocumentReader.getChildElements(e, "property")) {
                        String name = reader.getElementAttributeValue(p, "name");
                        String type = reader.getElementAttributeValue(p, "type");
                        if ("mergingmap".equals(type) ||
                            "map".equals(type)) {
                            Collection<Map.Entry<String, String>> entryList = null;
                            if ("mergingmap".equals(type)) {
                                entryList = maps.get(name);
                            }

                            if (entryList == null) {
                                entryList = new ArrayList<Map.Entry<String,String>>();
                            }

                            for (Element entry : DocumentReader.getChildElements(p, "entry")) {
                                String key = null;
                                String value = null;

                                for (Element keyorvalue : DocumentReader.getChildElements(entry, "*")) {
                                    if (keyorvalue.getTagName().equals("key")) {
                                        key = DocumentReader.getElementValue(keyorvalue);
                                    } else {
                                        value = DocumentReader.getNodeTextValue(keyorvalue, false);
                                    }
                                }
                                if (key != null) {
                                    entryList.add(getEntry(reader, key, value));
                                }
                            }
                            if (maps.containsKey(name) && ! "mergingmap".equals(type)) {
                                log.debug("Property '" + name + "' (" + entryList + ") of " + url + " is shadowed");
                            } else {
                                maps.put(name, entryList);
                            }
                        } else {
                            String value = DocumentReader.getElementValue(p);
                            Map.Entry<String, String> entry = getEntry(reader, name, value);
                            if (properties.containsKey(entry.getKey())) {
                                log.debug("Property '" + entry.getKey() + "' ('" + entry.getValue() + "') of " + url + " is shadowed");
                            } else {
                                properties.put(entry.getKey(), entry.getValue());
                            }
                        }
                    }
                }
            } else {
                log.debug("Resource " + s + " does not exist");
            }

        }
        if (properties.isEmpty() && maps.isEmpty()) {
            log.service("No properties read from " + configList);
        } else {
            log.service("Read " + properties.entrySet() + " from " + configList);
        }
    }

    /**
     * A unmodifiable Map, with extra 'Properties'-like methods. The entries of this Map are
     * typically backed by the resources of an UtilReader (and the Map dynamically changes if the
     * resources change).
     * @since MMBase-1.8
     */

    public static class PropertiesMap<E> extends AbstractMap<String, E> {

        private final Map<String, E> wrappedMap;

        /**
         * Creates an empty Map (not very useful since this Map is unmodifiable).
         */
        public PropertiesMap() {
            wrappedMap = new HashMap<String, E>();
        }

        /**
         * Wrapping the given map.
         */
        public PropertiesMap(Map<String, E> map) {
            wrappedMap = map;
        }
        /**
         * {@inheritDoc}
         */
        @Override
        public Set<Map.Entry<String, E>> entrySet() {
            return new EntrySet();

        }

        /**
         * Returns the object mapped with 'key', or defaultValue if there is none.
         */
        public E getProperty(String key, E defaultValue) {
            E result = get(key);
            return result == null ? defaultValue : result;
        }

        private class  EntrySet extends AbstractSet<Map.Entry<String, E>> {
            EntrySet() {}
            @Override
            public int size() {
                return PropertiesMap.this.wrappedMap.size();
            }
            @Override
            public Iterator<Map.Entry<String, E>> iterator() {
                return new EntrySetIterator();
            }
        }
        private class EntrySetIterator implements Iterator<Map.Entry<String, E>> {
            private Iterator<Map.Entry<String, E>> i;
            EntrySetIterator() {
                i = PropertiesMap.this.wrappedMap.entrySet().iterator();
            }
            @Override
            public boolean hasNext() {
                return i.hasNext();
            }
            @Override
            public Map.Entry<String, E> next() {
                return i.next();
            }
            @Override
            public void remove() {
                throw new UnsupportedOperationException("Unmodifiable");
            }
        }
    }


    @Override
    public String toString() {
        return super.toString() + " " + file;
    }


}
