/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.util.cc;

import java.io.FileNotFoundException;
import java.net.URI;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Driver;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.util.TestUtil;
import org.neo4j.driver.util.cc.ClusterDrivers;
import org.neo4j.driver.util.cc.ClusterMember;
import org.neo4j.driver.util.cc.ClusterMemberRole;
import org.neo4j.driver.util.cc.ClusterMemberRoleDiscoveryFactory;
import org.neo4j.driver.util.cc.ClusterUnavailableException;
import org.neo4j.driver.util.cc.SharedCluster;

public class Cluster
implements AutoCloseable {
    private static final String ADMIN_USER = "neo4j";
    private static final int STARTUP_TIMEOUT_SECONDS = 120;
    private static final int ONLINE_MEMBERS_CHECK_SLEEP_MS = 500;
    private final Path path;
    private final Set<ClusterMember> members;
    private final Set<ClusterMember> offlineMembers;
    private final ClusterDrivers clusterDrivers;

    public Cluster(Path path, String password) {
        this(path, Collections.emptySet(), new ClusterDrivers(ADMIN_USER, password));
    }

    private Cluster(Path path, Set<ClusterMember> members, ClusterDrivers clusterDrivers) {
        this.path = path;
        this.members = members;
        this.offlineMembers = new HashSet<ClusterMember>();
        this.clusterDrivers = clusterDrivers;
    }

    Cluster withMembers(Set<ClusterMember> newMembers) throws ClusterUnavailableException {
        Cluster.waitForMembersToBeOnline(newMembers, this.clusterDrivers);
        return new Cluster(this.path, newMembers, this.clusterDrivers);
    }

    public URI getRoutingUri() {
        return Cluster.randomOf(this.cores()).getRoutingUri();
    }

    public Path getPath() {
        return this.path;
    }

    public void deleteData() {
        Driver driverToLeader = this.clusterDrivers.getDriver(this.leader());
        Bookmark bookmark = TestUtil.cleanDb(driverToLeader);
        if (bookmark == null) {
            throw new IllegalStateException("Cleanup of the database did not produce a bookmark");
        }
        for (ClusterMember member : this.members) {
            Driver driver = this.clusterDrivers.getDriver(member);
            long nodeCount = TestUtil.countNodes(driver, bookmark);
            if (nodeCount == 0L) continue;
            throw new IllegalStateException("Not all nodes have been deleted. " + nodeCount + " still there somehow");
        }
    }

    public Set<ClusterMember> members() {
        return Collections.unmodifiableSet(this.members);
    }

    public ClusterMember leader() {
        Set<ClusterMember> leaders = this.membersWithRole(ClusterMemberRole.LEADER);
        if (leaders.size() != 1) {
            throw new IllegalStateException("Single leader expected. " + leaders);
        }
        return leaders.iterator().next();
    }

    public ClusterMember anyFollower() {
        return Cluster.randomOf(this.followers());
    }

    public Set<ClusterMember> followers() {
        return this.membersWithRole(ClusterMemberRole.FOLLOWER);
    }

    public ClusterMember anyReadReplica() {
        return Cluster.randomOf(this.readReplicas());
    }

    public Set<ClusterMember> cores() {
        Set<ClusterMember> readReplicas = this.membersWithRole(ClusterMemberRole.READ_REPLICA);
        HashSet<ClusterMember> cores = new HashSet<ClusterMember>(this.members);
        cores.removeAll(readReplicas);
        return cores;
    }

    public Set<ClusterMember> readReplicas() {
        return this.membersWithRole(ClusterMemberRole.READ_REPLICA);
    }

    public void start(ClusterMember member) {
        this.startNoWait(member);
        this.waitForMembersToBeOnline();
    }

    public Driver getDirectDriver(ClusterMember member) {
        return this.clusterDrivers.getDriver(member);
    }

    public void dumpClusterDebugLog() {
        for (ClusterMember member : this.members) {
            System.out.println("Debug log for: " + member.getPath().toString());
            try {
                member.dumpDebugLog();
            }
            catch (FileNotFoundException e) {
                System.out.println("Unable to find debug log file for: " + member.getPath().toString());
                e.printStackTrace();
            }
        }
    }

    @Override
    public void close() {
        this.clusterDrivers.close();
    }

    public String toString() {
        return "Cluster{path=" + this.path + ", members=" + this.members + "}";
    }

    private void addOfflineMember(ClusterMember member) {
        if (!this.offlineMembers.remove(member)) {
            throw new IllegalArgumentException("Cluster member is not offline: " + member);
        }
        this.members.add(member);
    }

    private void startNoWait(ClusterMember member) {
        this.addOfflineMember(member);
        SharedCluster.start(member);
    }

    private Set<ClusterMember> membersWithRole(ClusterMemberRole role) {
        HashSet<ClusterMember> membersWithRole = new HashSet<ClusterMember>();
        int retryCount = 0;
        while (membersWithRole.isEmpty() && retryCount < 10) {
            Driver driver = Cluster.driverToAnyCore(this.members, this.clusterDrivers);
            ClusterMemberRoleDiscoveryFactory.ClusterMemberRoleDiscovery discovery = this.clusterDrivers.getDiscovery();
            Map<BoltServerAddress, ClusterMemberRole> clusterOverview = discovery.findClusterOverview(driver);
            for (BoltServerAddress boltAddress : clusterOverview.keySet()) {
                if (role != clusterOverview.get(boltAddress)) continue;
                ClusterMember member = Cluster.findByBoltAddress(boltAddress, this.members);
                if (member == null) {
                    throw new IllegalStateException("Unknown cluster member: '" + boltAddress + "'\n" + this);
                }
                membersWithRole.add(member);
            }
            ++retryCount;
            if (!membersWithRole.isEmpty()) break;
            try {
                Thread.sleep(2000L);
            }
            catch (InterruptedException interruptedException) {}
        }
        if (membersWithRole.isEmpty()) {
            throw new IllegalStateException("No cluster members with role '" + (Object)((Object)role) + " " + this);
        }
        return membersWithRole;
    }

    private void waitForMembersToBeOnline() {
        try {
            Cluster.waitForMembersToBeOnline(this.members, this.clusterDrivers);
        }
        catch (ClusterUnavailableException e) {
            throw new RuntimeException(e);
        }
    }

    private static void waitForMembersToBeOnline(Set<ClusterMember> members, ClusterDrivers clusterDrivers) throws ClusterUnavailableException {
        if (members.isEmpty()) {
            throw new IllegalArgumentException("No members to wait for");
        }
        Set<BoltServerAddress> expectedOnlineAddresses = Cluster.extractBoltAddresses(members);
        Set<Object> actualOnlineAddresses = Collections.emptySet();
        long deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(120L);
        Throwable error = null;
        while (!expectedOnlineAddresses.equals(actualOnlineAddresses)) {
            TestUtil.sleep(500);
            Cluster.assertDeadlineNotReached(deadline, expectedOnlineAddresses, actualOnlineAddresses, error);
            Driver driver = Cluster.driverToAnyCore(members, clusterDrivers);
            ClusterMemberRoleDiscoveryFactory.ClusterMemberRoleDiscovery discovery = clusterDrivers.getDiscovery();
            try {
                Map<BoltServerAddress, ClusterMemberRole> clusterOverview = discovery.findClusterOverview(driver);
                actualOnlineAddresses = clusterOverview.keySet();
            }
            catch (Throwable t) {
                t.printStackTrace();
                if (error == null) {
                    error = t;
                    continue;
                }
                error.addSuppressed(t);
            }
        }
    }

    private static Driver driverToAnyCore(Set<ClusterMember> members, ClusterDrivers clusterDrivers) {
        if (members.isEmpty()) {
            throw new IllegalArgumentException("No members, can't create driver");
        }
        for (ClusterMember member : members) {
            Driver driver = clusterDrivers.getDriver(member);
            ClusterMemberRoleDiscoveryFactory.ClusterMemberRoleDiscovery discovery = clusterDrivers.getDiscovery();
            if (!discovery.isCoreMember(driver)) continue;
            return driver;
        }
        throw new IllegalStateException("No core members found among: " + members);
    }

    private static void assertDeadlineNotReached(long deadline, Set<?> expectedAddresses, Set<?> actualAddresses, Throwable error) throws ClusterUnavailableException {
        if (System.currentTimeMillis() > deadline) {
            String baseMessage = "Cluster did not become available in 120 seconds.\n";
            String errorMessage = error == null ? "" : "There were errors checking cluster members.\n";
            String expectedAddressesMessage = "Expected online addresses: " + expectedAddresses + "\n";
            String actualAddressesMessage = "Actual last seen online addresses: " + actualAddresses + "\n";
            String message = baseMessage + errorMessage + expectedAddressesMessage + actualAddressesMessage;
            ClusterUnavailableException clusterUnavailable = new ClusterUnavailableException(message);
            if (error != null) {
                clusterUnavailable.addSuppressed(error);
            }
            throw clusterUnavailable;
        }
    }

    private static Set<BoltServerAddress> extractBoltAddresses(Set<ClusterMember> members) {
        HashSet<BoltServerAddress> addresses = new HashSet<BoltServerAddress>();
        for (ClusterMember member : members) {
            addresses.add(member.getBoltAddress());
        }
        return addresses;
    }

    private static ClusterMember findByBoltAddress(BoltServerAddress boltAddress, Set<ClusterMember> members) {
        for (ClusterMember member : members) {
            if (!member.getBoltAddress().equals((Object)boltAddress)) continue;
            return member;
        }
        return null;
    }

    private static ClusterMember randomOf(Set<ClusterMember> members) {
        int randomIndex = ThreadLocalRandom.current().nextInt(members.size());
        int currentIndex = 0;
        for (ClusterMember member : members) {
            if (currentIndex == randomIndex) {
                return member;
            }
            ++currentIndex;
        }
        throw new AssertionError();
    }
}

