/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.async.inbound;

import io.netty.channel.Channel;
import io.netty.channel.ChannelConfig;
import io.netty.channel.ChannelPromise;
import io.netty.channel.DefaultChannelId;
import io.netty.util.Attribute;
import io.netty.util.AttributeKey;
import java.util.Collections;
import java.util.HashMap;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;
import org.mockito.ArgumentCaptor;
import org.mockito.ArgumentMatchers;
import org.mockito.InOrder;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.driver.Logger;
import org.neo4j.driver.Logging;
import org.neo4j.driver.Value;
import org.neo4j.driver.Values;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.Neo4jException;
import org.neo4j.driver.internal.async.inbound.InboundMessageDispatcher;
import org.neo4j.driver.internal.logging.ChannelActivityLogger;
import org.neo4j.driver.internal.logging.ChannelErrorLogger;
import org.neo4j.driver.internal.logging.DevNullLogging;
import org.neo4j.driver.internal.messaging.Message;
import org.neo4j.driver.internal.messaging.request.ResetMessage;
import org.neo4j.driver.internal.messaging.response.FailureMessage;
import org.neo4j.driver.internal.messaging.response.IgnoredMessage;
import org.neo4j.driver.internal.messaging.response.RecordMessage;
import org.neo4j.driver.internal.messaging.response.SuccessMessage;
import org.neo4j.driver.internal.spi.ResponseHandler;
import org.neo4j.driver.internal.value.IntegerValue;

class InboundMessageDispatcherTest {
    private static final String FAILURE_CODE = "Neo.ClientError.Security.Unauthorized";
    private static final String FAILURE_MESSAGE = "Error Message";

    InboundMessageDispatcherTest() {
    }

    @Test
    void shouldFailWhenCreatedWithNullChannel() {
        Assertions.assertThrows(NullPointerException.class, () -> new InboundMessageDispatcher(null, DevNullLogging.DEV_NULL_LOGGING));
    }

    @Test
    void shouldFailWhenCreatedWithNullLogging() {
        Assertions.assertThrows(NullPointerException.class, () -> new InboundMessageDispatcher(InboundMessageDispatcherTest.newChannelMock(), null));
    }

    @Test
    void shouldDequeHandlerOnSuccess() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
        HashMap<String, Value> metadata = new HashMap<String, Value>();
        metadata.put("key1", Values.value((int)1));
        metadata.put("key2", Values.value((String)"2"));
        dispatcher.handleSuccessMessage(metadata);
        Assertions.assertEquals((int)0, (int)dispatcher.queuedHandlersCount());
        ((ResponseHandler)Mockito.verify((Object)handler)).onSuccess(metadata);
    }

    @Test
    void shouldDequeHandlerOnFailure() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
        dispatcher.handleFailureMessage(FAILURE_CODE, FAILURE_MESSAGE);
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
        InboundMessageDispatcherTest.verifyFailure(handler);
        Assertions.assertEquals((Object)FAILURE_CODE, (Object)((Neo4jException)dispatcher.currentError()).code());
        Assertions.assertEquals((Object)FAILURE_MESSAGE, (Object)dispatcher.currentError().getMessage());
    }

    @Test
    void shouldSendResetOnFailure() {
        Channel channel = InboundMessageDispatcherTest.newChannelMock();
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher(channel);
        dispatcher.enqueue((ResponseHandler)Mockito.mock(ResponseHandler.class));
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
        dispatcher.handleFailureMessage(FAILURE_CODE, FAILURE_MESSAGE);
        ((Channel)Mockito.verify((Object)channel)).writeAndFlush(ArgumentMatchers.eq((Object)ResetMessage.RESET), (ChannelPromise)ArgumentMatchers.any());
    }

    @Test
    void shouldClearFailureOnSuccessOfResetAfterFailure() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        dispatcher.enqueue((ResponseHandler)Mockito.mock(ResponseHandler.class));
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
        dispatcher.handleFailureMessage(FAILURE_CODE, FAILURE_MESSAGE);
        dispatcher.handleSuccessMessage(Collections.emptyMap());
        Assertions.assertNull((Object)dispatcher.currentError());
    }

    @Test
    void shouldPeekHandlerOnRecord() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
        Value[] fields1 = new Value[]{new IntegerValue(1L)};
        Value[] fields2 = new Value[]{new IntegerValue(2L)};
        Value[] fields3 = new Value[]{new IntegerValue(3L)};
        dispatcher.handleRecordMessage(fields1);
        dispatcher.handleRecordMessage(fields2);
        dispatcher.handleRecordMessage(fields3);
        ((ResponseHandler)Mockito.verify((Object)handler)).onRecord(fields1);
        ((ResponseHandler)Mockito.verify((Object)handler)).onRecord(fields2);
        ((ResponseHandler)Mockito.verify((Object)handler)).onRecord(fields3);
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
    }

    @Test
    void shouldFailAllHandlersOnChannelError() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler1 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        ResponseHandler handler2 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        ResponseHandler handler3 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler1);
        dispatcher.enqueue(handler2);
        dispatcher.enqueue(handler3);
        RuntimeException fatalError = new RuntimeException("Fatal!");
        dispatcher.handleChannelError((Throwable)fatalError);
        InOrder inOrder = Mockito.inOrder((Object[])new Object[]{handler1, handler2, handler3});
        ((ResponseHandler)inOrder.verify((Object)handler1)).onFailure((Throwable)fatalError);
        ((ResponseHandler)inOrder.verify((Object)handler2)).onFailure((Throwable)fatalError);
        ((ResponseHandler)inOrder.verify((Object)handler3)).onFailure((Throwable)fatalError);
    }

    @Test
    void shouldFailNewHandlerAfterChannelError() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        RuntimeException fatalError = new RuntimeException("Fatal!");
        dispatcher.handleChannelError((Throwable)fatalError);
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        ((ResponseHandler)Mockito.verify((Object)handler)).onFailure((Throwable)fatalError);
    }

    @Test
    void shouldAttachChannelErrorOnExistingError() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        dispatcher.handleFailureMessage("Neo.ClientError", "First error!");
        RuntimeException fatalError = new RuntimeException("Second Error!");
        dispatcher.handleChannelError((Throwable)fatalError);
        ((ResponseHandler)Mockito.verify((Object)handler)).onFailure((Throwable)ArgumentMatchers.argThat(error -> error instanceof ClientException && error.getMessage().equals("First error!") && error.getSuppressed().length == 1 && error.getSuppressed()[0].getMessage().equals("Second Error!")));
    }

    @Test
    void shouldDequeHandlerOnIgnored() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        dispatcher.handleIgnoredMessage();
        Assertions.assertEquals((int)0, (int)dispatcher.queuedHandlersCount());
    }

    @Test
    void shouldFailHandlerOnIgnoredMessageWithExistingError() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler1 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        ResponseHandler handler2 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler1);
        dispatcher.enqueue(handler2);
        dispatcher.handleFailureMessage(FAILURE_CODE, FAILURE_MESSAGE);
        InboundMessageDispatcherTest.verifyFailure(handler1);
        ((ResponseHandler)Mockito.verify((Object)handler2, (VerificationMode)Mockito.only())).canManageAutoRead();
        dispatcher.handleIgnoredMessage();
        InboundMessageDispatcherTest.verifyFailure(handler2);
    }

    @Test
    void shouldFailHandlerOnIgnoredMessageWhenNoErrorAndNotHandlingReset() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        dispatcher.handleIgnoredMessage();
        ((ResponseHandler)Mockito.verify((Object)handler)).onFailure((Throwable)ArgumentMatchers.any(ClientException.class));
    }

    @Test
    void shouldDequeAndFailHandlerOnIgnoredWhenErrorHappened() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler1 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        ResponseHandler handler2 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler1);
        dispatcher.enqueue(handler2);
        dispatcher.handleFailureMessage(FAILURE_CODE, FAILURE_MESSAGE);
        dispatcher.handleIgnoredMessage();
        Assertions.assertEquals((int)1, (int)dispatcher.queuedHandlersCount());
        InboundMessageDispatcherTest.verifyFailure(handler1);
        InboundMessageDispatcherTest.verifyFailure(handler2);
    }

    @Test
    void shouldThrowWhenNoHandlerToHandleRecordMessage() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        Assertions.assertThrows(IllegalStateException.class, () -> dispatcher.handleRecordMessage(new Value[]{Values.value((int)1), Values.value((int)2)}));
    }

    @Test
    void shouldKeepSingleAutoReadManagingHandler() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler1 = InboundMessageDispatcherTest.newAutoReadManagingResponseHandler();
        ResponseHandler handler2 = InboundMessageDispatcherTest.newAutoReadManagingResponseHandler();
        ResponseHandler handler3 = InboundMessageDispatcherTest.newAutoReadManagingResponseHandler();
        dispatcher.enqueue(handler1);
        dispatcher.enqueue(handler2);
        dispatcher.enqueue(handler3);
        InOrder inOrder = Mockito.inOrder((Object[])new Object[]{handler1, handler2, handler3});
        ((ResponseHandler)inOrder.verify((Object)handler1)).disableAutoReadManagement();
        ((ResponseHandler)inOrder.verify((Object)handler2)).disableAutoReadManagement();
        ((ResponseHandler)inOrder.verify((Object)handler3, Mockito.never())).disableAutoReadManagement();
    }

    @Test
    void shouldKeepTrackOfAutoReadManagingHandler() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler1 = InboundMessageDispatcherTest.newAutoReadManagingResponseHandler();
        ResponseHandler handler2 = InboundMessageDispatcherTest.newAutoReadManagingResponseHandler();
        Assertions.assertNull((Object)dispatcher.autoReadManagingHandler());
        dispatcher.enqueue(handler1);
        Assertions.assertEquals((Object)handler1, (Object)dispatcher.autoReadManagingHandler());
        dispatcher.enqueue(handler2);
        Assertions.assertEquals((Object)handler2, (Object)dispatcher.autoReadManagingHandler());
    }

    @Test
    void shouldForgetAutoReadManagingHandlerWhenItIsRemoved() {
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher();
        ResponseHandler handler1 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        ResponseHandler handler2 = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        ResponseHandler handler3 = InboundMessageDispatcherTest.newAutoReadManagingResponseHandler();
        dispatcher.enqueue(handler1);
        dispatcher.enqueue(handler2);
        dispatcher.enqueue(handler3);
        Assertions.assertEquals((Object)handler3, (Object)dispatcher.autoReadManagingHandler());
        dispatcher.handleSuccessMessage(Collections.emptyMap());
        dispatcher.handleSuccessMessage(Collections.emptyMap());
        dispatcher.handleSuccessMessage(Collections.emptyMap());
        Assertions.assertNull((Object)dispatcher.autoReadManagingHandler());
    }

    @Test
    void shouldReEnableAutoReadWhenAutoReadManagingHandlerIsRemoved() {
        Channel channel = InboundMessageDispatcherTest.newChannelMock();
        InboundMessageDispatcher dispatcher = InboundMessageDispatcherTest.newDispatcher(channel);
        ResponseHandler handler = InboundMessageDispatcherTest.newAutoReadManagingResponseHandler();
        dispatcher.enqueue(handler);
        Assertions.assertEquals((Object)handler, (Object)dispatcher.autoReadManagingHandler());
        ((ResponseHandler)Mockito.verify((Object)handler, (VerificationMode)Mockito.never())).disableAutoReadManagement();
        ((ChannelConfig)Mockito.verify((Object)channel.config(), (VerificationMode)Mockito.never())).setAutoRead(ArgumentMatchers.anyBoolean());
        dispatcher.handleSuccessMessage(Collections.emptyMap());
        Assertions.assertNull((Object)dispatcher.autoReadManagingHandler());
        ((ResponseHandler)Mockito.verify((Object)handler)).disableAutoReadManagement();
        ((ChannelConfig)Mockito.verify((Object)channel.config())).setAutoRead(ArgumentMatchers.anyBoolean());
    }

    @ParameterizedTest
    @ValueSource(classes={SuccessMessage.class, FailureMessage.class, RecordMessage.class, IgnoredMessage.class})
    void shouldCreateChannelActivityLoggerAndLogDebugMessageOnMessageHandling(Class<? extends Message> message) {
        Channel channel = InboundMessageDispatcherTest.newChannelMock();
        Logging logging = (Logging)Mockito.mock(Logging.class);
        Logger logger = (Logger)Mockito.mock(Logger.class);
        Mockito.when((Object)logger.isDebugEnabled()).thenReturn((Object)true);
        Mockito.when((Object)logging.getLog(InboundMessageDispatcher.class)).thenReturn((Object)logger);
        ChannelErrorLogger errorLogger = (ChannelErrorLogger)Mockito.mock(ChannelErrorLogger.class);
        Mockito.when((Object)logging.getLog(ChannelErrorLogger.class)).thenReturn((Object)errorLogger);
        InboundMessageDispatcher dispatcher = new InboundMessageDispatcher(channel, logging);
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        if (SuccessMessage.class.isAssignableFrom(message)) {
            dispatcher.handleSuccessMessage(new HashMap());
        } else if (FailureMessage.class.isAssignableFrom(message)) {
            dispatcher.handleFailureMessage(FAILURE_CODE, FAILURE_MESSAGE);
        } else if (RecordMessage.class.isAssignableFrom(message)) {
            dispatcher.handleRecordMessage(Values.values((Object[])new Object[0]));
        } else if (IgnoredMessage.class.isAssignableFrom(message)) {
            dispatcher.handleIgnoredMessage();
        } else {
            Assertions.fail((String)"Unexpected message type parameter provided");
        }
        Assertions.assertTrue((boolean)(dispatcher.getLog() instanceof ChannelActivityLogger));
        Assertions.assertTrue((boolean)(dispatcher.getErrorLog() instanceof ChannelErrorLogger));
        ((Logger)Mockito.verify((Object)logger)).debug(ArgumentMatchers.anyString(), new Object[]{ArgumentMatchers.any(Object.class)});
    }

    @Test
    void shouldCreateChannelErrorLoggerAndLogDebugMessageOnChannelError() {
        Channel channel = InboundMessageDispatcherTest.newChannelMock();
        Logging logging = (Logging)Mockito.mock(Logging.class);
        Logger logger = (Logger)Mockito.mock(Logger.class);
        Mockito.when((Object)logger.isDebugEnabled()).thenReturn((Object)true);
        Mockito.when((Object)logging.getLog(InboundMessageDispatcher.class)).thenReturn((Object)logger);
        ChannelErrorLogger errorLogger = (ChannelErrorLogger)Mockito.mock(ChannelErrorLogger.class);
        Mockito.when((Object)errorLogger.isDebugEnabled()).thenReturn((Object)true);
        Mockito.when((Object)logging.getLog(ChannelErrorLogger.class)).thenReturn((Object)errorLogger);
        InboundMessageDispatcher dispatcher = new InboundMessageDispatcher(channel, logging);
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        dispatcher.enqueue(handler);
        Throwable throwable = (Throwable)Mockito.mock(Throwable.class);
        dispatcher.handleChannelError(throwable);
        Assertions.assertTrue((boolean)(dispatcher.getLog() instanceof ChannelActivityLogger));
        Assertions.assertTrue((boolean)(dispatcher.getErrorLog() instanceof ChannelErrorLogger));
        ((ChannelErrorLogger)Mockito.verify((Object)errorLogger)).debug(ArgumentMatchers.contains((String)throwable.getClass().toString()), new Object[0]);
    }

    private static void verifyFailure(ResponseHandler handler) {
        ArgumentCaptor captor = ArgumentCaptor.forClass(Neo4jException.class);
        ((ResponseHandler)Mockito.verify((Object)handler)).onFailure((Throwable)captor.capture());
        Assertions.assertEquals((Object)FAILURE_CODE, (Object)((Neo4jException)captor.getValue()).code());
        Assertions.assertEquals((Object)FAILURE_MESSAGE, (Object)((Neo4jException)captor.getValue()).getMessage());
    }

    private static InboundMessageDispatcher newDispatcher() {
        return InboundMessageDispatcherTest.newDispatcher(InboundMessageDispatcherTest.newChannelMock());
    }

    private static InboundMessageDispatcher newDispatcher(Channel channel) {
        return new InboundMessageDispatcher(channel, DevNullLogging.DEV_NULL_LOGGING);
    }

    private static Channel newChannelMock() {
        Channel channel = (Channel)Mockito.mock(Channel.class);
        Mockito.when((Object)channel.id()).thenReturn((Object)DefaultChannelId.newInstance());
        ChannelConfig channelConfig = (ChannelConfig)Mockito.mock(ChannelConfig.class);
        Mockito.when((Object)channel.config()).thenReturn((Object)channelConfig);
        Attribute attribute = (Attribute)Mockito.mock(Attribute.class);
        Mockito.when((Object)channel.attr((AttributeKey)ArgumentMatchers.any())).thenReturn((Object)attribute);
        return channel;
    }

    private static ResponseHandler newAutoReadManagingResponseHandler() {
        ResponseHandler handler = (ResponseHandler)Mockito.mock(ResponseHandler.class);
        Mockito.when((Object)handler.canManageAutoRead()).thenReturn((Object)true);
        return handler;
    }
}

