/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.integration.async;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import org.hamcrest.Matcher;
import org.hamcrest.junit.MatcherAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Query;
import org.neo4j.driver.Record;
import org.neo4j.driver.SessionConfig;
import org.neo4j.driver.Value;
import org.neo4j.driver.Values;
import org.neo4j.driver.async.AsyncSession;
import org.neo4j.driver.async.AsyncTransaction;
import org.neo4j.driver.async.AsyncTransactionWork;
import org.neo4j.driver.async.ResultCursor;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.DatabaseException;
import org.neo4j.driver.exceptions.NoSuchRecordException;
import org.neo4j.driver.exceptions.ResultConsumedException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.exceptions.SessionExpiredException;
import org.neo4j.driver.exceptions.TransientException;
import org.neo4j.driver.internal.InternalBookmark;
import org.neo4j.driver.internal.util.DisabledOnNeo4jWith;
import org.neo4j.driver.internal.util.EnabledOnNeo4jWith;
import org.neo4j.driver.internal.util.Futures;
import org.neo4j.driver.internal.util.Iterables;
import org.neo4j.driver.internal.util.Matchers;
import org.neo4j.driver.internal.util.Neo4jFeature;
import org.neo4j.driver.summary.QueryType;
import org.neo4j.driver.summary.ResultSummary;
import org.neo4j.driver.types.Node;
import org.neo4j.driver.util.DatabaseExtension;
import org.neo4j.driver.util.ParallelizableIT;
import org.neo4j.driver.util.TestUtil;

@ParallelizableIT
class AsyncSessionIT {
    @RegisterExtension
    static final DatabaseExtension neo4j = new DatabaseExtension();
    private AsyncSession session;

    AsyncSessionIT() {
    }

    @BeforeEach
    void setUp() {
        this.session = neo4j.driver().asyncSession();
    }

    @AfterEach
    void tearDown() {
        this.session.closeAsync();
    }

    @Test
    void shouldRunQueryWithEmptyResult() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("CREATE (:Person)"));
        Assertions.assertNull(TestUtil.await(cursor.nextAsync()));
    }

    @Test
    void shouldRunQueryWithSingleResult() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("CREATE (p:Person {name: 'Nick Fury'}) RETURN p"));
        Record record = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record);
        Node node = record.get(0).asNode();
        Assertions.assertEquals((Object)"Person", (Object)Iterables.single((Iterable)node.labels()));
        Assertions.assertEquals((Object)"Nick Fury", (Object)node.get("name").asString());
        Assertions.assertNull(TestUtil.await(cursor.nextAsync()));
    }

    @Test
    void shouldRunQueryWithMultipleResults() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND [1,2,3] AS x RETURN x"));
        Record record1 = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record1);
        Assertions.assertEquals((int)1, (int)record1.get(0).asInt());
        Record record2 = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record2);
        Assertions.assertEquals((int)2, (int)record2.get(0).asInt());
        Record record3 = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record3);
        Assertions.assertEquals((int)3, (int)record3.get(0).asInt());
        Assertions.assertNull(TestUtil.await(cursor.nextAsync()));
    }

    @Test
    void shouldFailForIncorrectQuery() {
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(this.session.runAsync("RETURN")));
        MatcherAssert.assertThat((Object)e, (Matcher)org.hamcrest.Matchers.is(Matchers.syntaxError()));
    }

    @Test
    void shouldFailWhenQueryFailsAtRuntime() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("CYPHER runtime=interpreted UNWIND [1, 2, 0] AS x RETURN 10 / x"));
        Record record1 = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record1);
        Assertions.assertEquals((int)10, (int)record1.get(0).asInt());
        Record record2 = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record2);
        Assertions.assertEquals((int)5, (int)record2.get(0).asInt());
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(cursor.nextAsync()));
        MatcherAssert.assertThat((Object)e, (Matcher)org.hamcrest.Matchers.is(Matchers.arithmeticError()));
    }

    @Test
    void shouldAllowNestedQueries() {
        Record record;
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND [1, 2, 3] AS x CREATE (p:Person {id: x}) RETURN p"));
        Future<List<CompletionStage<Record>>> queriesExecuted = this.runNestedQueries(cursor);
        List futures = (List)TestUtil.await(queriesExecuted);
        List futureResults = TestUtil.awaitAll(futures);
        Assertions.assertEquals((int)7, (int)futureResults.size());
        ResultCursor personCursor = (ResultCursor)TestUtil.await(this.session.runAsync("MATCH (p:Person) RETURN p ORDER BY p.id"));
        ArrayList<Node> personNodes = new ArrayList<Node>();
        while ((record = (Record)TestUtil.await(personCursor.nextAsync())) != null) {
            personNodes.add(record.get(0).asNode());
        }
        Assertions.assertEquals((int)3, (int)personNodes.size());
        Node node1 = (Node)personNodes.get(0);
        Assertions.assertEquals((int)1, (int)node1.get("id").asInt());
        Assertions.assertEquals((int)10, (int)node1.get("age").asInt());
        Node node2 = (Node)personNodes.get(1);
        Assertions.assertEquals((int)2, (int)node2.get("id").asInt());
        Assertions.assertEquals((int)20, (int)node2.get("age").asInt());
        Node node3 = (Node)personNodes.get(2);
        Assertions.assertEquals((int)3, (int)node3.get("id").asInt());
        Assertions.assertEquals((int)30, (int)((Node)personNodes.get(2)).get("age").asInt());
    }

    @Test
    void shouldAllowMultipleAsyncRunsWithoutConsumingResults() {
        int queryCount = 13;
        ArrayList cursors = new ArrayList();
        for (int i = 0; i < queryCount; ++i) {
            cursors.add(this.session.runAsync("CREATE (:Person)"));
        }
        ArrayList records = new ArrayList();
        for (ResultCursor cursor : TestUtil.awaitAll(cursors)) {
            records.add(cursor.nextAsync());
        }
        TestUtil.awaitAll(records);
        TestUtil.await(this.session.closeAsync());
        this.session = neo4j.driver().asyncSession();
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("MATCH (p:Person) RETURN count(p)"));
        Record record = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record);
        Assertions.assertEquals((int)queryCount, (int)record.get(0).asInt());
    }

    @Test
    void shouldExposeQueryKeysForColumnsWithAliases() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("RETURN 1 AS one, 2 AS two, 3 AS three, 4 AS five"));
        Assertions.assertEquals(Arrays.asList("one", "two", "three", "five"), (Object)cursor.keys());
    }

    @Test
    void shouldExposeQueryKeysForColumnsWithoutAliases() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("RETURN 1, 2, 3, 5"));
        Assertions.assertEquals(Arrays.asList("1", "2", "3", "5"), (Object)cursor.keys());
    }

    @Test
    void shouldExposeResultSummaryForSimpleQuery() {
        String query = "CREATE (:Node {id: $id, name: $name})";
        Value params = Values.parameters((Object[])new Object[]{"id", 1, "name", "TheNode"});
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync(query, params));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertEquals((Object)new Query(query, params), (Object)summary.query());
        Assertions.assertEquals((int)1, (int)summary.counters().nodesCreated());
        Assertions.assertEquals((int)1, (int)summary.counters().labelsAdded());
        Assertions.assertEquals((int)2, (int)summary.counters().propertiesSet());
        Assertions.assertEquals((int)0, (int)summary.counters().relationshipsCreated());
        Assertions.assertEquals((Object)QueryType.WRITE_ONLY, (Object)summary.queryType());
        Assertions.assertFalse((boolean)summary.hasPlan());
        Assertions.assertFalse((boolean)summary.hasProfile());
        Assertions.assertNull((Object)summary.plan());
        Assertions.assertNull((Object)summary.profile());
        Assertions.assertEquals((int)0, (int)summary.notifications().size());
        MatcherAssert.assertThat((Object)summary, Matchers.containsResultAvailableAfterAndResultConsumedAfter());
    }

    @Test
    void shouldExposeResultSummaryForExplainQuery() {
        String query = "EXPLAIN CREATE (),() WITH * MATCH (n)-->(m) CREATE (n)-[:HI {id: 'id'}]->(m) RETURN n, m";
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync(query));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertEquals((Object)new Query(query), (Object)summary.query());
        Assertions.assertEquals((int)0, (int)summary.counters().nodesCreated());
        Assertions.assertEquals((int)0, (int)summary.counters().propertiesSet());
        Assertions.assertEquals((int)0, (int)summary.counters().relationshipsCreated());
        Assertions.assertEquals((Object)QueryType.READ_WRITE, (Object)summary.queryType());
        Assertions.assertTrue((boolean)summary.hasPlan());
        Assertions.assertFalse((boolean)summary.hasProfile());
        Assertions.assertNotNull((Object)summary.plan());
        String planAsString = summary.plan().toString().toLowerCase();
        MatcherAssert.assertThat((Object)planAsString, (Matcher)org.hamcrest.Matchers.containsString((String)"create"));
        MatcherAssert.assertThat((Object)planAsString, (Matcher)org.hamcrest.Matchers.containsString((String)"expand"));
        Assertions.assertNull((Object)summary.profile());
        Assertions.assertEquals((int)0, (int)summary.notifications().size());
        MatcherAssert.assertThat((Object)summary, Matchers.containsResultAvailableAfterAndResultConsumedAfter());
    }

    @Test
    void shouldExposeResultSummaryForProfileQuery() {
        String query = "PROFILE CREATE (:Node)-[:KNOWS]->(:Node) WITH * MATCH (n) RETURN n";
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync(query));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertEquals((Object)new Query(query), (Object)summary.query());
        Assertions.assertEquals((int)2, (int)summary.counters().nodesCreated());
        Assertions.assertEquals((int)0, (int)summary.counters().propertiesSet());
        Assertions.assertEquals((int)1, (int)summary.counters().relationshipsCreated());
        Assertions.assertEquals((Object)QueryType.READ_WRITE, (Object)summary.queryType());
        Assertions.assertTrue((boolean)summary.hasPlan());
        Assertions.assertTrue((boolean)summary.hasProfile());
        Assertions.assertNotNull((Object)summary.plan());
        Assertions.assertNotNull((Object)summary.profile());
        String profileAsString = summary.profile().toString().toLowerCase();
        MatcherAssert.assertThat((Object)profileAsString, (Matcher)org.hamcrest.Matchers.containsString((String)"hits"));
        Assertions.assertEquals((int)0, (int)summary.notifications().size());
        MatcherAssert.assertThat((Object)summary, Matchers.containsResultAvailableAfterAndResultConsumedAfter());
    }

    @Test
    void shouldRunAsyncTransactionWithoutRetries() {
        InvocationTrackingWork work = new InvocationTrackingWork("CREATE (:Apa) RETURN 42");
        CompletionStage txStage = this.session.writeTransactionAsync((AsyncTransactionWork)work);
        Record record = (Record)TestUtil.await(txStage);
        Assertions.assertNotNull((Object)record);
        Assertions.assertEquals((long)42L, (long)record.get(0).asLong());
        Assertions.assertEquals((int)1, (int)work.invocationCount());
        Assertions.assertEquals((long)1L, (long)this.countNodesByLabel("Apa"));
    }

    @Test
    void shouldRunAsyncTransactionWithRetriesOnAsyncFailures() {
        InvocationTrackingWork work = new InvocationTrackingWork("CREATE (:Node) RETURN 24").withAsyncFailures(new RuntimeException[]{new ServiceUnavailableException("Oh!"), new SessionExpiredException("Ah!"), new TransientException("Code", "Message")});
        CompletionStage txStage = this.session.writeTransactionAsync((AsyncTransactionWork)work);
        Record record = (Record)TestUtil.await(txStage);
        Assertions.assertNotNull((Object)record);
        Assertions.assertEquals((long)24L, (long)record.get(0).asLong());
        Assertions.assertEquals((int)4, (int)work.invocationCount());
        Assertions.assertEquals((long)1L, (long)this.countNodesByLabel("Node"));
    }

    @Test
    void shouldRunAsyncTransactionWithRetriesOnSyncFailures() {
        InvocationTrackingWork work = new InvocationTrackingWork("CREATE (:Test) RETURN 12").withSyncFailures(new RuntimeException[]{new TransientException("Oh!", "Deadlock!"), new ServiceUnavailableException("Oh! Network Failure")});
        CompletionStage txStage = this.session.writeTransactionAsync((AsyncTransactionWork)work);
        Record record = (Record)TestUtil.await(txStage);
        Assertions.assertNotNull((Object)record);
        Assertions.assertEquals((long)12L, (long)record.get(0).asLong());
        Assertions.assertEquals((int)3, (int)work.invocationCount());
        Assertions.assertEquals((long)1L, (long)this.countNodesByLabel("Test"));
    }

    @Test
    void shouldRunAsyncTransactionThatCanNotBeRetried() {
        InvocationTrackingWork work = new InvocationTrackingWork("UNWIND [10, 5, 0] AS x CREATE (:Hi) RETURN 10/x");
        CompletionStage txStage = this.session.writeTransactionAsync((AsyncTransactionWork)work);
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(txStage));
        TestUtil.assertNoCircularReferences(e);
        Assertions.assertEquals((int)1, (int)work.invocationCount());
        Assertions.assertEquals((long)0L, (long)this.countNodesByLabel("Hi"));
    }

    @Test
    void shouldRunAsyncTransactionThatCanNotBeRetriedAfterATransientFailure() {
        InvocationTrackingWork work = new InvocationTrackingWork("CREATE (:Person) RETURN 1").withSyncFailures(new RuntimeException[]{new TransientException("Oh!", "Deadlock!")}).withAsyncFailures(new RuntimeException[]{new DatabaseException("Oh!", "OutOfMemory!")});
        CompletionStage txStage = this.session.writeTransactionAsync((AsyncTransactionWork)work);
        DatabaseException e = (DatabaseException)Assertions.assertThrows(DatabaseException.class, () -> TestUtil.await(txStage));
        Assertions.assertEquals((int)1, (int)e.getSuppressed().length);
        MatcherAssert.assertThat((Object)e.getSuppressed()[0], (Matcher)org.hamcrest.Matchers.instanceOf(TransientException.class));
        Assertions.assertEquals((int)2, (int)work.invocationCount());
        Assertions.assertEquals((long)0L, (long)this.countNodesByLabel("Person"));
    }

    @Test
    void shouldPeekRecordFromCursor() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND [1, 2, 42] AS x RETURN x"));
        Assertions.assertEquals((int)1, (int)((Record)TestUtil.await(cursor.peekAsync())).get(0).asInt());
        Assertions.assertEquals((int)1, (int)((Record)TestUtil.await(cursor.peekAsync())).get(0).asInt());
        Assertions.assertEquals((int)1, (int)((Record)TestUtil.await(cursor.peekAsync())).get(0).asInt());
        Assertions.assertEquals((int)1, (int)((Record)TestUtil.await(cursor.nextAsync())).get(0).asInt());
        Assertions.assertEquals((int)2, (int)((Record)TestUtil.await(cursor.peekAsync())).get(0).asInt());
        Assertions.assertEquals((int)2, (int)((Record)TestUtil.await(cursor.peekAsync())).get(0).asInt());
        Assertions.assertEquals((int)2, (int)((Record)TestUtil.await(cursor.nextAsync())).get(0).asInt());
        Assertions.assertEquals((int)42, (int)((Record)TestUtil.await(cursor.nextAsync())).get(0).asInt());
        Assertions.assertNull(TestUtil.await(cursor.peekAsync()));
        Assertions.assertNull(TestUtil.await(cursor.nextAsync()));
    }

    @Test
    void shouldForEachWithEmptyCursor() {
        this.testForEach("CREATE ()", 0);
    }

    @Test
    void shouldForEachWithNonEmptyCursor() {
        this.testForEach("UNWIND range(1, 100000) AS x RETURN x", 100000);
    }

    @Test
    void shouldFailForEachWhenActionFails() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("RETURN 42"));
        IOException error = new IOException("Hi");
        IOException e = (IOException)Assertions.assertThrows(IOException.class, () -> TestUtil.await(cursor.forEachAsync(record -> {
            throw new CompletionException(error);
        })));
        Assertions.assertEquals((Object)error, (Object)e);
    }

    @Test
    void shouldConvertToListWithEmptyCursor() {
        this.testList("MATCH (n:NoSuchLabel) RETURN n", Collections.emptyList());
    }

    @Test
    void shouldConvertToListWithNonEmptyCursor() {
        this.testList("UNWIND range(1, 100, 10) AS x RETURN x", Arrays.asList(1L, 11L, 21L, 31L, 41L, 51L, 61L, 71L, 81L, 91L));
    }

    @Test
    void shouldConvertToTransformedListWithEmptyCursor() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("CREATE ()"));
        List strings = (List)TestUtil.await(cursor.listAsync(record -> "Hi!"));
        Assertions.assertEquals((int)0, (int)strings.size());
    }

    @Test
    void shouldConvertToTransformedListWithNonEmptyCursor() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND [1,2,3] AS x RETURN x"));
        List ints = (List)TestUtil.await(cursor.listAsync(record -> record.get(0).asInt() + 1));
        Assertions.assertEquals(Arrays.asList(2, 3, 4), (Object)ints);
    }

    @Test
    void shouldFailWhenListTransformationFunctionFails() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("RETURN 42"));
        RuntimeException error = new RuntimeException("Hi!");
        RuntimeException e = (RuntimeException)Assertions.assertThrows(RuntimeException.class, () -> TestUtil.await(cursor.listAsync(record -> {
            throw error;
        })));
        Assertions.assertEquals((Object)error, (Object)e);
    }

    @Test
    void shouldFailSingleWithEmptyCursor() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("CREATE ()"));
        NoSuchRecordException e = (NoSuchRecordException)Assertions.assertThrows(NoSuchRecordException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.containsString((String)"result is empty"));
    }

    @Test
    void shouldFailSingleWithMultiRecordCursor() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND [1, 2, 3] AS x RETURN x"));
        NoSuchRecordException e = (NoSuchRecordException)Assertions.assertThrows(NoSuchRecordException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.startsWith((String)"Expected a result with a single record"));
    }

    @Test
    void shouldReturnSingleWithSingleRecordCursor() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("RETURN 42"));
        Record record = (Record)TestUtil.await(cursor.singleAsync());
        Assertions.assertEquals((int)42, (int)record.get(0).asInt());
    }

    @Test
    void shouldPropagateFailureFromFirstRecordInSingleAsync() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND [0] AS x RETURN 10 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.containsString((String)"/ by zero"));
    }

    @Test
    void shouldNotPropagateFailureFromSecondRecordInSingleAsync() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND [1, 0] AS x RETURN 10 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.containsString((String)"/ by zero"));
    }

    @Test
    void shouldConsumeEmptyCursor() {
        this.testConsume("CREATE ()");
    }

    @Test
    void shouldConsumeNonEmptyCursor() {
        this.testConsume("UNWIND [42, 42] AS x RETURN x");
    }

    @Test
    @DisabledOnNeo4jWith(value=Neo4jFeature.BOLT_V3)
    void shouldRunAfterBeginTxFailureOnBookmark() {
        Bookmark illegalBookmark = InternalBookmark.parse((String)"Illegal Bookmark");
        this.session = neo4j.driver().asyncSession(SessionConfig.builder().withBookmarks(new Bookmark[]{illegalBookmark}).build());
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(this.session.beginTransactionAsync()));
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("RETURN 'Hello!'"));
        Record record = (Record)TestUtil.await(cursor.singleAsync());
        Assertions.assertEquals((Object)"Hello!", (Object)record.get(0).asString());
    }

    @Test
    void shouldNotBeginTxAfterBeginTxFailureOnBookmark() {
        Bookmark illegalBookmark = InternalBookmark.parse((String)"Illegal Bookmark");
        this.session = neo4j.driver().asyncSession(SessionConfig.builder().withBookmarks(new Bookmark[]{illegalBookmark}).build());
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(this.session.beginTransactionAsync()));
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(this.session.beginTransactionAsync()));
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V3)
    void shouldNotRunAfterBeginTxFailureOnBookmark() {
        Bookmark illegalBookmark = InternalBookmark.parse((String)"Illegal Bookmark");
        this.session = neo4j.driver().asyncSession(SessionConfig.builder().withBookmarks(new Bookmark[]{illegalBookmark}).build());
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(this.session.beginTransactionAsync()));
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(this.session.runAsync("RETURN 'Hello!'")));
    }

    @Test
    void shouldExecuteReadTransactionUntilSuccessWhenWorkThrows() {
        final int maxFailures = 1;
        CompletionStage result = this.session.readTransactionAsync((AsyncTransactionWork)new AsyncTransactionWork<CompletionStage<Integer>>(){
            final AtomicInteger failures = new AtomicInteger();

            public CompletionStage<Integer> execute(AsyncTransaction tx) {
                if (this.failures.getAndIncrement() < maxFailures) {
                    throw new SessionExpiredException("Oh!");
                }
                return tx.runAsync("UNWIND range(1, 10) AS x RETURN count(x)").thenCompose(ResultCursor::singleAsync).thenApply(record -> record.get(0).asInt());
            }
        });
        Assertions.assertEquals((int)10, (int)((Integer)TestUtil.await(result)));
    }

    @Test
    void shouldExecuteWriteTransactionUntilSuccessWhenWorkThrows() {
        final int maxFailures = 2;
        CompletionStage result = this.session.writeTransactionAsync((AsyncTransactionWork)new AsyncTransactionWork<CompletionStage<Integer>>(){
            final AtomicInteger failures = new AtomicInteger();

            public CompletionStage<Integer> execute(AsyncTransaction tx) {
                if (this.failures.getAndIncrement() < maxFailures) {
                    throw new ServiceUnavailableException("Oh!");
                }
                return tx.runAsync("CREATE (n1:TestNode), (n2:TestNode) RETURN 2").thenCompose(ResultCursor::singleAsync).thenApply(record -> record.get(0).asInt());
            }
        });
        Assertions.assertEquals((int)2, (int)((Integer)TestUtil.await(result)));
        Assertions.assertEquals((long)2L, (long)this.countNodesByLabel("TestNode"));
    }

    @Test
    void shouldExecuteReadTransactionUntilSuccessWhenWorkFails() {
        final int maxFailures = 3;
        CompletionStage result = this.session.readTransactionAsync((AsyncTransactionWork)new AsyncTransactionWork<CompletionStage<Integer>>(){
            final AtomicInteger failures = new AtomicInteger();

            public CompletionStage<Integer> execute(AsyncTransaction tx) {
                return tx.runAsync("RETURN 42").thenCompose(ResultCursor::singleAsync).thenApply(record -> record.get(0).asInt()).thenCompose(result -> {
                    if (this.failures.getAndIncrement() < maxFailures) {
                        return Futures.failedFuture((Throwable)new TransientException("A", "B"));
                    }
                    return CompletableFuture.completedFuture(result);
                });
            }
        });
        Assertions.assertEquals((int)42, (int)((Integer)TestUtil.await(result)));
    }

    @Test
    void shouldExecuteWriteTransactionUntilSuccessWhenWorkFails() {
        final int maxFailures = 2;
        CompletionStage result = this.session.writeTransactionAsync((AsyncTransactionWork)new AsyncTransactionWork<CompletionStage<String>>(){
            final AtomicInteger failures = new AtomicInteger();

            public CompletionStage<String> execute(AsyncTransaction tx) {
                return tx.runAsync("CREATE (:MyNode) RETURN 'Hello'").thenCompose(ResultCursor::singleAsync).thenApply(record -> record.get(0).asString()).thenCompose(result -> {
                    if (this.failures.getAndIncrement() < maxFailures) {
                        return Futures.failedFuture((Throwable)new ServiceUnavailableException("Hi"));
                    }
                    return CompletableFuture.completedFuture(result);
                });
            }
        });
        Assertions.assertEquals((Object)"Hello", TestUtil.await(result));
        Assertions.assertEquals((long)1L, (long)this.countNodesByLabel("MyNode"));
    }

    @Test
    void shouldNotPropagateRunFailureWhenClosed() {
        this.session.runAsync("RETURN 10 / 0");
        TestUtil.await(this.session.closeAsync());
    }

    @Test
    void shouldPropagateRunFailureImmediately() {
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(this.session.runAsync("RETURN 10 / 0")));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.containsString((String)"/ by zero"));
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldNotPropagateFailureWhenStreamingIsCancelled() {
        this.session.runAsync("UNWIND range(20000, 0, -1) AS x RETURN 10 / x");
        TestUtil.await(this.session.closeAsync());
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldNotPropagateBlockedPullAllFailureWhenClosed() {
        TestUtil.await(this.session.runAsync("UNWIND range(20000, 0, -1) AS x RETURN 10 / x"));
        TestUtil.await(this.session.closeAsync());
    }

    @Test
    void shouldCloseCleanlyWhenRunErrorConsumed() {
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(this.session.runAsync("SomeWrongQuery")));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.startsWith((String)"Invalid input"));
        Assertions.assertNull(TestUtil.await(this.session.closeAsync()));
    }

    @Test
    void shouldCloseCleanlyWhenPullAllErrorConsumed() {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("UNWIND range(10, 0, -1) AS x RETURN 1 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(cursor.consumeAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.containsString((String)"/ by zero"));
        Assertions.assertNull(TestUtil.await(this.session.closeAsync()));
    }

    @Test
    void shouldNotPropagateFailureInCloseFromPreviousRun() {
        this.session.runAsync("CREATE ()");
        this.session.runAsync("CREATE ()");
        this.session.runAsync("CREATE ()");
        this.session.runAsync("RETURN invalid");
        TestUtil.await(this.session.closeAsync());
    }

    @Test
    void shouldCloseCleanlyAfterFailure() {
        CompletionStage runWithOpenTx = this.session.beginTransactionAsync().thenCompose(tx -> this.session.runAsync("RETURN 1"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(runWithOpenTx));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)org.hamcrest.Matchers.startsWith((String)"Queries cannot be run directly on a session with an open transaction"));
        TestUtil.await(this.session.closeAsync());
    }

    @Test
    void shouldPropagateFailureFromFirstIllegalQuery() {
        CompletionStage allQueries = this.session.runAsync("CREATE (:Node1)").thenCompose(ignore -> this.session.runAsync("CREATE (:Node2)")).thenCompose(ignore -> this.session.runAsync("RETURN invalid")).thenCompose(ignore -> this.session.runAsync("CREATE (:Node3)"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(allQueries));
        MatcherAssert.assertThat((Object)((Object)e), (Matcher)org.hamcrest.Matchers.is(Matchers.syntaxError("Variable `invalid` not defined")));
        Assertions.assertEquals((long)1L, (long)this.countNodesByLabel("Node1"));
        Assertions.assertEquals((long)1L, (long)this.countNodesByLabel("Node2"));
        Assertions.assertEquals((long)0L, (long)this.countNodesByLabel("Node3"));
    }

    @Test
    void shouldAllowReturningNullFromAsyncTransactionFunction() {
        CompletionStage readResult = this.session.readTransactionAsync(tx -> null);
        Assertions.assertNull(TestUtil.await(readResult));
        CompletionStage writeResult = this.session.writeTransactionAsync(tx -> null);
        Assertions.assertNull(TestUtil.await(writeResult));
    }

    private Future<List<CompletionStage<Record>>> runNestedQueries(ResultCursor inputCursor) {
        CompletableFuture<List<CompletionStage<Record>>> resultFuture = new CompletableFuture<List<CompletionStage<Record>>>();
        this.runNestedQueries(inputCursor, new ArrayList<CompletionStage<Record>>(), resultFuture);
        return resultFuture;
    }

    private void runNestedQueries(ResultCursor inputCursor, List<CompletionStage<Record>> stages, CompletableFuture<List<CompletionStage<Record>>> resultFuture) {
        CompletionStage recordResponse = inputCursor.nextAsync();
        stages.add(recordResponse);
        recordResponse.whenComplete((record, error) -> {
            if (error != null) {
                resultFuture.completeExceptionally((Throwable)error);
            } else if (record != null) {
                this.runNestedQuery(inputCursor, (Record)record, stages, resultFuture);
            } else {
                resultFuture.complete(stages);
            }
        });
    }

    private void runNestedQuery(ResultCursor inputCursor, Record record, List<CompletionStage<Record>> stages, CompletableFuture<List<CompletionStage<Record>>> resultFuture) {
        Node node = record.get(0).asNode();
        long id = node.get("id").asLong();
        long age = id * 10L;
        CompletionStage response = this.session.runAsync("MATCH (p:Person {id: $id}) SET p.age = $age RETURN p", Values.parameters((Object[])new Object[]{"id", id, "age", age}));
        response.whenComplete((cursor, error) -> {
            if (error != null) {
                resultFuture.completeExceptionally(Futures.completionExceptionCause((Throwable)error));
            } else {
                stages.add(cursor.nextAsync());
                this.runNestedQueries(inputCursor, stages, resultFuture);
            }
        });
    }

    private long countNodesByLabel(String label) {
        CompletionStage<Long> countStage = this.session.runAsync("MATCH (n:" + label + ") RETURN count(n)").thenCompose(ResultCursor::singleAsync).thenApply(record -> record.get(0).asLong());
        return (Long)TestUtil.await(countStage);
    }

    private void testForEach(String query, int expectedSeenRecords) {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync(query));
        AtomicInteger recordsSeen = new AtomicInteger();
        CompletionStage forEachDone = cursor.forEachAsync(record -> recordsSeen.incrementAndGet());
        ResultSummary summary = (ResultSummary)TestUtil.await(forEachDone);
        Assertions.assertNotNull((Object)summary);
        Assertions.assertEquals((Object)query, (Object)summary.query().text());
        Assertions.assertEquals(Collections.emptyMap(), (Object)summary.query().parameters().asMap());
        Assertions.assertEquals((int)expectedSeenRecords, (int)recordsSeen.get());
    }

    private <T> void testList(String query, List<T> expectedList) {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync(query));
        List records = (List)TestUtil.await(cursor.listAsync());
        ArrayList<Object> actualList = new ArrayList<Object>();
        for (Record record : records) {
            actualList.add(record.get(0).asObject());
        }
        Assertions.assertEquals(expectedList, actualList);
    }

    private void testConsume(String query) {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync(query));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertNotNull((Object)summary);
        Assertions.assertEquals((Object)query, (Object)summary.query().text());
        Assertions.assertEquals(Collections.emptyMap(), (Object)summary.query().parameters().asMap());
        Assertions.assertThrows(ResultConsumedException.class, () -> TestUtil.await(cursor.nextAsync()));
    }

    private static class InvocationTrackingWork
    implements AsyncTransactionWork<CompletionStage<Record>> {
        final String query;
        final AtomicInteger invocationCount;
        Iterator<RuntimeException> asyncFailures = Collections.emptyIterator();
        Iterator<RuntimeException> syncFailures = Collections.emptyIterator();

        InvocationTrackingWork(String query) {
            this.query = query;
            this.invocationCount = new AtomicInteger();
        }

        InvocationTrackingWork withAsyncFailures(RuntimeException ... failures) {
            this.asyncFailures = Arrays.asList(failures).iterator();
            return this;
        }

        InvocationTrackingWork withSyncFailures(RuntimeException ... failures) {
            this.syncFailures = Arrays.asList(failures).iterator();
            return this;
        }

        int invocationCount() {
            return this.invocationCount.get();
        }

        public CompletionStage<Record> execute(AsyncTransaction tx) {
            this.invocationCount.incrementAndGet();
            if (this.syncFailures.hasNext()) {
                throw this.syncFailures.next();
            }
            CompletableFuture<Record> resultFuture = new CompletableFuture<Record>();
            tx.runAsync(this.query).whenComplete((cursor, error) -> this.processQueryResult((ResultCursor)cursor, Futures.completionExceptionCause((Throwable)error), resultFuture));
            return resultFuture;
        }

        private void processQueryResult(ResultCursor cursor, Throwable error, CompletableFuture<Record> resultFuture) {
            if (error != null) {
                resultFuture.completeExceptionally(error);
                return;
            }
            cursor.nextAsync().whenComplete((record, fetchError) -> this.processFetchResult((Record)record, Futures.completionExceptionCause((Throwable)fetchError), resultFuture));
        }

        private void processFetchResult(Record record, Throwable error, CompletableFuture<Record> resultFuture) {
            if (error != null) {
                resultFuture.completeExceptionally(error);
                return;
            }
            if (record == null) {
                resultFuture.completeExceptionally((Throwable)((Object)new AssertionError((Object)"Record not available")));
                return;
            }
            if (this.asyncFailures.hasNext()) {
                resultFuture.completeExceptionally(this.asyncFailures.next());
            } else {
                resultFuture.complete(record);
            }
        }
    }
}

