/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.integration;

import java.net.URI;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.junit.MatcherAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.neo4j.driver.AccessMode;
import org.neo4j.driver.AuthToken;
import org.neo4j.driver.Config;
import org.neo4j.driver.Driver;
import org.neo4j.driver.GraphDatabase;
import org.neo4j.driver.QueryRunner;
import org.neo4j.driver.Record;
import org.neo4j.driver.Result;
import org.neo4j.driver.Session;
import org.neo4j.driver.SessionConfig;
import org.neo4j.driver.Transaction;
import org.neo4j.driver.TransactionWork;
import org.neo4j.driver.Values;
import org.neo4j.driver.exceptions.AuthenticationException;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.ResultConsumedException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.exceptions.TransientException;
import org.neo4j.driver.internal.cluster.RoutingSettings;
import org.neo4j.driver.internal.logging.DevNullLogging;
import org.neo4j.driver.internal.retry.RetrySettings;
import org.neo4j.driver.internal.security.SecurityPlanImpl;
import org.neo4j.driver.internal.util.BookmarkUtil;
import org.neo4j.driver.internal.util.DisabledOnNeo4jWith;
import org.neo4j.driver.internal.util.DriverFactoryWithFixedRetryLogic;
import org.neo4j.driver.internal.util.EnabledOnNeo4jWith;
import org.neo4j.driver.internal.util.Matchers;
import org.neo4j.driver.internal.util.Neo4jFeature;
import org.neo4j.driver.reactive.RxResult;
import org.neo4j.driver.reactive.RxSession;
import org.neo4j.driver.summary.QueryType;
import org.neo4j.driver.summary.ResultSummary;
import org.neo4j.driver.util.DaemonThreadFactory;
import org.neo4j.driver.util.DatabaseExtension;
import org.neo4j.driver.util.Neo4jRunner;
import org.neo4j.driver.util.ParallelizableIT;
import org.neo4j.driver.util.TestUtil;
import org.reactivestreams.Publisher;
import reactor.test.StepVerifier;

@ParallelizableIT
class SessionIT {
    @RegisterExtension
    static final DatabaseExtension neo4j = new DatabaseExtension();
    private Driver driver;
    private ExecutorService executor;

    SessionIT() {
    }

    @AfterEach
    void tearDown() {
        if (this.driver != null) {
            this.driver.close();
        }
        if (this.executor != null) {
            this.executor.shutdownNow();
        }
    }

    @Test
    void shouldKnowSessionIsClosed() {
        Session session = neo4j.driver().session();
        session.close();
        Assertions.assertFalse((boolean)session.isOpen());
    }

    @Test
    void shouldHandleNullConfig() {
        this.driver = GraphDatabase.driver((URI)neo4j.uri(), (AuthToken)neo4j.authToken(), null);
        Session session = this.driver.session();
        session.close();
        Assertions.assertFalse((boolean)session.isOpen());
    }

    @Test
    void shouldHandleNullAuthToken() {
        AuthToken token = null;
        Assertions.assertThrows(AuthenticationException.class, () -> GraphDatabase.driver((URI)neo4j.uri(), (AuthToken)token).verifyConnectivity());
    }

    @Test
    void executeReadTxInReadSession() {
        this.testExecuteReadTx(AccessMode.READ);
    }

    @Test
    void executeReadTxInWriteSession() {
        this.testExecuteReadTx(AccessMode.WRITE);
    }

    @Test
    void executeWriteTxInReadSession() {
        this.testExecuteWriteTx(AccessMode.READ);
    }

    @Test
    void executeWriteTxInWriteSession() {
        this.testExecuteWriteTx(AccessMode.WRITE);
    }

    @Test
    void rollsBackWriteTxInReadSessionWhenFunctionThrows() {
        this.testTxRollbackWhenFunctionThrows(AccessMode.READ);
    }

    @Test
    void rollsBackWriteTxInWriteSessionWhenFunctionThrows() {
        this.testTxRollbackWhenFunctionThrows(AccessMode.WRITE);
    }

    @Test
    void readTxRetriedUntilSuccess() {
        int failures = 6;
        int retries = failures + 1;
        try (Driver driver = this.newDriverWithFixedRetries(retries);){
            try (Session session = driver.session();){
                session.run("CREATE (:Person {name: 'Bruce Banner'})");
            }
            ThrowingWork work = SessionIT.newThrowingWorkSpy("MATCH (n) RETURN n.name", failures);
            try (Session session = driver.session();){
                Record record = (Record)session.readTransaction((TransactionWork)work);
                Assertions.assertEquals((Object)"Bruce Banner", (Object)record.get(0).asString());
            }
            ((ThrowingWork)Mockito.verify((Object)work, (VerificationMode)Mockito.times((int)retries))).execute((Transaction)ArgumentMatchers.any(Transaction.class));
        }
    }

    @Test
    void writeTxRetriedUntilSuccess() {
        int failures = 4;
        int retries = failures + 1;
        try (Driver driver = this.newDriverWithFixedRetries(retries);){
            Record record;
            ThrowingWork work = SessionIT.newThrowingWorkSpy("CREATE (p:Person {name: 'Hulk'}) RETURN p", failures);
            try (Session session = driver.session();){
                record = (Record)session.writeTransaction((TransactionWork)work);
                Assertions.assertEquals((Object)"Hulk", (Object)record.get(0).asNode().get("name").asString());
            }
            session = driver.session();
            try {
                record = session.run("MATCH (p: Person {name: 'Hulk'}) RETURN count(p)").single();
                Assertions.assertEquals((int)1, (int)record.get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
            ((ThrowingWork)Mockito.verify((Object)work, (VerificationMode)Mockito.times((int)retries))).execute((Transaction)ArgumentMatchers.any(Transaction.class));
        }
    }

    @Test
    void readTxRetriedUntilFailure() {
        int failures = 3;
        int retries = failures - 1;
        try (Driver driver = this.newDriverWithFixedRetries(retries);){
            ThrowingWork work = SessionIT.newThrowingWorkSpy("MATCH (n) RETURN n.name", failures);
            try (Session session = driver.session();){
                Assertions.assertThrows(ServiceUnavailableException.class, () -> session.readTransaction((TransactionWork)work));
            }
            ((ThrowingWork)Mockito.verify((Object)work, (VerificationMode)Mockito.times((int)failures))).execute((Transaction)ArgumentMatchers.any(Transaction.class));
        }
    }

    @Test
    void writeTxRetriedUntilFailure() {
        int failures = 8;
        int retries = failures - 1;
        try (Driver driver = this.newDriverWithFixedRetries(retries);){
            ThrowingWork work = SessionIT.newThrowingWorkSpy("CREATE (:Person {name: 'Ronan'})", failures);
            try (Session session = driver.session();){
                Assertions.assertThrows(ServiceUnavailableException.class, () -> session.writeTransaction((TransactionWork)work));
            }
            session = driver.session();
            try {
                Result result = session.run("MATCH (p:Person {name: 'Ronan'}) RETURN count(p)");
                Assertions.assertEquals((int)0, (int)result.single().get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
            ((ThrowingWork)Mockito.verify((Object)work, (VerificationMode)Mockito.times((int)failures))).execute((Transaction)ArgumentMatchers.any(Transaction.class));
        }
    }

    @Test
    void writeTxRetryErrorsAreCollected() {
        try (Driver driver = this.newDriverWithLimitedRetries(5, TimeUnit.SECONDS);){
            int suppressedErrors;
            ThrowingWork work = SessionIT.newThrowingWorkSpy("CREATE (:Person {name: 'Ronan'})", Integer.MAX_VALUE);
            try (Session session = driver.session();){
                ServiceUnavailableException e = (ServiceUnavailableException)Assertions.assertThrows(ServiceUnavailableException.class, () -> session.writeTransaction((TransactionWork)work));
                MatcherAssert.assertThat((Object)e.getSuppressed(), (Matcher)org.hamcrest.Matchers.not((Matcher)org.hamcrest.Matchers.emptyArray()));
                suppressedErrors = e.getSuppressed().length;
            }
            session = driver.session();
            try {
                Result result = session.run("MATCH (p:Person {name: 'Ronan'}) RETURN count(p)");
                Assertions.assertEquals((int)0, (int)result.single().get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
            ((ThrowingWork)Mockito.verify((Object)work, (VerificationMode)Mockito.times((int)(suppressedErrors + 1)))).execute((Transaction)ArgumentMatchers.any(Transaction.class));
        }
    }

    @Test
    void readTxRetryErrorsAreCollected() {
        try (Driver driver = this.newDriverWithLimitedRetries(4, TimeUnit.SECONDS);){
            int suppressedErrors;
            ThrowingWork work = SessionIT.newThrowingWorkSpy("MATCH (n) RETURN n.name", Integer.MAX_VALUE);
            try (Session session = driver.session();){
                ServiceUnavailableException e = (ServiceUnavailableException)Assertions.assertThrows(ServiceUnavailableException.class, () -> session.readTransaction((TransactionWork)work));
                MatcherAssert.assertThat((Object)e.getSuppressed(), (Matcher)org.hamcrest.Matchers.not((Matcher)org.hamcrest.Matchers.emptyArray()));
                suppressedErrors = e.getSuppressed().length;
            }
            ((ThrowingWork)Mockito.verify((Object)work, (VerificationMode)Mockito.times((int)(suppressedErrors + 1)))).execute((Transaction)ArgumentMatchers.any(Transaction.class));
        }
    }

    @Test
    void readTxCommittedWithoutTxSuccess() {
        try (Driver driver = this.newDriverWithoutRetries();
             Session session = driver.session();){
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
            long answer = (Long)session.readTransaction(tx -> tx.run("RETURN 42").single().get(0).asLong());
            Assertions.assertEquals((long)42L, (long)answer);
            BookmarkUtil.assertBookmarkContainsSingleValue(session.lastBookmark());
        }
    }

    @Test
    void writeTxCommittedWithoutTxSuccess() {
        try (Driver driver = this.newDriverWithoutRetries();){
            try (Session session = driver.session();){
                long answer = (Long)session.writeTransaction(tx -> tx.run("CREATE (:Person {name: 'Thor Odinson'}) RETURN 42").single().get(0).asLong());
                Assertions.assertEquals((long)42L, (long)answer);
            }
            session = driver.session();
            try {
                Result result = session.run("MATCH (p:Person {name: 'Thor Odinson'}) RETURN count(p)");
                Assertions.assertEquals((int)1, (int)result.single().get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
        }
    }

    @Test
    void readTxRolledBackWithTxFailure() {
        try (Driver driver = this.newDriverWithoutRetries();
             Session session = driver.session();){
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
            long answer = (Long)session.readTransaction(tx -> {
                Result result = tx.run("RETURN 42");
                long single = result.single().get(0).asLong();
                tx.rollback();
                return single;
            });
            Assertions.assertEquals((long)42L, (long)answer);
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
        }
    }

    @Test
    void writeTxRolledBackWithTxFailure() {
        try (Driver driver = this.newDriverWithoutRetries();){
            try (Session session = driver.session();){
                int answer = (Integer)session.writeTransaction(tx -> {
                    tx.run("CREATE (:Person {name: 'Natasha Romanoff'})");
                    tx.rollback();
                    return 42;
                });
                Assertions.assertEquals((int)42, (int)answer);
            }
            session = driver.session();
            try {
                Result result = session.run("MATCH (p:Person {name: 'Natasha Romanoff'}) RETURN count(p)");
                Assertions.assertEquals((int)0, (int)result.single().get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
        }
    }

    @Test
    void readTxRolledBackWhenExceptionIsThrown() {
        try (Driver driver = this.newDriverWithoutRetries();
             Session session = driver.session();){
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
            Assertions.assertThrows(IllegalStateException.class, () -> session.readTransaction(tx -> {
                Result result = tx.run("RETURN 42");
                if (result.single().get(0).asLong() == 42L) {
                    throw new IllegalStateException();
                }
                return 1L;
            }));
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
        }
    }

    @Test
    void writeTxRolledBackWhenExceptionIsThrown() {
        try (Driver driver = this.newDriverWithoutRetries();){
            try (Session session = driver.session();){
                Assertions.assertThrows(IllegalStateException.class, () -> session.writeTransaction(tx -> {
                    tx.run("CREATE (:Person {name: 'Loki Odinson'})");
                    throw new IllegalStateException();
                }));
            }
            session = driver.session();
            try {
                Result result = session.run("MATCH (p:Person {name: 'Natasha Romanoff'}) RETURN count(p)");
                Assertions.assertEquals((int)0, (int)result.single().get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
        }
    }

    @Test
    void readTxRolledBackWhenMarkedBothSuccessAndFailure() {
        try (Driver driver = this.newDriverWithoutRetries();
             Session session = driver.session();){
            ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.readTransaction(tx -> {
                Result result = tx.run("RETURN 42");
                tx.commit();
                tx.rollback();
                return result.single().get(0).asLong();
            }));
            MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.startsWith((String)"Can't rollback, transaction has been committed"));
        }
    }

    @Test
    void writeTxFailWhenBothCommitAndRollback() {
        try (Driver driver = this.newDriverWithoutRetries();
             Session session = driver.session();){
            ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.writeTransaction(tx -> {
                tx.run("CREATE (:Person {name: 'Natasha Romanoff'})");
                tx.commit();
                tx.rollback();
                return 42;
            }));
            MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.startsWith((String)"Can't rollback, transaction has been committed"));
        }
    }

    @Test
    void readTxCommittedWhenCommitAndThrowsException() {
        try (Driver driver = this.newDriverWithoutRetries();
             Session session = driver.session();){
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
            Assertions.assertThrows(IllegalStateException.class, () -> session.readTransaction(tx -> {
                tx.run("RETURN 42");
                tx.commit();
                throw new IllegalStateException();
            }));
            BookmarkUtil.assertBookmarkIsNotEmpty(session.lastBookmark());
        }
    }

    @Test
    void writeTxCommittedWhenCommitAndThrowsException() {
        try (Driver driver = this.newDriverWithoutRetries();){
            try (Session session = driver.session();){
                Assertions.assertThrows(IllegalStateException.class, () -> session.writeTransaction(tx -> {
                    tx.run("CREATE (:Person {name: 'Natasha Romanoff'})");
                    tx.commit();
                    throw new IllegalStateException();
                }));
            }
            session = driver.session();
            try {
                Result result = session.run("MATCH (p:Person {name: 'Natasha Romanoff'}) RETURN count(p)");
                Assertions.assertEquals((int)1, (int)result.single().get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
        }
    }

    @Test
    void readRolledBackWhenRollbackAndThrowsException() {
        try (Driver driver = this.newDriverWithoutRetries();
             Session session = driver.session();){
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
            Assertions.assertThrows(IllegalStateException.class, () -> session.readTransaction(tx -> {
                tx.run("RETURN 42");
                tx.rollback();
                throw new IllegalStateException();
            }));
            BookmarkUtil.assertBookmarkIsEmpty(session.lastBookmark());
        }
    }

    @Test
    void writeTxRolledBackWhenRollbackAndThrowsException() {
        try (Driver driver = this.newDriverWithoutRetries();){
            try (Session session = driver.session();){
                Assertions.assertThrows(IllegalStateException.class, () -> session.writeTransaction(tx -> {
                    tx.run("CREATE (:Person {name: 'Natasha Romanoff'})");
                    tx.rollback();
                    throw new IllegalStateException();
                }));
            }
            session = driver.session();
            try {
                Result result = session.run("MATCH (p:Person {name: 'Natasha Romanoff'}) RETURN count(p)");
                Assertions.assertEquals((int)0, (int)result.single().get(0).asInt());
            }
            finally {
                if (session != null) {
                    session.close();
                }
            }
        }
    }

    @Test
    void transactionRunShouldFailOnDeadlocks() throws Exception {
        int nodeId1 = 42;
        int nodeId2 = 4242;
        boolean newNodeId1 = true;
        int newNodeId2 = 2;
        this.createNodeWithId(42);
        this.createNodeWithId(4242);
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        Future<Void> result1 = this.executeInDifferentThread(() -> {
            try (Session session = neo4j.driver().session();
                 Transaction tx = session.beginTransaction();){
                SessionIT.updateNodeId((QueryRunner)tx, 42, 1).consume();
                latch1.await();
                latch2.countDown();
                SessionIT.updateNodeId((QueryRunner)tx, 4242, 1).consume();
                tx.commit();
            }
            return null;
        });
        Future<Void> result2 = this.executeInDifferentThread(() -> {
            try (Session session = neo4j.driver().session();
                 Transaction tx = session.beginTransaction();){
                SessionIT.updateNodeId((QueryRunner)tx, 4242, 2).consume();
                latch1.countDown();
                latch2.await();
                SessionIT.updateNodeId((QueryRunner)tx, 42, 2).consume();
                tx.commit();
            }
            return null;
        });
        boolean firstResultFailed = SessionIT.assertOneOfTwoFuturesFailWithDeadlock(result1, result2);
        if (firstResultFailed) {
            Assertions.assertEquals((int)0, (int)this.countNodesWithId(1));
            Assertions.assertEquals((int)2, (int)this.countNodesWithId(2));
        } else {
            Assertions.assertEquals((int)2, (int)this.countNodesWithId(1));
            Assertions.assertEquals((int)0, (int)this.countNodesWithId(2));
        }
    }

    @Test
    void writeTransactionFunctionShouldRetryDeadlocks() throws Exception {
        int nodeId1 = 42;
        int nodeId2 = 4242;
        int nodeId3 = 424242;
        boolean newNodeId1 = true;
        int newNodeId2 = 2;
        this.createNodeWithId(42);
        this.createNodeWithId(4242);
        CountDownLatch latch1 = new CountDownLatch(1);
        CountDownLatch latch2 = new CountDownLatch(1);
        Future<Void> result1 = this.executeInDifferentThread(() -> {
            try (Session session = neo4j.driver().session();
                 Transaction tx = session.beginTransaction();){
                SessionIT.updateNodeId((QueryRunner)tx, 42, 1).consume();
                latch1.await();
                latch2.countDown();
                SessionIT.updateNodeId((QueryRunner)tx, 4242, 1).consume();
                tx.commit();
            }
            return null;
        });
        Future<Void> result2 = this.executeInDifferentThread(() -> {
            try (Session session = neo4j.driver().session();){
                session.writeTransaction(tx -> {
                    SessionIT.updateNodeId((QueryRunner)tx, 4242, 2).consume();
                    latch1.countDown();
                    SessionIT.await(latch2);
                    SessionIT.updateNodeId((QueryRunner)tx, 42, 2).consume();
                    this.createNodeWithId(424242);
                    return null;
                });
            }
            return null;
        });
        boolean firstResultFailed = false;
        try {
            Assertions.assertNull((Object)result1.get(20L, TimeUnit.SECONDS));
        }
        catch (ExecutionException e) {
            firstResultFailed = true;
        }
        Assertions.assertNull((Object)result2.get(20L, TimeUnit.SECONDS));
        if (firstResultFailed) {
            Assertions.assertEquals((int)0, (int)this.countNodesWithId(1));
            Assertions.assertEquals((int)2, (int)this.countNodesWithId(2));
        } else {
            Assertions.assertEquals((int)2, (int)this.countNodesWithId(1));
            Assertions.assertEquals((int)0, (int)this.countNodesWithId(2));
        }
        Assertions.assertEquals((int)1, (int)this.countNodesWithId(424242));
    }

    @Test
    void shouldExecuteTransactionWorkInCallerThread() {
        final int maxFailures = 3;
        final Thread callerThread = Thread.currentThread();
        try (Session session = neo4j.driver().session();){
            String result = (String)session.readTransaction((TransactionWork)new TransactionWork<String>(){
                int failures;

                public String execute(Transaction tx) {
                    Assertions.assertSame((Object)callerThread, (Object)Thread.currentThread());
                    if (this.failures++ < maxFailures) {
                        throw new ServiceUnavailableException("Oh no");
                    }
                    return "Hello";
                }
            });
            Assertions.assertEquals((Object)"Hello", (Object)result);
        }
    }

    @Test
    void shouldThrowRunFailureImmediatelyAndCloseSuccessfully() {
        try (Session session = neo4j.driver().session();){
            ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.run("RETURN 10 / 0"));
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"/ by zero"));
        }
    }

    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    @Test
    void shouldNotPropagateFailureWhenStreamingIsCancelled() {
        Session session = neo4j.driver().session();
        session.run("UNWIND range(20000, 0, -1) AS x RETURN 10 / x");
        session.close();
    }

    @Test
    void shouldNotBePossibleToConsumeResultAfterSessionIsClosed() {
        Result result;
        try (Session session = neo4j.driver().session();){
            result = session.run("UNWIND range(1, 20000) AS x RETURN x");
        }
        Assertions.assertThrows(ResultConsumedException.class, () -> result.list(record -> record.get(0).asInt()));
    }

    @Test
    void shouldThrowRunFailureImmediatelyAfterMultipleSuccessfulRunsAndCloseSuccessfully() {
        try (Session session = neo4j.driver().session();){
            session.run("CREATE ()");
            session.run("CREATE ()");
            ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.run("RETURN 10 / 0"));
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"/ by zero"));
        }
    }

    @Test
    void shouldThrowRunFailureImmediatelyAndAcceptSubsequentRun() {
        try (Session session = neo4j.driver().session();){
            session.run("CREATE ()");
            session.run("CREATE ()");
            ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.run("RETURN 10 / 0"));
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"/ by zero"));
            session.run("CREATE ()");
        }
    }

    @Test
    void shouldCloseCleanlyWhenRunErrorConsumed() {
        Session session = neo4j.driver().session();
        session.run("CREATE ()");
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.run("RETURN 10 / 0").consume());
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"/ by zero"));
        session.run("CREATE ()");
        session.close();
        Assertions.assertFalse((boolean)session.isOpen());
    }

    @Test
    void shouldConsumePreviousResultBeforeRunningNewQuery() {
        try (Session session = neo4j.driver().session();){
            session.run("UNWIND range(1000, 0, -1) AS x RETURN 42 / x");
            ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.run("RETURN 1"));
            MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"/ by zero"));
        }
    }

    @Test
    void shouldNotRetryOnConnectionAcquisitionTimeout() {
        int maxPoolSize = 3;
        Config config = Config.builder().withMaxConnectionPoolSize(maxPoolSize).withConnectionAcquisitionTimeout(0L, TimeUnit.SECONDS).withMaxTransactionRetryTime(42L, TimeUnit.DAYS).withEventLoopThreads(1).build();
        this.driver = GraphDatabase.driver((URI)neo4j.uri(), (AuthToken)neo4j.authToken(), (Config)config);
        for (int i = 0; i < maxPoolSize; ++i) {
            this.driver.session().beginTransaction();
        }
        AtomicInteger invocations = new AtomicInteger();
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> this.driver.session().writeTransaction(tx -> invocations.incrementAndGet()));
        MatcherAssert.assertThat((Object)e, (Matcher)org.hamcrest.Matchers.is(Matchers.connectionAcquisitionTimeoutError(0)));
        Assertions.assertEquals((int)0, (int)invocations.get());
    }

    @Test
    void shouldReportFailureInClose() {
        Session session = neo4j.driver().session();
        Result result = session.run("CYPHER runtime=interpreted UNWIND [2, 4, 8, 0] AS x RETURN 32 / x");
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> ((Session)session).close());
        MatcherAssert.assertThat((Object)e, (Matcher)org.hamcrest.Matchers.is(Matchers.arithmeticError()));
    }

    @Test
    void shouldNotAllowAccessingRecordsAfterSummary() {
        int recordCount = 10000;
        String query = "UNWIND range(1, " + recordCount + ") AS x RETURN x";
        try (Session session = neo4j.driver().session();){
            Result result = session.run(query);
            ResultSummary summary = result.consume();
            Assertions.assertEquals((Object)query, (Object)summary.query().text());
            Assertions.assertEquals((Object)QueryType.READ_ONLY, (Object)summary.queryType());
            Assertions.assertThrows(ResultConsumedException.class, () -> ((Result)result).list());
        }
    }

    @Test
    void shouldNotAllowAccessingRecordsAfterSessionClosed() {
        Result result;
        int recordCount = 11333;
        String query = "UNWIND range(1, " + recordCount + ") AS x RETURN 'Result-' + x";
        try (Session session = neo4j.driver().session();){
            result = session.run(query);
        }
        Assertions.assertThrows(ResultConsumedException.class, () -> ((Result)result).list());
    }

    @Test
    @DisabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldAllowToConsumeRecordsSlowlyAndCloseSession() throws InterruptedException {
        Session session = neo4j.driver().session();
        Result result = session.run("UNWIND range(10000, 0, -1) AS x RETURN 10 / x");
        for (int i = 0; i < 10; ++i) {
            Assertions.assertTrue((boolean)result.hasNext());
            Assertions.assertNotNull((Object)result.next());
            Thread.sleep(50L);
        }
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> ((Session)session).close());
        MatcherAssert.assertThat((Object)e, (Matcher)org.hamcrest.Matchers.is(Matchers.arithmeticError()));
    }

    @Test
    void shouldAllowToConsumeRecordsSlowlyAndRetrieveSummary() throws InterruptedException {
        try (Session session = neo4j.driver().session();){
            Result result = session.run("UNWIND range(8000, 1, -1) AS x RETURN 42 / x");
            for (int i = 0; i < 12; ++i) {
                Assertions.assertTrue((boolean)result.hasNext());
                Assertions.assertNotNull((Object)result.next());
                Thread.sleep(50L);
            }
            ResultSummary summary = result.consume();
            Assertions.assertNotNull((Object)summary);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void shouldBeResponsiveToThreadInterruptWhenWaitingForResult() {
        try (Session session1 = neo4j.driver().session();
             Session session2 = neo4j.driver().session();){
            session1.run("CREATE (:Person {name: 'Beta Ray Bill'})").consume();
            Transaction tx = session1.beginTransaction();
            tx.run("MATCH (n:Person {name: 'Beta Ray Bill'}) SET n.hammer = 'Mjolnir'").consume();
            TestUtil.interruptWhenInWaitingState(Thread.currentThread());
            try {
                ServiceUnavailableException e = (ServiceUnavailableException)Assertions.assertThrows(ServiceUnavailableException.class, () -> session2.run("MATCH (n:Person {name: 'Beta Ray Bill'}) SET n.hammer = 'Stormbreaker'").consume());
                MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"Connection to the database terminated"));
                MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"Thread interrupted"));
            }
            finally {
                Thread.interrupted();
            }
        }
    }

    @Test
    void shouldAllowLongRunningQueryWithConnectTimeout() throws Exception {
        int connectionTimeoutMs = 3000;
        Config config = Config.builder().withLogging(DevNullLogging.DEV_NULL_LOGGING).withConnectionTimeout((long)connectionTimeoutMs, TimeUnit.MILLISECONDS).build();
        try (Driver driver = GraphDatabase.driver((URI)neo4j.uri(), (AuthToken)neo4j.authToken(), (Config)config);){
            Session session1 = driver.session();
            Session session2 = driver.session();
            session1.run("CREATE (:Avenger {name: 'Hulk'})").consume();
            Transaction tx = session1.beginTransaction();
            tx.run("MATCH (a:Avenger {name: 'Hulk'}) SET a.power = 100 RETURN a").consume();
            CountDownLatch latch = new CountDownLatch(1);
            Future<Long> updateFuture = this.executeInDifferentThread(() -> {
                latch.countDown();
                return session2.run("MATCH (a:Avenger {name: 'Hulk'}) SET a.weight = 1000 RETURN a.power").single().get(0).asLong();
            });
            latch.await();
            Thread.sleep(connectionTimeoutMs + 1000);
            Assertions.assertFalse((boolean)updateFuture.isDone());
            tx.commit();
            long hulkPower = updateFuture.get(10L, TimeUnit.SECONDS);
            Assertions.assertEquals((long)100L, (long)hulkPower);
        }
    }

    @Test
    void shouldAllowReturningNullFromTransactionFunction() {
        try (Session session = neo4j.driver().session();){
            Assertions.assertNull((Object)session.readTransaction(tx -> null));
            Assertions.assertNull((Object)session.writeTransaction(tx -> null));
        }
    }

    @Test
    void shouldAllowIteratingOverEmptyResult() {
        try (Session session = neo4j.driver().session();){
            Result result = session.run("UNWIND [] AS x RETURN x");
            Assertions.assertFalse((boolean)result.hasNext());
            Assertions.assertThrows(NoSuchElementException.class, () -> ((Result)result).next());
        }
    }

    @Test
    void shouldAllowConsumingEmptyResult() {
        try (Session session = neo4j.driver().session();){
            Result result = session.run("UNWIND [] AS x RETURN x");
            ResultSummary summary = result.consume();
            Assertions.assertNotNull((Object)summary);
            Assertions.assertEquals((Object)QueryType.READ_ONLY, (Object)summary.queryType());
        }
    }

    @Test
    void shouldAllowListEmptyResult() {
        try (Session session = neo4j.driver().session();){
            Result result = session.run("UNWIND [] AS x RETURN x");
            Assertions.assertEquals(Collections.emptyList(), (Object)result.list());
        }
    }

    @Test
    void shouldReportFailureInSummary() {
        try (Session session = neo4j.driver().session();){
            String query = "UNWIND [1, 2, 3, 4, 0] AS x RETURN 10 / x";
            Result result = session.run(query);
            ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> ((Result)result).consume());
            MatcherAssert.assertThat((Object)e, (Matcher)org.hamcrest.Matchers.is(Matchers.arithmeticError()));
            ResultSummary summary = result.consume();
            Assertions.assertEquals((Object)query, (Object)summary.query().text());
        }
    }

    @Test
    void shouldNotAllowStartingMultipleTransactions() {
        try (Session session = neo4j.driver().session();){
            Transaction tx = session.beginTransaction();
            Assertions.assertNotNull((Object)tx);
            for (int i = 0; i < 3; ++i) {
                ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> ((Session)session).beginTransaction());
                MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)CoreMatchers.containsString((String)"You cannot begin a transaction on a session with an open transaction"));
            }
            tx.close();
            Assertions.assertNotNull((Object)session.beginTransaction());
        }
    }

    @Test
    void shouldCloseOpenTransactionWhenClosed() {
        try (Session session = neo4j.driver().session();){
            Transaction tx = session.beginTransaction();
            tx.run("CREATE (:Node {id: 123})");
            tx.run("CREATE (:Node {id: 456})");
            tx.commit();
        }
        Assertions.assertEquals((int)1, (int)this.countNodesWithId(123));
        Assertions.assertEquals((int)1, (int)this.countNodesWithId(456));
    }

    @Test
    void shouldRollbackOpenTransactionWhenClosed() {
        try (Session session = neo4j.driver().session();){
            Transaction tx = session.beginTransaction();
            tx.run("CREATE (:Node {id: 123})");
            tx.run("CREATE (:Node {id: 456})");
            tx.rollback();
        }
        Assertions.assertEquals((int)0, (int)this.countNodesWithId(123));
        Assertions.assertEquals((int)0, (int)this.countNodesWithId(456));
    }

    @Test
    void shouldSupportNestedQueries() {
        try (Session session = neo4j.driver().session();){
            session.run("UNWIND range(1, 100) AS x CREATE (:Property {id: x})").consume();
            session.run("UNWIND range(1, 10) AS x CREATE (:Resource {id: x})").consume();
            int seenProperties = 0;
            int seenResources = 0;
            Result properties = session.run("MATCH (p:Property) RETURN p");
            while (properties.hasNext()) {
                Assertions.assertNotNull((Object)properties.next());
                ++seenProperties;
                Result resources = session.run("MATCH (r:Resource) RETURN r");
                while (resources.hasNext()) {
                    Assertions.assertNotNull((Object)resources.next());
                    ++seenResources;
                }
            }
            Assertions.assertEquals((int)100, (int)seenProperties);
            Assertions.assertEquals((int)1000, (int)seenResources);
        }
    }

    @Test
    @DisabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldErrorWhenTryingToUseRxAPIWithoutBoltV4() throws Throwable {
        RxSession session = neo4j.driver().rxSession();
        RxResult result = session.run("RETURN 1");
        StepVerifier.create((Publisher)result.records()).expectErrorSatisfies(error -> {
            MatcherAssert.assertThat((Object)error, (Matcher)CoreMatchers.instanceOf(ClientException.class));
            MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.containsString((String)"Driver is connected to the database that does not support driver reactive API"));
        }).verify();
    }

    @Test
    @DisabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldErrorWhenTryingToUseDatabaseNameWithoutBoltV4() throws Throwable {
        Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"foo"));
        ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.run("RETURN 1"));
        MatcherAssert.assertThat((Object)error, (Matcher)CoreMatchers.instanceOf(ClientException.class));
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.containsString((String)"Database name parameter for selecting database is not supported"));
    }

    @Test
    @DisabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldErrorWhenTryingToUseDatabaseNameWithoutBoltV4UsingTx() throws Throwable {
        Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"foo"));
        ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> ((Session)session).beginTransaction());
        MatcherAssert.assertThat((Object)error, (Matcher)CoreMatchers.instanceOf(ClientException.class));
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.containsString((String)"Database name parameter for selecting database is not supported"));
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldAllowDatabaseName() throws Throwable {
        try (Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"neo4j"));){
            Result result = session.run("RETURN 1");
            MatcherAssert.assertThat((Object)result.single().get(0).asInt(), (Matcher)CoreMatchers.equalTo((Object)1));
        }
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldAllowDatabaseNameUsingTx() throws Throwable {
        try (Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"neo4j"));
             Transaction transaction = session.beginTransaction();){
            Result result = transaction.run("RETURN 1");
            MatcherAssert.assertThat((Object)result.single().get(0).asInt(), (Matcher)CoreMatchers.equalTo((Object)1));
        }
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldAllowDatabaseNameUsingTxWithRetries() throws Throwable {
        try (Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"neo4j"));){
            int num = (Integer)session.readTransaction(tx -> tx.run("RETURN 1").single().get(0).asInt());
            MatcherAssert.assertThat((Object)num, (Matcher)CoreMatchers.equalTo((Object)1));
        }
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldErrorDatabaseWhenDatabaseIsAbsent() throws Throwable {
        Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"foo"));
        ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> {
            Result result = session.run("RETURN 1");
            result.consume();
        });
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.containsString((String)"Database does not exist. Database name: 'foo'"));
        session.close();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldErrorDatabaseNameUsingTxWhenDatabaseIsAbsent() throws Throwable {
        Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"foo"));
        ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> {
            Transaction transaction = session.beginTransaction();
            Result result = transaction.run("RETURN 1");
            result.consume();
        });
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.containsString((String)"Database does not exist. Database name: 'foo'"));
        session.close();
    }

    @Test
    @EnabledOnNeo4jWith(value=Neo4jFeature.BOLT_V4)
    void shouldErrorDatabaseNameUsingTxWithRetriesWhenDatabaseIsAbsent() throws Throwable {
        Session session = neo4j.driver().session(SessionConfig.forDatabase((String)"foo"));
        ClientException error = (ClientException)Assertions.assertThrows(ClientException.class, () -> session.readTransaction(tx -> tx.run("RETURN 1").consume()));
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)CoreMatchers.containsString((String)"Database does not exist. Database name: 'foo'"));
        session.close();
    }

    private void testExecuteReadTx(AccessMode sessionMode) {
        Driver driver = neo4j.driver();
        try (Session session = driver.session();){
            session.run("CREATE (:Person {name: 'Tony Stark'})");
            session.run("CREATE (:Person {name: 'Steve Rogers'})");
        }
        session = driver.session(SessionConfig.builder().withDefaultAccessMode(sessionMode).build());
        try {
            Set names = (Set)session.readTransaction(tx -> {
                List records = tx.run("MATCH (p:Person) RETURN p.name AS name").list();
                HashSet<String> names1 = new HashSet<String>(records.size());
                for (Record record : records) {
                    names1.add(record.get("name").asString());
                }
                return names1;
            });
            MatcherAssert.assertThat((Object)names, (Matcher)org.hamcrest.Matchers.containsInAnyOrder((Object[])new String[]{"Tony Stark", "Steve Rogers"}));
        }
        finally {
            if (session != null) {
                session.close();
            }
        }
    }

    private void testExecuteWriteTx(AccessMode sessionMode) {
        Driver driver = neo4j.driver();
        try (Session session = driver.session(SessionConfig.builder().withDefaultAccessMode(sessionMode).build());){
            String material = (String)session.writeTransaction(tx -> {
                Result result = tx.run("CREATE (s:Shield {material: 'Vibranium'}) RETURN s");
                Record record = result.single();
                tx.commit();
                return record.get(0).asNode().get("material").asString();
            });
            Assertions.assertEquals((Object)"Vibranium", (Object)material);
        }
        session = driver.session();
        try {
            Record record = session.run("MATCH (s:Shield) RETURN s.material").single();
            Assertions.assertEquals((Object)"Vibranium", (Object)record.get(0).asString());
        }
        finally {
            if (session != null) {
                session.close();
            }
        }
    }

    private void testTxRollbackWhenFunctionThrows(AccessMode sessionMode) {
        Driver driver = neo4j.driver();
        try (Session session = driver.session(SessionConfig.builder().withDefaultAccessMode(sessionMode).build());){
            Assertions.assertThrows(ClientException.class, () -> session.writeTransaction(tx -> {
                tx.run("CREATE (:Person {name: 'Thanos'})");
                tx.run("UNWIND range(0, 1) AS i RETURN 10/i");
                tx.commit();
                return null;
            }));
        }
        session = driver.session();
        try {
            Record record = session.run("MATCH (p:Person {name: 'Thanos'}) RETURN count(p)").single();
            Assertions.assertEquals((int)0, (int)record.get(0).asInt());
        }
        finally {
            if (session != null) {
                session.close();
            }
        }
    }

    private Driver newDriverWithoutRetries() {
        return this.newDriverWithFixedRetries(0);
    }

    private Driver newDriverWithFixedRetries(int maxRetriesCount) {
        DriverFactoryWithFixedRetryLogic driverFactory = new DriverFactoryWithFixedRetryLogic(maxRetriesCount);
        AuthToken auth = Neo4jRunner.DEFAULT_AUTH_TOKEN;
        return driverFactory.newInstance(neo4j.uri(), auth, RoutingSettings.DEFAULT, RetrySettings.DEFAULT, SessionIT.noLoggingConfig(), SecurityPlanImpl.insecure());
    }

    private Driver newDriverWithLimitedRetries(int maxTxRetryTime, TimeUnit unit) {
        Config config = Config.builder().withLogging(DevNullLogging.DEV_NULL_LOGGING).withMaxTransactionRetryTime((long)maxTxRetryTime, unit).build();
        return GraphDatabase.driver((URI)neo4j.uri(), (AuthToken)neo4j.authToken(), (Config)config);
    }

    private static Config noLoggingConfig() {
        return Config.builder().withLogging(DevNullLogging.DEV_NULL_LOGGING).build();
    }

    private static ThrowingWork newThrowingWorkSpy(String query, int failures) {
        return (ThrowingWork)Mockito.spy((Object)new ThrowingWork(query, failures));
    }

    private int countNodesWithId(int id) {
        try (Session session = neo4j.driver().session();){
            Result result = session.run("MATCH (n {id: $id}) RETURN count(n)", Values.parameters((Object[])new Object[]{"id", id}));
            int n = result.single().get(0).asInt();
            return n;
        }
    }

    private void createNodeWithId(int id) {
        try (Session session = neo4j.driver().session();){
            session.run("CREATE (n {id: $id})", Values.parameters((Object[])new Object[]{"id", id}));
        }
    }

    private static Result updateNodeId(QueryRunner queryRunner, int currentId, int newId) {
        return queryRunner.run("MATCH (n {id: $currentId}) SET n.id = $newId", Values.parameters((Object[])new Object[]{"currentId", currentId, "newId", newId}));
    }

    private static boolean assertOneOfTwoFuturesFailWithDeadlock(Future<Void> future1, Future<Void> future2) throws Exception {
        boolean firstFailed = false;
        try {
            Assertions.assertNull((Object)future1.get(20L, TimeUnit.SECONDS));
        }
        catch (ExecutionException e) {
            SessionIT.assertDeadlockDetectedError(e);
            firstFailed = true;
        }
        try {
            Assertions.assertNull((Object)future2.get(20L, TimeUnit.SECONDS));
        }
        catch (ExecutionException e) {
            Assertions.assertFalse((boolean)firstFailed, (String)"Both futures failed");
            SessionIT.assertDeadlockDetectedError(e);
        }
        return firstFailed;
    }

    private static void assertDeadlockDetectedError(ExecutionException e) {
        MatcherAssert.assertThat((Object)e.getCause(), (Matcher)CoreMatchers.instanceOf(TransientException.class));
        String errorCode = ((TransientException)e.getCause()).code();
        Assertions.assertEquals((Object)"Neo.TransientError.Transaction.DeadlockDetected", (Object)errorCode);
    }

    private <T> Future<T> executeInDifferentThread(Callable<T> callable) {
        if (this.executor == null) {
            this.executor = Executors.newCachedThreadPool(DaemonThreadFactory.daemon(this.getClass().getSimpleName() + "-thread-"));
        }
        return this.executor.submit(callable);
    }

    private static void await(CountDownLatch latch) {
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }

    private static class ThrowingWork
    implements TransactionWork<Record> {
        final String query;
        final int failures;
        int invoked;

        ThrowingWork(String query, int failures) {
            this.query = query;
            this.failures = failures;
        }

        public Record execute(Transaction tx) {
            Result result = tx.run(this.query);
            if (this.invoked++ < this.failures) {
                throw new ServiceUnavailableException("");
            }
            Record single = result.single();
            tx.commit();
            return single;
        }
    }
}

