/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.keyworddriven.feeds;

import java.util.ArrayList;
import java.util.List;
import org.mazarineblue.datasources.DataSource;
import org.mazarineblue.datasources.exceptions.DataValidationException;
import org.mazarineblue.datasources.exceptions.IllegalSourceStateException;
import org.mazarineblue.keyworddriven.InstructionLine;
import org.mazarineblue.keyworddriven.exceptions.FeedColumnValidationException;
import org.mazarineblue.keyworddriven.exceptions.IllegalFeedStateException;

/**
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 */
public class DataSourceFeed
        implements Feed {

    private final DataSource source;

    public DataSourceFeed(DataSource source) {
        this.source = source;
    }

    @Override
    public String getIdentifier() {
        return source.getSourceIdentifier();
    }

    @Override
    public boolean hasNext() {
        return source.hasNext();
    }

    @Override
    public InstructionLine next() {
        try {
            source.next();
            String lineIdentifier = source.getLineIdentifier();
            String namespace = getNamespace();
            String keyword = getKeyword();
            Object[] parameters = getParameters();
            return new InstructionLine(lineIdentifier, namespace, keyword,
                                       parameters);
        } catch (IllegalSourceStateException ex) {
            throw new IllegalFeedStateException(ex);
        }
    }

    // <editor-fold defaultstate="collapsed" desc="Methods first used in next()">
    private String getNamespace() {
        try {
            String path = getPath();
            return InstructionLine.getNamespace(path);
        } catch (IllegalSourceStateException ex) {
            throw new IllegalFeedStateException(ex);
        } catch (DataValidationException ex) {
            throw new FeedColumnValidationException(ex);
        }
    }

    private String getPath()
            throws IllegalSourceStateException, DataValidationException {
        String path = source.getData(0, String.class);
        return path == null ? "" : path;
    }

    private String getKeyword() {
        try {
            String path = getPath();
            return InstructionLine.getKeyword(path);
        } catch (IllegalSourceStateException ex) {
            throw new IllegalFeedStateException(ex);
        } catch (DataValidationException ex) {
            throw new FeedColumnValidationException(ex);
        }
    }

    private Object[] getParameters()
            throws IllegalFeedStateException {
        try {
            int index = 0;
            List<Object> list = new ArrayList();
            while (true) {
                Object obj = source.getData(++index);
                if (obj == null)
                    break;
                list.add(obj);
            }
            return list.toArray();
        } catch (IllegalSourceStateException ex) {
            throw new IllegalFeedStateException(ex);
        }
    }
}
