/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.keyworddriven.chainmanager;

import org.mazarineblue.eventbus.EventService;
import org.mazarineblue.eventbus.Event;
import org.mazarineblue.keyworddriven.librarymanager.LibraryManager;
import org.mazarineblue.keyworddriven.links.Link;
import org.mazarineblue.keyworddriven.proceduremanager.ProcedureManager;
import org.mazarineblue.parser.Parser;

/**
 * Manages a chain of links.
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 */
public class ChainManager {

    private final LibraryManager libraryManager;
    private Chain chain;

    public ChainManager(ProcedureManager procedureManager,
                        LibraryManager libraryManager, Parser parser,
                        EventService<Event> eventService) {
        this.libraryManager = libraryManager;
        chain = new Chain(createDefaultLinks(procedureManager, libraryManager,
                                             parser, eventService));
    }

    private Link[] createDefaultLinks(
            ProcedureManager procedureManager,
            LibraryManager libraryManager, Parser parser,
            EventService<Event> eventService) {
        Link[] arr = new Link[4];
        arr[0] = new ConvertLineParametersLink(parser);
        arr[1] = new ProcedureDispatcherLink(procedureManager);
        arr[2] = new LibraryDispatcherLink(libraryManager);
        arr[3] = new EventServiceLink(eventService);
        return arr;
    }

    /**
     * Passes the event to the first link, then the second link and so forth.
     * This method returns when there are no more links or when the event is
     * consumed.
     *
     * @param event the event to publish
     */
    public void publish(Event event) {
        chain.eventHandler(event);
    }

    /**
     * Inserts a link to the front of the chain.
     *
     * @param link the link to insert in from of the chain.
     */
    public void insert(Link link) {
        chain = new Chain(link, chain);
    }

    /**
     * Removes a link from the chain.
     *
     * @param link the link to remove from the chain.
     * @return true when the link was removed.
     */
    public boolean remove(Link link) {
        if (chain == null)
            return false;
        if (chain.firstLinkEqualsTo(link)) {
            chain = chain.next();
            return true;
        }
        return chain.remove(link);
    }

    /**
     * Fetches the chain count of this manager
     *
     * @return the amount of chains in this manager.
     */
    public int fetchChainCount() {
        int count = 0;
        Chain tmp = chain;
        while (tmp != null) {
            tmp = tmp.next();
            ++count;
        }
        return count;
    }
}
