/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.keyworddriven.chainmanager;

import org.mazarineblue.eventbus.Event;
import org.mazarineblue.keyworddriven.links.Link;

/**
 * A chain of link objects
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 */
class Chain {

    private final Link link;
    private Chain next;

    /**
     * Creates a chain with just one link.
     */
    Chain(Link link) {
        this.link = link;
        this.next = null;
    }

    /**
     * Creates a chain with multiple links.
     */
    Chain(Link... link) {
        next = null;
        for (int i = link.length - 1; i > 0; --i)
            next = new Chain(link[i], next);
        this.link = link[0];
    }

    /**
     * Creates a chain with the specified link in front of the specified chain.
     */
    Chain(Link link, Chain next) {
        this.link = link;
        this.next = next;
    }

    /**
     * Returns the remaining links as a chain.
     */
    Chain next() {
        return next;
    }

    /**
     * Passes the event to the first link, then the second link and so forth.
     * This method returns when there are no more links or when the event is
     * consumed.
     */
    void eventHandler(Event event) {
        link.eventHandler(event);
        if (event.isConsumed())
            return;
        if (next != null)
            next.eventHandler(event);
    }

    /**
     * Removes a link from the chain.
     *
     * @return true when the link was removed.
     */
    boolean remove(Link link) {
        if (next == null)
            return false;
        if (next.firstLinkEqualsTo(link)) {
            next = next.next;
            return true;
        }
        return next.remove(link);
    }

    boolean firstLinkEqualsTo(Link link) {
        return this.link.equals(link);
    }
}
