/*
 * Copyright (c) 2012-2014 Alex de Kruijff
 * Copyright (c) 2014-2015 Specialisterren
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package org.mazarineblue.keyworddriven;

import java.util.Date;
import java.util.Map;
import org.mazarineblue.datasources.BlackboardSource;
import org.mazarineblue.datasources.DataSource;
import org.mazarineblue.datasources.SourceChain;
import org.mazarineblue.datasources.SourceWrapper;
import org.mazarineblue.datasources.exceptions.IllegalSourceStateException;
import org.mazarineblue.eventbus.EventService;
import org.mazarineblue.eventbus.SimpleEventService;
import org.mazarineblue.eventbus.Event;
import org.mazarineblue.eventbus.exceptions.IllegalEventTypeException;
import org.mazarineblue.eventbus.exceptions.SubscriberTargetException;
import org.mazarineblue.events.EndSheetEvent;
import org.mazarineblue.events.StartSheetEvent;
import org.mazarineblue.events.instructions.ExecuteInstructionLineEvent;
import org.mazarineblue.events.instructions.InstructionLineEvent;
import org.mazarineblue.events.instructions.ValidateInstructionLineEvent;
import org.mazarineblue.keyworddriven.chainmanager.ChainManager;
import org.mazarineblue.keyworddriven.exceptions.ConsumableException;
import org.mazarineblue.keyworddriven.exceptions.GenericFeedException;
import org.mazarineblue.keyworddriven.exceptions.IllegalFeedStateException;
import org.mazarineblue.keyworddriven.exceptions.InterpreterAlReadyRunningException;
import org.mazarineblue.keyworddriven.exceptions.InterpreterSetupException;
import org.mazarineblue.keyworddriven.feeds.Feed;
import org.mazarineblue.keyworddriven.librarymanager.DefaultLibrary;
import org.mazarineblue.keyworddriven.librarymanager.LibraryManager;
import org.mazarineblue.keyworddriven.logs.Log;
import org.mazarineblue.keyworddriven.proceduremanager.ProcedureManager;
import org.mazarineblue.parser.variable.VariableParser;

/**
 *
 * @author Alex de Kruijff {@literal <alex.de.kruijff@MazarineBlue.org>}
 */
public class Processor
        implements Interpreter {

    private final BlackboardSource blackboard = new BlackboardSource(
            "Processor.BlackboardSource");
    private final SourceWrapper externalSource = new SourceWrapper();
    private final SourceChain chain = new SourceChain(blackboard, externalSource);

    private final InterpreterContext context;
    private final EventService<Event> eventService;
    private final LibraryManager libraryManager;
    private final ProcedureManager procedureManager;
    private final ChainManager chainManager;
    private int nested = 0;
    private long sleep = 0;
    private Interpreter.State state = Interpreter.State.WAITING;
    private final Date startDate;
    private InstructionLine previousLine;

    static public enum ProcessingType {

        EXECUTED,
        VALIDATED,
    }

    static public ConsumableException convertException(Throwable ex) {
        return convertException(ex.getMessage(), ex);
    }

    static public ConsumableException convertException(String msg,
                                                                Throwable ex) {
        if (ex.getCause() != null)
            ex = ex.getCause();
        return ex instanceof ConsumableException
                ? (ConsumableException) ex
                : new ConsumableException(msg, ex);
    }

    public Processor() {
        this(new Date());
    }

    public Processor(Date startDate) {
        this.startDate = startDate;
        eventService = createEventService();
        context = createContext(blackboard, this);
        procedureManager = new ProcedureManager();
        libraryManager = new LibraryManager(eventService, context);
        chainManager = new ChainManager(procedureManager, libraryManager,
                                        new VariableParser(), eventService);
        libraryManager.register(new DefaultLibrary(eventService,
                                                   procedureManager));
    }

    // <editor-fold defaultstate="collapsed" desc="Create objects">
    static private EventService createEventService() {
        try {
            return new SimpleEventService(Event.class);
        } catch (IllegalEventTypeException ex) {
            throw new InterpreterSetupException(ex);
        }
    }

    static private InterpreterContext createContext(BlackboardSource blackboard,
                                                    Interpreter executor) {
        InterpreterContext context = new InterpreterContext();
        context.setBlackboard(blackboard);
        context.setExecutor(executor);
        return context;
    }
    // </editor-fold>

    @Override
    public void setSleep(long sleep) {
        if (sleep < 0)
            throw new IllegalArgumentException(
                    "Sleep needs to be greater or equal to 0.");
        this.sleep = sleep;
    }

    @Override
    public LibraryManager libraries() {
        return libraryManager;
    }

    @Override
    public ProcedureManager procedures() {
        return procedureManager;
    }

    @Override
    public ChainManager chain() {
        return chainManager;
    }

    @Override
    public void publish(Event event) {
        chainManager.publish(event);
    }

    @Override
    public void setSource(DataSource externalSource) {
        this.externalSource.setSource(externalSource);
    }

    @Override
    public void execute(Feed feed, Log log, DocumentMediator documentMediator,
                        SheetFactory sheetFactory,
                        Map<String, Object> presetVariables) {
        new ExecuteInstructionHelper().doMain(feed, log, documentMediator,
                                              sheetFactory,
                                              presetVariables);
    }

    @Override
    public void executeNested(Feed feed, Log log, InterpreterContext context) {
        new ExecuteInstructionHelper().doNested(feed, log, context);
    }

    @Override
    public void validate(Feed feed, Log log, DocumentMediator documentMediator,
                         SheetFactory sheetFactory) {
        new ValidateInstructionHelper().doMain(feed, log, documentMediator,
                                               sheetFactory, null);
    }

    @Override
    public void validateNested(Feed feed, Log log, InterpreterContext context) {
        new ValidateInstructionHelper().doNested(feed, log, context);
    }

    private abstract class InstructionHelper {

        protected void doMain(Feed feed, Log log,
                              DocumentMediator documentMediator,
                              SheetFactory sheetFactory,
                              Map<String, Object> presetVariables) {
            if (state != Interpreter.State.WAITING)
                throw new InterpreterAlReadyRunningException();
            publish(new StartSheetEvent(null));
            try {
                state = Interpreter.State.RUNNING;
                context.set(feed, log, documentMediator, sheetFactory);
                blackboard.setup(presetVariables);
                doNested(feed, log, context);
            } finally {
                state = Interpreter.State.WAITING;
                context.set(null, null, null, null);
                blackboard.teardown();
                previousLine = null;
            }
            publish(new EndSheetEvent(null));
        }

        // <editor-fold defaultstate="collapsed" desc="Helper methods for doMain()">
        private void publish(Event event) {
            try {
                chainManager.publish(event);
            } catch (SubscriberTargetException ex) {
                Throwable cause = ex.getCause();
                throw new ConsumableException(cause.getMessage(), cause);
            }
        }
        // </editor-fold>

        public final void doNested(Feed feed, Log log,
                                   InterpreterContext context) {
            try {
                ++nested;
                incrementNestedInstruction(log, new Date());
                registerLibraries(context);
                doLines(feed, log, context);
            } finally {
                --nested;
                teardownLibraries();
                decrementNestedInstruction(log, new Date());
            }
        }

        // <editor-fold defaultstate="collapsed" desc="Helper methods for doNested()">
        private void incrementNestedInstruction(Log log, Date startDate) {
            if (nested != 1)
                log.incrementNestedInstruction(startDate);
        }

        private void decrementNestedInstruction(Log log, Date endDate) {
            if (nested != 0)
                log.decrementNestedInstruction(endDate);
        }

        private void registerLibraries(InterpreterContext context) {
            if (nested != 1)
                return;
            libraryManager.setup(context);
        }

        private void teardownLibraries() {
            if (nested == 0)
                libraryManager.teardown();
        }
        // </editor-fold>

        public final void doLines(Feed feed, Log log, InterpreterContext context) {
            try {
                while (feed.hasNext()) {
                    if (shouldAbort())
                        return;
                    InstructionLine line = feed.next();
                    doLine(line, log, context);
                    waitingForNonPausedState();
                }
            } catch (IllegalFeedStateException ex) {
                throw new GenericFeedException(ex.getMessage(), ex);
            } finally {
                if (feed.hasNext())
                    validateNested(feed, log, context);
            }
        }

        // <editor-fold defaultstate="collapsed" desc="Helper methods for doLines()">
        private boolean shouldAbort() {
            return state == Interpreter.State.CANCELED;
        }

        private void waitingForNonPausedState() {
            if (state == Interpreter.State.PAUSED)
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException ex) {
                    state = Interpreter.State.CANCELED;
                }
        }
        // </editor-fold>

        public final void doLine(InstructionLine line, Log log,
                                 InterpreterContext context) {
            if (previousLine != null && previousLine.isEmpty() == true && line.isEmpty() == true)
                return;
            try {
                setup(line, log, getProcessingType());
                Event event = createInstructionLineEvent(line);
                fireLineEvent(event, log);
                sleep(log);
            } finally {
                log.done(new Date());
                context.startDate(null);
                previousLine = line;
            }
        }

        // <editor-fold defaultstate="collapsed" desc="Helper methods for doLines()">
        private void setup(InstructionLine line, Log log, ProcessingType type) {
            try {
                context.startDate(new Date());
                context.clear(line);
                log.next(line, chain, context, type);
            } catch (IllegalSourceStateException ex) {
                throw processException(ex, log);
            }
        }

        private void fireLineEvent(Event event, Log log) {
            try {
                chainManager.publish(event);
            } catch (RuntimeException ex) {
                ConsumableException iex = processException(ex, log);
                if (iex != null)
                    throw iex;
            }
        }
        // </editor-fold>

        protected abstract ProcessingType getProcessingType();

        protected abstract Event createInstructionLineEvent(InstructionLine line);

        protected abstract void sleep(Log log);

        protected abstract ConsumableException processException(
                Exception ex,
                Log log);
    }

    private class ExecuteInstructionHelper
            extends InstructionHelper {

        @Override
        protected ProcessingType getProcessingType() {
            return ProcessingType.EXECUTED;
        }

        @Override
        protected Event createInstructionLineEvent(InstructionLine line) {
            InstructionLineEvent event = new ExecuteInstructionLineEvent(line);
            event.setDataSource(chain);
            event.setContext(context);
            return event;
        }

        @Override
        protected void sleep(Log log) {
            try {
                Thread.sleep(sleep);
            } catch (InterruptedException ex) {
                log.warning(ex);
            }
        }

        @Override
        protected ConsumableException processException(Exception ex,
                                                       Log log) {
            ConsumableException tex = convertException(ex);
            if (tex.isConsumed() == false) {
                log.error(ex);
                tex.setConsumed();
            }
            return tex;
        }
    }

    private class ValidateInstructionHelper
            extends InstructionHelper {

        @Override
        protected ProcessingType getProcessingType() {
            return ProcessingType.VALIDATED;
        }

        @Override
        protected Event createInstructionLineEvent(InstructionLine line) {
            InstructionLineEvent event = new ValidateInstructionLineEvent(line);
            event.setDataSource(chain);
            event.setContext(context);
            return event;
        }

        @Override
        protected void sleep(Log log) {
        }

        @Override
        protected ConsumableException processException(Exception ex,
                                                       Log log) {
            ConsumableException tex = convertException(ex);
            if (tex.isConsumed() == false) {
                log.error(ex);
                tex.setConsumed();
            }
            return null;
        }
    }

    @Override
    public Interpreter.State getState() {
        return state;
    }

    @Override
    public void pause() {
        state = Interpreter.State.PAUSED;
    }

    @Override
    public void resume() {
        state = Interpreter.State.RUNNING;
    }

    @Override
    public void cancle() {
        state = Interpreter.State.CANCELED;
    }

    @Override
    public Date getStartDate() {
        return startDate;
    }
}
