package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link NodeDescription}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNodeDescription.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNodeDescription implements NodeDescription {
  private final String hostname;
  private final Platform platform;
  private final Resources resources;
  private final EngineConfig engine;

  private ImmutableNodeDescription(
      String hostname,
      Platform platform,
      Resources resources,
      EngineConfig engine) {
    this.hostname = hostname;
    this.platform = platform;
    this.resources = resources;
    this.engine = engine;
  }

  /**
   * @return The value of the {@code hostname} attribute
   */
  @JsonProperty("Hostname")
  @Override
  public String hostname() {
    return hostname;
  }

  /**
   * @return The value of the {@code platform} attribute
   */
  @JsonProperty("Platform")
  @Override
  public Platform platform() {
    return platform;
  }

  /**
   * @return The value of the {@code resources} attribute
   */
  @JsonProperty("Resources")
  @Override
  public Resources resources() {
    return resources;
  }

  /**
   * @return The value of the {@code engine} attribute
   */
  @JsonProperty("Engine")
  @Override
  public EngineConfig engine() {
    return engine;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeDescription#hostname() hostname} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for hostname
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNodeDescription withHostname(String value) {
    String newValue = Objects.requireNonNull(value, "hostname");
    if (this.hostname.equals(newValue)) return this;
    return new ImmutableNodeDescription(newValue, this.platform, this.resources, this.engine);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeDescription#platform() platform} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for platform
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNodeDescription withPlatform(Platform value) {
    if (this.platform == value) return this;
    Platform newValue = Objects.requireNonNull(value, "platform");
    return new ImmutableNodeDescription(this.hostname, newValue, this.resources, this.engine);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeDescription#resources() resources} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for resources
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNodeDescription withResources(Resources value) {
    if (this.resources == value) return this;
    Resources newValue = Objects.requireNonNull(value, "resources");
    return new ImmutableNodeDescription(this.hostname, this.platform, newValue, this.engine);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NodeDescription#engine() engine} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for engine
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNodeDescription withEngine(EngineConfig value) {
    if (this.engine == value) return this;
    EngineConfig newValue = Objects.requireNonNull(value, "engine");
    return new ImmutableNodeDescription(this.hostname, this.platform, this.resources, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNodeDescription} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNodeDescription
        && equalTo(0, (ImmutableNodeDescription) another);
  }

  private boolean equalTo(int synthetic, ImmutableNodeDescription another) {
    return hostname.equals(another.hostname)
        && platform.equals(another.platform)
        && resources.equals(another.resources)
        && engine.equals(another.engine);
  }

  /**
   * Computes a hash code from attributes: {@code hostname}, {@code platform}, {@code resources}, {@code engine}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + hostname.hashCode();
    h += (h << 5) + platform.hashCode();
    h += (h << 5) + resources.hashCode();
    h += (h << 5) + engine.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code NodeDescription} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NodeDescription{"
        + "hostname=" + hostname
        + ", platform=" + platform
        + ", resources=" + resources
        + ", engine=" + engine
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NodeDescription} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NodeDescription instance
   */
  public static ImmutableNodeDescription copyOf(NodeDescription instance) {
    if (instance instanceof ImmutableNodeDescription) {
      return (ImmutableNodeDescription) instance;
    }
    return ImmutableNodeDescription.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNodeDescription ImmutableNodeDescription}.
   * <pre>
   * ImmutableNodeDescription.builder()
   *    .hostname(String) // required {@link NodeDescription#hostname() hostname}
   *    .platform(org.mandas.docker.client.messages.swarm.Platform) // required {@link NodeDescription#platform() platform}
   *    .resources(org.mandas.docker.client.messages.swarm.Resources) // required {@link NodeDescription#resources() resources}
   *    .engine(org.mandas.docker.client.messages.swarm.EngineConfig) // required {@link NodeDescription#engine() engine}
   *    .build();
   * </pre>
   * @return A new ImmutableNodeDescription builder
   */
  public static ImmutableNodeDescription.Builder builder() {
    return new ImmutableNodeDescription.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNodeDescription ImmutableNodeDescription}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_HOSTNAME = 0x1L;
    private static final long INIT_BIT_PLATFORM = 0x2L;
    private static final long INIT_BIT_RESOURCES = 0x4L;
    private static final long INIT_BIT_ENGINE = 0x8L;
    private long initBits = 0xfL;

    private String hostname;
    private Platform platform;
    private Resources resources;
    private EngineConfig engine;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NodeDescription} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NodeDescription instance) {
      Objects.requireNonNull(instance, "instance");
      hostname(instance.hostname());
      platform(instance.platform());
      resources(instance.resources());
      engine(instance.engine());
      return this;
    }

    /**
     * Initializes the value for the {@link NodeDescription#hostname() hostname} attribute.
     * @param hostname The value for hostname 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Hostname")
    public final Builder hostname(String hostname) {
      this.hostname = Objects.requireNonNull(hostname, "hostname");
      initBits &= ~INIT_BIT_HOSTNAME;
      return this;
    }

    /**
     * Initializes the value for the {@link NodeDescription#platform() platform} attribute.
     * @param platform The value for platform 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Platform")
    public final Builder platform(Platform platform) {
      this.platform = Objects.requireNonNull(platform, "platform");
      initBits &= ~INIT_BIT_PLATFORM;
      return this;
    }

    /**
     * Initializes the value for the {@link NodeDescription#resources() resources} attribute.
     * @param resources The value for resources 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Resources")
    public final Builder resources(Resources resources) {
      this.resources = Objects.requireNonNull(resources, "resources");
      initBits &= ~INIT_BIT_RESOURCES;
      return this;
    }

    /**
     * Initializes the value for the {@link NodeDescription#engine() engine} attribute.
     * @param engine The value for engine 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Engine")
    public final Builder engine(EngineConfig engine) {
      this.engine = Objects.requireNonNull(engine, "engine");
      initBits &= ~INIT_BIT_ENGINE;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNodeDescription ImmutableNodeDescription}.
     * @return An immutable instance of NodeDescription
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNodeDescription build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNodeDescription(hostname, platform, resources, engine);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_HOSTNAME) != 0) attributes.add("hostname");
      if ((initBits & INIT_BIT_PLATFORM) != 0) attributes.add("platform");
      if ((initBits & INIT_BIT_RESOURCES) != 0) attributes.add("resources");
      if ((initBits & INIT_BIT_ENGINE) != 0) attributes.add("engine");
      return "Cannot build NodeDescription, some of required attributes are not set " + attributes;
    }
  }
}
