package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link IpamConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableIpamConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableIpamConfig implements IpamConfig {
  private final String subnet;
  private final @Nullable String range;
  private final String gateway;

  private ImmutableIpamConfig(
      String subnet,
      @Nullable String range,
      String gateway) {
    this.subnet = subnet;
    this.range = range;
    this.gateway = gateway;
  }

  /**
   * @return The value of the {@code subnet} attribute
   */
  @JsonProperty("Subnet")
  @Override
  public String subnet() {
    return subnet;
  }

  /**
   * @return The value of the {@code range} attribute
   */
  @JsonProperty("Range")
  @Override
  public @Nullable String range() {
    return range;
  }

  /**
   * @return The value of the {@code gateway} attribute
   */
  @JsonProperty("Gateway")
  @Override
  public String gateway() {
    return gateway;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link IpamConfig#subnet() subnet} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for subnet
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableIpamConfig withSubnet(String value) {
    String newValue = Objects.requireNonNull(value, "subnet");
    if (this.subnet.equals(newValue)) return this;
    return new ImmutableIpamConfig(newValue, this.range, this.gateway);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link IpamConfig#range() range} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for range (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableIpamConfig withRange(@Nullable String value) {
    if (Objects.equals(this.range, value)) return this;
    return new ImmutableIpamConfig(this.subnet, value, this.gateway);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link IpamConfig#gateway() gateway} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for gateway
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableIpamConfig withGateway(String value) {
    String newValue = Objects.requireNonNull(value, "gateway");
    if (this.gateway.equals(newValue)) return this;
    return new ImmutableIpamConfig(this.subnet, this.range, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableIpamConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableIpamConfig
        && equalTo(0, (ImmutableIpamConfig) another);
  }

  private boolean equalTo(int synthetic, ImmutableIpamConfig another) {
    return subnet.equals(another.subnet)
        && Objects.equals(range, another.range)
        && gateway.equals(another.gateway);
  }

  /**
   * Computes a hash code from attributes: {@code subnet}, {@code range}, {@code gateway}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + subnet.hashCode();
    h += (h << 5) + Objects.hashCode(range);
    h += (h << 5) + gateway.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code IpamConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "IpamConfig{"
        + "subnet=" + subnet
        + ", range=" + range
        + ", gateway=" + gateway
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link IpamConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable IpamConfig instance
   */
  public static ImmutableIpamConfig copyOf(IpamConfig instance) {
    if (instance instanceof ImmutableIpamConfig) {
      return (ImmutableIpamConfig) instance;
    }
    return ImmutableIpamConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableIpamConfig ImmutableIpamConfig}.
   * <pre>
   * ImmutableIpamConfig.builder()
   *    .subnet(String) // required {@link IpamConfig#subnet() subnet}
   *    .range(String | null) // nullable {@link IpamConfig#range() range}
   *    .gateway(String) // required {@link IpamConfig#gateway() gateway}
   *    .build();
   * </pre>
   * @return A new ImmutableIpamConfig builder
   */
  public static ImmutableIpamConfig.Builder builder() {
    return new ImmutableIpamConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableIpamConfig ImmutableIpamConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_SUBNET = 0x1L;
    private static final long INIT_BIT_GATEWAY = 0x2L;
    private long initBits = 0x3L;

    private String subnet;
    private String range;
    private String gateway;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code IpamConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(IpamConfig instance) {
      Objects.requireNonNull(instance, "instance");
      subnet(instance.subnet());
      @Nullable String rangeValue = instance.range();
      if (rangeValue != null) {
        range(rangeValue);
      }
      gateway(instance.gateway());
      return this;
    }

    /**
     * Initializes the value for the {@link IpamConfig#subnet() subnet} attribute.
     * @param subnet The value for subnet 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Subnet")
    public final Builder subnet(String subnet) {
      this.subnet = Objects.requireNonNull(subnet, "subnet");
      initBits &= ~INIT_BIT_SUBNET;
      return this;
    }

    /**
     * Initializes the value for the {@link IpamConfig#range() range} attribute.
     * @param range The value for range (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Range")
    public final Builder range(@Nullable String range) {
      this.range = range;
      return this;
    }

    /**
     * Initializes the value for the {@link IpamConfig#gateway() gateway} attribute.
     * @param gateway The value for gateway 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Gateway")
    public final Builder gateway(String gateway) {
      this.gateway = Objects.requireNonNull(gateway, "gateway");
      initBits &= ~INIT_BIT_GATEWAY;
      return this;
    }

    /**
     * Builds a new {@link ImmutableIpamConfig ImmutableIpamConfig}.
     * @return An immutable instance of IpamConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableIpamConfig build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableIpamConfig(subnet, range, gateway);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_SUBNET) != 0) attributes.add("subnet");
      if ((initBits & INIT_BIT_GATEWAY) != 0) attributes.add("gateway");
      return "Cannot build IpamConfig, some of required attributes are not set " + attributes;
    }
  }
}
