package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Endpoint}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableEndpoint.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableEndpoint implements Endpoint {
  private final EndpointSpec spec;
  private final @Nullable List<PortConfig> exposedPorts;
  private final @Nullable List<PortConfig> ports;
  private final @Nullable List<EndpointVirtualIp> virtualIps;

  private ImmutableEndpoint(
      EndpointSpec spec,
      @Nullable List<PortConfig> exposedPorts,
      @Nullable List<PortConfig> ports,
      @Nullable List<EndpointVirtualIp> virtualIps) {
    this.spec = spec;
    this.exposedPorts = exposedPorts;
    this.ports = ports;
    this.virtualIps = virtualIps;
  }

  /**
   * @return The value of the {@code spec} attribute
   */
  @JsonProperty("Spec")
  @Override
  public EndpointSpec spec() {
    return spec;
  }

  /**
   * @return The value of the {@code exposedPorts} attribute
   */
  @JsonProperty("ExposedPorts")
  @Override
  public @Nullable List<PortConfig> exposedPorts() {
    return exposedPorts;
  }

  /**
   * @return The value of the {@code ports} attribute
   */
  @JsonProperty("Ports")
  @Override
  public @Nullable List<PortConfig> ports() {
    return ports;
  }

  /**
   * @return The value of the {@code virtualIps} attribute
   */
  @JsonProperty("VirtualIPs")
  @Override
  public @Nullable List<EndpointVirtualIp> virtualIps() {
    return virtualIps;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Endpoint#spec() spec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for spec
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableEndpoint withSpec(EndpointSpec value) {
    if (this.spec == value) return this;
    EndpointSpec newValue = Objects.requireNonNull(value, "spec");
    return new ImmutableEndpoint(newValue, this.exposedPorts, this.ports, this.virtualIps);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Endpoint#exposedPorts() exposedPorts}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEndpoint withExposedPorts(@Nullable PortConfig... elements) {
    if (elements == null) {
      return new ImmutableEndpoint(this.spec, null, this.ports, this.virtualIps);
    }
    @Nullable List<PortConfig> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableEndpoint(this.spec, newValue, this.ports, this.virtualIps);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Endpoint#exposedPorts() exposedPorts}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of exposedPorts elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEndpoint withExposedPorts(@Nullable Iterable<? extends PortConfig> elements) {
    if (this.exposedPorts == elements) return this;
    @Nullable List<PortConfig> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableEndpoint(this.spec, newValue, this.ports, this.virtualIps);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Endpoint#ports() ports}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEndpoint withPorts(@Nullable PortConfig... elements) {
    if (elements == null) {
      return new ImmutableEndpoint(this.spec, this.exposedPorts, null, this.virtualIps);
    }
    @Nullable List<PortConfig> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableEndpoint(this.spec, this.exposedPorts, newValue, this.virtualIps);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Endpoint#ports() ports}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of ports elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEndpoint withPorts(@Nullable Iterable<? extends PortConfig> elements) {
    if (this.ports == elements) return this;
    @Nullable List<PortConfig> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableEndpoint(this.spec, this.exposedPorts, newValue, this.virtualIps);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Endpoint#virtualIps() virtualIps}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEndpoint withVirtualIps(@Nullable EndpointVirtualIp... elements) {
    if (elements == null) {
      return new ImmutableEndpoint(this.spec, this.exposedPorts, this.ports, null);
    }
    @Nullable List<EndpointVirtualIp> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableEndpoint(this.spec, this.exposedPorts, this.ports, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Endpoint#virtualIps() virtualIps}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of virtualIps elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEndpoint withVirtualIps(@Nullable Iterable<? extends EndpointVirtualIp> elements) {
    if (this.virtualIps == elements) return this;
    @Nullable List<EndpointVirtualIp> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableEndpoint(this.spec, this.exposedPorts, this.ports, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableEndpoint} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableEndpoint
        && equalTo(0, (ImmutableEndpoint) another);
  }

  private boolean equalTo(int synthetic, ImmutableEndpoint another) {
    return spec.equals(another.spec)
        && Objects.equals(exposedPorts, another.exposedPorts)
        && Objects.equals(ports, another.ports)
        && Objects.equals(virtualIps, another.virtualIps);
  }

  /**
   * Computes a hash code from attributes: {@code spec}, {@code exposedPorts}, {@code ports}, {@code virtualIps}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + spec.hashCode();
    h += (h << 5) + Objects.hashCode(exposedPorts);
    h += (h << 5) + Objects.hashCode(ports);
    h += (h << 5) + Objects.hashCode(virtualIps);
    return h;
  }

  /**
   * Prints the immutable value {@code Endpoint} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Endpoint{"
        + "spec=" + spec
        + ", exposedPorts=" + exposedPorts
        + ", ports=" + ports
        + ", virtualIps=" + virtualIps
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Endpoint} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Endpoint instance
   */
  public static ImmutableEndpoint copyOf(Endpoint instance) {
    if (instance instanceof ImmutableEndpoint) {
      return (ImmutableEndpoint) instance;
    }
    return ImmutableEndpoint.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableEndpoint ImmutableEndpoint}.
   * <pre>
   * ImmutableEndpoint.builder()
   *    .spec(org.mandas.docker.client.messages.swarm.EndpointSpec) // required {@link Endpoint#spec() spec}
   *    .exposedPorts(List&amp;lt;org.mandas.docker.client.messages.swarm.PortConfig&amp;gt; | null) // nullable {@link Endpoint#exposedPorts() exposedPorts}
   *    .ports(List&amp;lt;org.mandas.docker.client.messages.swarm.PortConfig&amp;gt; | null) // nullable {@link Endpoint#ports() ports}
   *    .virtualIps(List&amp;lt;org.mandas.docker.client.messages.swarm.EndpointVirtualIp&amp;gt; | null) // nullable {@link Endpoint#virtualIps() virtualIps}
   *    .build();
   * </pre>
   * @return A new ImmutableEndpoint builder
   */
  public static ImmutableEndpoint.Builder builder() {
    return new ImmutableEndpoint.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableEndpoint ImmutableEndpoint}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_SPEC = 0x1L;
    private long initBits = 0x1L;

    private EndpointSpec spec;
    private List<PortConfig> exposedPorts = null;
    private List<PortConfig> ports = null;
    private List<EndpointVirtualIp> virtualIps = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Endpoint} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Endpoint instance) {
      Objects.requireNonNull(instance, "instance");
      spec(instance.spec());
      @Nullable List<PortConfig> exposedPortsValue = instance.exposedPorts();
      if (exposedPortsValue != null) {
        addAllExposedPorts(exposedPortsValue);
      }
      @Nullable List<PortConfig> portsValue = instance.ports();
      if (portsValue != null) {
        addAllPorts(portsValue);
      }
      @Nullable List<EndpointVirtualIp> virtualIpsValue = instance.virtualIps();
      if (virtualIpsValue != null) {
        addAllVirtualIps(virtualIpsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Endpoint#spec() spec} attribute.
     * @param spec The value for spec 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Spec")
    public final Builder spec(EndpointSpec spec) {
      this.spec = Objects.requireNonNull(spec, "spec");
      initBits &= ~INIT_BIT_SPEC;
      return this;
    }

    /**
     * Adds one element to {@link Endpoint#exposedPorts() exposedPorts} list.
     * @param element A exposedPorts element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder exposedPort(PortConfig element) {
      if (this.exposedPorts == null) {
        this.exposedPorts = new ArrayList<PortConfig>();
      }
      this.exposedPorts.add(Objects.requireNonNull(element, "exposedPorts element"));
      return this;
    }

    /**
     * Adds elements to {@link Endpoint#exposedPorts() exposedPorts} list.
     * @param elements An array of exposedPorts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder exposedPorts(PortConfig... elements) {
      if (this.exposedPorts == null) {
        this.exposedPorts = new ArrayList<PortConfig>();
      }
      for (PortConfig element : elements) {
        this.exposedPorts.add(Objects.requireNonNull(element, "exposedPorts element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Endpoint#exposedPorts() exposedPorts} list.
     * @param elements An iterable of exposedPorts elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ExposedPorts")
    public final Builder exposedPorts(@Nullable Iterable<? extends PortConfig> elements) {
      if (elements == null) {
        this.exposedPorts = null;
        return this;
      }
      this.exposedPorts = new ArrayList<PortConfig>();
      return addAllExposedPorts(elements);
    }

    /**
     * Adds elements to {@link Endpoint#exposedPorts() exposedPorts} list.
     * @param elements An iterable of exposedPorts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllExposedPorts(Iterable<? extends PortConfig> elements) {
      Objects.requireNonNull(elements, "exposedPorts element");
      if (this.exposedPorts == null) {
        this.exposedPorts = new ArrayList<PortConfig>();
      }
      for (PortConfig element : elements) {
        this.exposedPorts.add(Objects.requireNonNull(element, "exposedPorts element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link Endpoint#ports() ports} list.
     * @param element A ports element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder port(PortConfig element) {
      if (this.ports == null) {
        this.ports = new ArrayList<PortConfig>();
      }
      this.ports.add(Objects.requireNonNull(element, "ports element"));
      return this;
    }

    /**
     * Adds elements to {@link Endpoint#ports() ports} list.
     * @param elements An array of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder ports(PortConfig... elements) {
      if (this.ports == null) {
        this.ports = new ArrayList<PortConfig>();
      }
      for (PortConfig element : elements) {
        this.ports.add(Objects.requireNonNull(element, "ports element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Endpoint#ports() ports} list.
     * @param elements An iterable of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Ports")
    public final Builder ports(@Nullable Iterable<? extends PortConfig> elements) {
      if (elements == null) {
        this.ports = null;
        return this;
      }
      this.ports = new ArrayList<PortConfig>();
      return addAllPorts(elements);
    }

    /**
     * Adds elements to {@link Endpoint#ports() ports} list.
     * @param elements An iterable of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllPorts(Iterable<? extends PortConfig> elements) {
      Objects.requireNonNull(elements, "ports element");
      if (this.ports == null) {
        this.ports = new ArrayList<PortConfig>();
      }
      for (PortConfig element : elements) {
        this.ports.add(Objects.requireNonNull(element, "ports element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link Endpoint#virtualIps() virtualIps} list.
     * @param element A virtualIps element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder virtualIp(EndpointVirtualIp element) {
      if (this.virtualIps == null) {
        this.virtualIps = new ArrayList<EndpointVirtualIp>();
      }
      this.virtualIps.add(Objects.requireNonNull(element, "virtualIps element"));
      return this;
    }

    /**
     * Adds elements to {@link Endpoint#virtualIps() virtualIps} list.
     * @param elements An array of virtualIps elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder virtualIps(EndpointVirtualIp... elements) {
      if (this.virtualIps == null) {
        this.virtualIps = new ArrayList<EndpointVirtualIp>();
      }
      for (EndpointVirtualIp element : elements) {
        this.virtualIps.add(Objects.requireNonNull(element, "virtualIps element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Endpoint#virtualIps() virtualIps} list.
     * @param elements An iterable of virtualIps elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("VirtualIPs")
    public final Builder virtualIps(@Nullable Iterable<? extends EndpointVirtualIp> elements) {
      if (elements == null) {
        this.virtualIps = null;
        return this;
      }
      this.virtualIps = new ArrayList<EndpointVirtualIp>();
      return addAllVirtualIps(elements);
    }

    /**
     * Adds elements to {@link Endpoint#virtualIps() virtualIps} list.
     * @param elements An iterable of virtualIps elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllVirtualIps(Iterable<? extends EndpointVirtualIp> elements) {
      Objects.requireNonNull(elements, "virtualIps element");
      if (this.virtualIps == null) {
        this.virtualIps = new ArrayList<EndpointVirtualIp>();
      }
      for (EndpointVirtualIp element : elements) {
        this.virtualIps.add(Objects.requireNonNull(element, "virtualIps element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableEndpoint ImmutableEndpoint}.
     * @return An immutable instance of Endpoint
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableEndpoint build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableEndpoint(
          spec,
          exposedPorts == null ? null : createUnmodifiableList(true, exposedPorts),
          ports == null ? null : createUnmodifiableList(true, ports),
          virtualIps == null ? null : createUnmodifiableList(true, virtualIps));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_SPEC) != 0) attributes.add("spec");
      return "Cannot build Endpoint, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
