package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link NetworkAttachment}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNetworkAttachment.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNetworkAttachment
    implements NetworkAttachment {
  private final Network network;
  private final List<String> addresses;

  private ImmutableNetworkAttachment(Network network, List<String> addresses) {
    this.network = network;
    this.addresses = addresses;
  }

  /**
   * @return The value of the {@code network} attribute
   */
  @JsonProperty("Network")
  @Override
  public Network network() {
    return network;
  }

  /**
   * @return The value of the {@code addresses} attribute
   */
  @JsonProperty("Addresses")
  @Override
  public List<String> addresses() {
    return addresses;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkAttachment#network() network} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for network
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNetworkAttachment withNetwork(Network value) {
    if (this.network == value) return this;
    Network newValue = Objects.requireNonNull(value, "network");
    return new ImmutableNetworkAttachment(newValue, this.addresses);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link NetworkAttachment#addresses() addresses}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableNetworkAttachment withAddresses(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableNetworkAttachment(this.network, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link NetworkAttachment#addresses() addresses}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of addresses elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableNetworkAttachment withAddresses(Iterable<String> elements) {
    if (this.addresses == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableNetworkAttachment(this.network, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNetworkAttachment} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNetworkAttachment
        && equalTo(0, (ImmutableNetworkAttachment) another);
  }

  private boolean equalTo(int synthetic, ImmutableNetworkAttachment another) {
    return network.equals(another.network)
        && addresses.equals(another.addresses);
  }

  /**
   * Computes a hash code from attributes: {@code network}, {@code addresses}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + network.hashCode();
    h += (h << 5) + addresses.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code NetworkAttachment} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NetworkAttachment{"
        + "network=" + network
        + ", addresses=" + addresses
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NetworkAttachment} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NetworkAttachment instance
   */
  public static ImmutableNetworkAttachment copyOf(NetworkAttachment instance) {
    if (instance instanceof ImmutableNetworkAttachment) {
      return (ImmutableNetworkAttachment) instance;
    }
    return ImmutableNetworkAttachment.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNetworkAttachment ImmutableNetworkAttachment}.
   * <pre>
   * ImmutableNetworkAttachment.builder()
   *    .network(org.mandas.docker.client.messages.swarm.Network) // required {@link NetworkAttachment#network() network}
   *    .addresse|addAllAddresses(String) // {@link NetworkAttachment#addresses() addresses} elements
   *    .build();
   * </pre>
   * @return A new ImmutableNetworkAttachment builder
   */
  public static ImmutableNetworkAttachment.Builder builder() {
    return new ImmutableNetworkAttachment.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNetworkAttachment ImmutableNetworkAttachment}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_NETWORK = 0x1L;
    private long initBits = 0x1L;

    private Network network;
    private List<String> addresses = new ArrayList<String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NetworkAttachment} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NetworkAttachment instance) {
      Objects.requireNonNull(instance, "instance");
      network(instance.network());
      addAllAddresses(instance.addresses());
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkAttachment#network() network} attribute.
     * @param network The value for network 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Network")
    public final Builder network(Network network) {
      this.network = Objects.requireNonNull(network, "network");
      initBits &= ~INIT_BIT_NETWORK;
      return this;
    }

    /**
     * Adds one element to {@link NetworkAttachment#addresses() addresses} list.
     * @param element A addresses element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addresse(String element) {
      this.addresses.add(Objects.requireNonNull(element, "addresses element"));
      return this;
    }

    /**
     * Adds elements to {@link NetworkAttachment#addresses() addresses} list.
     * @param elements An array of addresses elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addresses(String... elements) {
      for (String element : elements) {
        this.addresses.add(Objects.requireNonNull(element, "addresses element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link NetworkAttachment#addresses() addresses} list.
     * @param elements An iterable of addresses elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Addresses")
    public final Builder addresses(Iterable<String> elements) {
      this.addresses.clear();
      return addAllAddresses(elements);
    }

    /**
     * Adds elements to {@link NetworkAttachment#addresses() addresses} list.
     * @param elements An iterable of addresses elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllAddresses(Iterable<String> elements) {
      for (String element : elements) {
        this.addresses.add(Objects.requireNonNull(element, "addresses element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableNetworkAttachment ImmutableNetworkAttachment}.
     * @return An immutable instance of NetworkAttachment
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNetworkAttachment build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNetworkAttachment(network, createUnmodifiableList(true, addresses));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_NETWORK) != 0) attributes.add("network");
      return "Cannot build NetworkAttachment, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
