package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link TopResults}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableTopResults.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableTopResults implements TopResults {
  private final List<String> titles;
  private final List<List<String>> processes;

  private ImmutableTopResults(List<String> titles, List<List<String>> processes) {
    this.titles = titles;
    this.processes = processes;
  }

  /**
   * @return The value of the {@code titles} attribute
   */
  @JsonProperty("Titles")
  @Override
  public List<String> titles() {
    return titles;
  }

  /**
   * @return The value of the {@code processes} attribute
   */
  @JsonProperty("Processes")
  @Override
  public List<List<String>> processes() {
    return processes;
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link TopResults#titles() titles}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTopResults withTitles(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableTopResults(newValue, this.processes);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link TopResults#titles() titles}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of titles elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTopResults withTitles(Iterable<String> elements) {
    if (this.titles == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableTopResults(newValue, this.processes);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link TopResults#processes() processes}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  @SafeVarargs @SuppressWarnings("varargs")
  public final ImmutableTopResults withProcesses(List<String>... elements) {
    List<List<String>> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableTopResults(this.titles, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link TopResults#processes() processes}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of processes elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTopResults withProcesses(Iterable<? extends List<String>> elements) {
    if (this.processes == elements) return this;
    List<List<String>> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableTopResults(this.titles, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableTopResults} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableTopResults
        && equalTo(0, (ImmutableTopResults) another);
  }

  private boolean equalTo(int synthetic, ImmutableTopResults another) {
    return titles.equals(another.titles)
        && processes.equals(another.processes);
  }

  /**
   * Computes a hash code from attributes: {@code titles}, {@code processes}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + titles.hashCode();
    h += (h << 5) + processes.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code TopResults} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "TopResults{"
        + "titles=" + titles
        + ", processes=" + processes
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link TopResults} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable TopResults instance
   */
  public static ImmutableTopResults copyOf(TopResults instance) {
    if (instance instanceof ImmutableTopResults) {
      return (ImmutableTopResults) instance;
    }
    return ImmutableTopResults.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableTopResults ImmutableTopResults}.
   * <pre>
   * ImmutableTopResults.builder()
   *    .title|addAllTitles(String) // {@link TopResults#titles() titles} elements
   *    .processe|addAllProcesses(List&amp;lt;String&amp;gt;) // {@link TopResults#processes() processes} elements
   *    .build();
   * </pre>
   * @return A new ImmutableTopResults builder
   */
  public static ImmutableTopResults.Builder builder() {
    return new ImmutableTopResults.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableTopResults ImmutableTopResults}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private List<String> titles = new ArrayList<String>();
    private List<List<String>> processes = new ArrayList<List<String>>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code TopResults} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(TopResults instance) {
      Objects.requireNonNull(instance, "instance");
      addAllTitles(instance.titles());
      addAllProcesses(instance.processes());
      return this;
    }

    /**
     * Adds one element to {@link TopResults#titles() titles} list.
     * @param element A titles element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder title(String element) {
      this.titles.add(Objects.requireNonNull(element, "titles element"));
      return this;
    }

    /**
     * Adds elements to {@link TopResults#titles() titles} list.
     * @param elements An array of titles elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder titles(String... elements) {
      for (String element : elements) {
        this.titles.add(Objects.requireNonNull(element, "titles element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link TopResults#titles() titles} list.
     * @param elements An iterable of titles elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Titles")
    public final Builder titles(Iterable<String> elements) {
      this.titles.clear();
      return addAllTitles(elements);
    }

    /**
     * Adds elements to {@link TopResults#titles() titles} list.
     * @param elements An iterable of titles elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllTitles(Iterable<String> elements) {
      for (String element : elements) {
        this.titles.add(Objects.requireNonNull(element, "titles element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link TopResults#processes() processes} list.
     * @param element A processes element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder processe(List<String> element) {
      this.processes.add(Objects.requireNonNull(element, "processes element"));
      return this;
    }

    /**
     * Adds elements to {@link TopResults#processes() processes} list.
     * @param elements An array of processes elements
     * @return {@code this} builder for use in a chained invocation
     */
    @SafeVarargs @SuppressWarnings("varargs")
    public final Builder processes(List<String>... elements) {
      for (List<String> element : elements) {
        this.processes.add(Objects.requireNonNull(element, "processes element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link TopResults#processes() processes} list.
     * @param elements An iterable of processes elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Processes")
    public final Builder processes(Iterable<? extends List<String>> elements) {
      this.processes.clear();
      return addAllProcesses(elements);
    }

    /**
     * Adds elements to {@link TopResults#processes() processes} list.
     * @param elements An iterable of processes elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllProcesses(Iterable<? extends List<String>> elements) {
      for (List<String> element : elements) {
        this.processes.add(Objects.requireNonNull(element, "processes element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableTopResults ImmutableTopResults}.
     * @return An immutable instance of TopResults
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableTopResults build() {
      return new ImmutableTopResults(createUnmodifiableList(true, titles), createUnmodifiableList(true, processes));
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
