package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link TaskSpec}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableTaskSpec.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableTaskSpec implements TaskSpec {
  private final @Nullable ContainerSpec containerSpec;
  private final @Nullable ResourceRequirements resources;
  private final @Nullable RestartPolicy restartPolicy;
  private final @Nullable Placement placement;
  private final @Nullable List<NetworkAttachmentConfig> networks;
  private final @Nullable Driver logDriver;

  private ImmutableTaskSpec(
      @Nullable ContainerSpec containerSpec,
      @Nullable ResourceRequirements resources,
      @Nullable RestartPolicy restartPolicy,
      @Nullable Placement placement,
      @Nullable List<NetworkAttachmentConfig> networks,
      @Nullable Driver logDriver) {
    this.containerSpec = containerSpec;
    this.resources = resources;
    this.restartPolicy = restartPolicy;
    this.placement = placement;
    this.networks = networks;
    this.logDriver = logDriver;
  }

  /**
   * @return The value of the {@code containerSpec} attribute
   */
  @JsonProperty("ContainerSpec")
  @Override
  public @Nullable ContainerSpec containerSpec() {
    return containerSpec;
  }

  /**
   * @return The value of the {@code resources} attribute
   */
  @JsonProperty("Resources")
  @Override
  public @Nullable ResourceRequirements resources() {
    return resources;
  }

  /**
   * @return The value of the {@code restartPolicy} attribute
   */
  @JsonProperty("RestartPolicy")
  @Override
  public @Nullable RestartPolicy restartPolicy() {
    return restartPolicy;
  }

  /**
   * @return The value of the {@code placement} attribute
   */
  @JsonProperty("Placement")
  @Override
  public @Nullable Placement placement() {
    return placement;
  }

  /**
   * @return The value of the {@code networks} attribute
   */
  @JsonProperty("Networks")
  @Override
  public @Nullable List<NetworkAttachmentConfig> networks() {
    return networks;
  }

  /**
   * @return The value of the {@code logDriver} attribute
   */
  @JsonProperty("LogDriver")
  @Override
  public @Nullable Driver logDriver() {
    return logDriver;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TaskSpec#containerSpec() containerSpec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for containerSpec (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTaskSpec withContainerSpec(@Nullable ContainerSpec value) {
    if (this.containerSpec == value) return this;
    return new ImmutableTaskSpec(value, this.resources, this.restartPolicy, this.placement, this.networks, this.logDriver);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TaskSpec#resources() resources} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for resources (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTaskSpec withResources(@Nullable ResourceRequirements value) {
    if (this.resources == value) return this;
    return new ImmutableTaskSpec(this.containerSpec, value, this.restartPolicy, this.placement, this.networks, this.logDriver);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TaskSpec#restartPolicy() restartPolicy} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for restartPolicy (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTaskSpec withRestartPolicy(@Nullable RestartPolicy value) {
    if (this.restartPolicy == value) return this;
    return new ImmutableTaskSpec(this.containerSpec, this.resources, value, this.placement, this.networks, this.logDriver);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TaskSpec#placement() placement} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for placement (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTaskSpec withPlacement(@Nullable Placement value) {
    if (this.placement == value) return this;
    return new ImmutableTaskSpec(this.containerSpec, this.resources, this.restartPolicy, value, this.networks, this.logDriver);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link TaskSpec#networks() networks}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTaskSpec withNetworks(@Nullable NetworkAttachmentConfig... elements) {
    if (elements == null) {
      return new ImmutableTaskSpec(this.containerSpec, this.resources, this.restartPolicy, this.placement, null, this.logDriver);
    }
    @Nullable List<NetworkAttachmentConfig> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableTaskSpec(
        this.containerSpec,
        this.resources,
        this.restartPolicy,
        this.placement,
        newValue,
        this.logDriver);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link TaskSpec#networks() networks}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of networks elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableTaskSpec withNetworks(@Nullable Iterable<? extends NetworkAttachmentConfig> elements) {
    if (this.networks == elements) return this;
    @Nullable List<NetworkAttachmentConfig> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableTaskSpec(
        this.containerSpec,
        this.resources,
        this.restartPolicy,
        this.placement,
        newValue,
        this.logDriver);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TaskSpec#logDriver() logDriver} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for logDriver (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTaskSpec withLogDriver(@Nullable Driver value) {
    if (this.logDriver == value) return this;
    return new ImmutableTaskSpec(this.containerSpec, this.resources, this.restartPolicy, this.placement, this.networks, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableTaskSpec} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableTaskSpec
        && equalTo((ImmutableTaskSpec) another);
  }

  private boolean equalTo(ImmutableTaskSpec another) {
    return Objects.equals(containerSpec, another.containerSpec)
        && Objects.equals(resources, another.resources)
        && Objects.equals(restartPolicy, another.restartPolicy)
        && Objects.equals(placement, another.placement)
        && Objects.equals(networks, another.networks)
        && Objects.equals(logDriver, another.logDriver);
  }

  /**
   * Computes a hash code from attributes: {@code containerSpec}, {@code resources}, {@code restartPolicy}, {@code placement}, {@code networks}, {@code logDriver}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(containerSpec);
    h += (h << 5) + Objects.hashCode(resources);
    h += (h << 5) + Objects.hashCode(restartPolicy);
    h += (h << 5) + Objects.hashCode(placement);
    h += (h << 5) + Objects.hashCode(networks);
    h += (h << 5) + Objects.hashCode(logDriver);
    return h;
  }

  /**
   * Prints the immutable value {@code TaskSpec} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "TaskSpec{"
        + "containerSpec=" + containerSpec
        + ", resources=" + resources
        + ", restartPolicy=" + restartPolicy
        + ", placement=" + placement
        + ", networks=" + networks
        + ", logDriver=" + logDriver
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link TaskSpec} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable TaskSpec instance
   */
  public static ImmutableTaskSpec copyOf(TaskSpec instance) {
    if (instance instanceof ImmutableTaskSpec) {
      return (ImmutableTaskSpec) instance;
    }
    return ImmutableTaskSpec.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableTaskSpec ImmutableTaskSpec}.
   * <pre>
   * ImmutableTaskSpec.builder()
   *    .containerSpec(org.mandas.docker.client.messages.swarm.ContainerSpec | null) // nullable {@link TaskSpec#containerSpec() containerSpec}
   *    .resources(org.mandas.docker.client.messages.swarm.ResourceRequirements | null) // nullable {@link TaskSpec#resources() resources}
   *    .restartPolicy(org.mandas.docker.client.messages.swarm.RestartPolicy | null) // nullable {@link TaskSpec#restartPolicy() restartPolicy}
   *    .placement(org.mandas.docker.client.messages.swarm.Placement | null) // nullable {@link TaskSpec#placement() placement}
   *    .networks(List&amp;lt;org.mandas.docker.client.messages.swarm.NetworkAttachmentConfig&amp;gt; | null) // nullable {@link TaskSpec#networks() networks}
   *    .logDriver(org.mandas.docker.client.messages.swarm.Driver | null) // nullable {@link TaskSpec#logDriver() logDriver}
   *    .build();
   * </pre>
   * @return A new ImmutableTaskSpec builder
   */
  public static ImmutableTaskSpec.Builder builder() {
    return new ImmutableTaskSpec.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableTaskSpec ImmutableTaskSpec}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements TaskSpec.Builder {
    private ContainerSpec containerSpec;
    private ResourceRequirements resources;
    private RestartPolicy restartPolicy;
    private Placement placement;
    private List<NetworkAttachmentConfig> networks = null;
    private Driver logDriver;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code TaskSpec} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(TaskSpec instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable ContainerSpec containerSpecValue = instance.containerSpec();
      if (containerSpecValue != null) {
        containerSpec(containerSpecValue);
      }
      @Nullable ResourceRequirements resourcesValue = instance.resources();
      if (resourcesValue != null) {
        resources(resourcesValue);
      }
      @Nullable RestartPolicy restartPolicyValue = instance.restartPolicy();
      if (restartPolicyValue != null) {
        restartPolicy(restartPolicyValue);
      }
      @Nullable Placement placementValue = instance.placement();
      if (placementValue != null) {
        placement(placementValue);
      }
      @Nullable List<NetworkAttachmentConfig> networksValue = instance.networks();
      if (networksValue != null) {
        addAllNetworks(networksValue);
      }
      @Nullable Driver logDriverValue = instance.logDriver();
      if (logDriverValue != null) {
        logDriver(logDriverValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link TaskSpec#containerSpec() containerSpec} attribute.
     * @param containerSpec The value for containerSpec (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ContainerSpec")
    public final Builder containerSpec(@Nullable ContainerSpec containerSpec) {
      this.containerSpec = containerSpec;
      return this;
    }

    /**
     * Initializes the value for the {@link TaskSpec#resources() resources} attribute.
     * @param resources The value for resources (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Resources")
    public final Builder resources(@Nullable ResourceRequirements resources) {
      this.resources = resources;
      return this;
    }

    /**
     * Initializes the value for the {@link TaskSpec#restartPolicy() restartPolicy} attribute.
     * @param restartPolicy The value for restartPolicy (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("RestartPolicy")
    public final Builder restartPolicy(@Nullable RestartPolicy restartPolicy) {
      this.restartPolicy = restartPolicy;
      return this;
    }

    /**
     * Initializes the value for the {@link TaskSpec#placement() placement} attribute.
     * @param placement The value for placement (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Placement")
    public final Builder placement(@Nullable Placement placement) {
      this.placement = placement;
      return this;
    }

    /**
     * Adds one element to {@link TaskSpec#networks() networks} list.
     * @param element A networks element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder network(NetworkAttachmentConfig element) {
      if (this.networks == null) {
        this.networks = new ArrayList<NetworkAttachmentConfig>();
      }
      this.networks.add(Objects.requireNonNull(element, "networks element"));
      return this;
    }

    /**
     * Adds elements to {@link TaskSpec#networks() networks} list.
     * @param elements An array of networks elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder networks(NetworkAttachmentConfig... elements) {
      if (this.networks == null) {
        this.networks = new ArrayList<NetworkAttachmentConfig>();
      }
      for (NetworkAttachmentConfig element : elements) {
        this.networks.add(Objects.requireNonNull(element, "networks element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link TaskSpec#networks() networks} list.
     * @param elements An iterable of networks elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Networks")
    public final Builder networks(@Nullable Iterable<? extends NetworkAttachmentConfig> elements) {
      if (elements == null) {
        this.networks = null;
        return this;
      }
      this.networks = new ArrayList<NetworkAttachmentConfig>();
      return addAllNetworks(elements);
    }

    /**
     * Adds elements to {@link TaskSpec#networks() networks} list.
     * @param elements An iterable of networks elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllNetworks(Iterable<? extends NetworkAttachmentConfig> elements) {
      Objects.requireNonNull(elements, "networks element");
      if (this.networks == null) {
        this.networks = new ArrayList<NetworkAttachmentConfig>();
      }
      for (NetworkAttachmentConfig element : elements) {
        this.networks.add(Objects.requireNonNull(element, "networks element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link TaskSpec#logDriver() logDriver} attribute.
     * @param logDriver The value for logDriver (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("LogDriver")
    public final Builder logDriver(@Nullable Driver logDriver) {
      this.logDriver = logDriver;
      return this;
    }

    /**
     * Builds a new {@link ImmutableTaskSpec ImmutableTaskSpec}.
     * @return An immutable instance of TaskSpec
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableTaskSpec build() {
      return new ImmutableTaskSpec(
          containerSpec,
          resources,
          restartPolicy,
          placement,
          networks == null ? null : createUnmodifiableList(true, networks),
          logDriver);
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
