package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Secret}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableSecret.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableSecret implements Secret {
  private final String id;
  private final Version version;
  private final Date createdAt;
  private final Date updatedAt;
  private final SecretSpec secretSpec;

  private ImmutableSecret(
      String id,
      Version version,
      Date createdAt,
      Date updatedAt,
      SecretSpec secretSpec) {
    this.id = id;
    this.version = version;
    this.createdAt = createdAt;
    this.updatedAt = updatedAt;
    this.secretSpec = secretSpec;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("ID")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code version} attribute
   */
  @JsonProperty("Version")
  @Override
  public Version version() {
    return version;
  }

  /**
   * @return The value of the {@code createdAt} attribute
   */
  @JsonProperty("CreatedAt")
  @Override
  public Date createdAt() {
    return createdAt;
  }

  /**
   * @return The value of the {@code updatedAt} attribute
   */
  @JsonProperty("UpdatedAt")
  @Override
  public Date updatedAt() {
    return updatedAt;
  }

  /**
   * @return The value of the {@code secretSpec} attribute
   */
  @JsonProperty("Spec")
  @Override
  public SecretSpec secretSpec() {
    return secretSpec;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Secret#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSecret withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableSecret(newValue, this.version, this.createdAt, this.updatedAt, this.secretSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Secret#version() version} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for version
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSecret withVersion(Version value) {
    if (this.version == value) return this;
    Version newValue = Objects.requireNonNull(value, "version");
    return new ImmutableSecret(this.id, newValue, this.createdAt, this.updatedAt, this.secretSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Secret#createdAt() createdAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for createdAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSecret withCreatedAt(Date value) {
    if (this.createdAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "createdAt");
    return new ImmutableSecret(this.id, this.version, newValue, this.updatedAt, this.secretSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Secret#updatedAt() updatedAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updatedAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSecret withUpdatedAt(Date value) {
    if (this.updatedAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "updatedAt");
    return new ImmutableSecret(this.id, this.version, this.createdAt, newValue, this.secretSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Secret#secretSpec() secretSpec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for secretSpec
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableSecret withSecretSpec(SecretSpec value) {
    if (this.secretSpec == value) return this;
    SecretSpec newValue = Objects.requireNonNull(value, "secretSpec");
    return new ImmutableSecret(this.id, this.version, this.createdAt, this.updatedAt, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableSecret} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableSecret
        && equalTo((ImmutableSecret) another);
  }

  private boolean equalTo(ImmutableSecret another) {
    return id.equals(another.id)
        && version.equals(another.version)
        && createdAt.equals(another.createdAt)
        && updatedAt.equals(another.updatedAt)
        && secretSpec.equals(another.secretSpec);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code version}, {@code createdAt}, {@code updatedAt}, {@code secretSpec}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + version.hashCode();
    h += (h << 5) + createdAt.hashCode();
    h += (h << 5) + updatedAt.hashCode();
    h += (h << 5) + secretSpec.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Secret} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Secret{"
        + "id=" + id
        + ", version=" + version
        + ", createdAt=" + createdAt
        + ", updatedAt=" + updatedAt
        + ", secretSpec=" + secretSpec
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Secret} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Secret instance
   */
  public static ImmutableSecret copyOf(Secret instance) {
    if (instance instanceof ImmutableSecret) {
      return (ImmutableSecret) instance;
    }
    return ImmutableSecret.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableSecret ImmutableSecret}.
   * <pre>
   * ImmutableSecret.builder()
   *    .id(String) // required {@link Secret#id() id}
   *    .version(org.mandas.docker.client.messages.swarm.Version) // required {@link Secret#version() version}
   *    .createdAt(Date) // required {@link Secret#createdAt() createdAt}
   *    .updatedAt(Date) // required {@link Secret#updatedAt() updatedAt}
   *    .secretSpec(org.mandas.docker.client.messages.swarm.SecretSpec) // required {@link Secret#secretSpec() secretSpec}
   *    .build();
   * </pre>
   * @return A new ImmutableSecret builder
   */
  public static ImmutableSecret.Builder builder() {
    return new ImmutableSecret.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableSecret ImmutableSecret}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private static final long INIT_BIT_VERSION = 0x2L;
    private static final long INIT_BIT_CREATED_AT = 0x4L;
    private static final long INIT_BIT_UPDATED_AT = 0x8L;
    private static final long INIT_BIT_SECRET_SPEC = 0x10L;
    private long initBits = 0x1fL;

    private String id;
    private Version version;
    private Date createdAt;
    private Date updatedAt;
    private SecretSpec secretSpec;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Secret} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Secret instance) {
      Objects.requireNonNull(instance, "instance");
      id(instance.id());
      version(instance.version());
      createdAt(instance.createdAt());
      updatedAt(instance.updatedAt());
      secretSpec(instance.secretSpec());
      return this;
    }

    /**
     * Initializes the value for the {@link Secret#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ID")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link Secret#version() version} attribute.
     * @param version The value for version 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Version")
    public final Builder version(Version version) {
      this.version = Objects.requireNonNull(version, "version");
      initBits &= ~INIT_BIT_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Secret#createdAt() createdAt} attribute.
     * @param createdAt The value for createdAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CreatedAt")
    public final Builder createdAt(Date createdAt) {
      this.createdAt = Objects.requireNonNull(createdAt, "createdAt");
      initBits &= ~INIT_BIT_CREATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Secret#updatedAt() updatedAt} attribute.
     * @param updatedAt The value for updatedAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UpdatedAt")
    public final Builder updatedAt(Date updatedAt) {
      this.updatedAt = Objects.requireNonNull(updatedAt, "updatedAt");
      initBits &= ~INIT_BIT_UPDATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Secret#secretSpec() secretSpec} attribute.
     * @param secretSpec The value for secretSpec 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Spec")
    public final Builder secretSpec(SecretSpec secretSpec) {
      this.secretSpec = Objects.requireNonNull(secretSpec, "secretSpec");
      initBits &= ~INIT_BIT_SECRET_SPEC;
      return this;
    }

    /**
     * Builds a new {@link ImmutableSecret ImmutableSecret}.
     * @return An immutable instance of Secret
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableSecret build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableSecret(id, version, createdAt, updatedAt, secretSpec);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_VERSION) != 0) attributes.add("version");
      if ((initBits & INIT_BIT_CREATED_AT) != 0) attributes.add("createdAt");
      if ((initBits & INIT_BIT_UPDATED_AT) != 0) attributes.add("updatedAt");
      if ((initBits & INIT_BIT_SECRET_SPEC) != 0) attributes.add("secretSpec");
      return "Cannot build Secret, some of required attributes are not set " + attributes;
    }
  }

  /**
   * Immutable implementation of {@link Secret.Criteria}.
   * <p>
   * Use the builder to create immutable instances:
   * {@code ImmutableSecret.Criteria.builder()}.
   */
  static final class Criteria implements Secret.Criteria {
    private final @Nullable String id;
    private final @Nullable String label;
    private final @Nullable String name;

    private Criteria(
        @Nullable String id,
        @Nullable String label,
        @Nullable String name) {
      this.id = id;
      this.label = label;
      this.name = name;
    }

    /**
     * @return The value of the {@code id} attribute
     */
    @Override
    public @Nullable String id() {
      return id;
    }

    /**
     * @return The value of the {@code label} attribute
     */
    @Override
    public @Nullable String label() {
      return label;
    }

    /**
     * @return The value of the {@code name} attribute
     */
    @Override
    public @Nullable String name() {
      return name;
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Secret.Criteria#id() id} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for id (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableSecret.Criteria withId(@Nullable String value) {
      if (Objects.equals(this.id, value)) return this;
      return new ImmutableSecret.Criteria(value, this.label, this.name);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Secret.Criteria#label() label} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for label (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableSecret.Criteria withLabel(@Nullable String value) {
      if (Objects.equals(this.label, value)) return this;
      return new ImmutableSecret.Criteria(this.id, value, this.name);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Secret.Criteria#name() name} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for name (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableSecret.Criteria withName(@Nullable String value) {
      if (Objects.equals(this.name, value)) return this;
      return new ImmutableSecret.Criteria(this.id, this.label, value);
    }

    /**
     * This instance is equal to all instances of {@code Criteria} that have equal attribute values.
     * @return {@code true} if {@code this} is equal to {@code another} instance
     */
    @Override
    public boolean equals(Object another) {
      if (this == another) return true;
      return another instanceof ImmutableSecret.Criteria
          && equalTo((ImmutableSecret.Criteria) another);
    }

    private boolean equalTo(ImmutableSecret.Criteria another) {
      return Objects.equals(id, another.id)
          && Objects.equals(label, another.label)
          && Objects.equals(name, another.name);
    }

    /**
     * Computes a hash code from attributes: {@code id}, {@code label}, {@code name}.
     * @return hashCode value
     */
    @Override
    public int hashCode() {
      int h = 5381;
      h += (h << 5) + Objects.hashCode(id);
      h += (h << 5) + Objects.hashCode(label);
      h += (h << 5) + Objects.hashCode(name);
      return h;
    }

    /**
     * Prints the immutable value {@code Criteria} with attribute values.
     * @return A string representation of the value
     */
    @Override
    public String toString() {
      return "Criteria{"
          + "id=" + id
          + ", label=" + label
          + ", name=" + name
          + "}";
    }

    /**
     * Creates an immutable copy of a {@link Secret.Criteria} value.
     * Uses accessors to get values to initialize the new immutable instance.
     * If an instance is already immutable, it is returned as is.
     * @param instance The instance to copy
     * @return A copied immutable Criteria instance
     */
    public static ImmutableSecret.Criteria copyOf(Secret.Criteria instance) {
      if (instance instanceof ImmutableSecret.Criteria) {
        return (ImmutableSecret.Criteria) instance;
      }
      return ImmutableSecret.Criteria.builder()
          .from(instance)
          .build();
    }

    /**
     * Creates a builder for {@link ImmutableSecret.Criteria Criteria}.
     * <pre>
     * ImmutableSecret.Criteria.builder()
     *    .id(String | null) // nullable {@link Secret.Criteria#id() id}
     *    .label(String | null) // nullable {@link Secret.Criteria#label() label}
     *    .name(String | null) // nullable {@link Secret.Criteria#name() name}
     *    .build();
     * </pre>
     * @return A new Criteria builder
     */
    public static ImmutableSecret.Criteria.Builder builder() {
      return new ImmutableSecret.Criteria.Builder();
    }

    /**
     * Builds instances of type {@link ImmutableSecret.Criteria Criteria}.
     * Initialize attributes and then invoke the {@link #build()} method to create an
     * immutable instance.
     * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
     * but instead used immediately to create instances.</em>
     */
    static final class Builder implements Secret.Criteria.Builder {
      private String id;
      private String label;
      private String name;

      private Builder() {
      }

      /**
       * Fill a builder with attribute values from the provided {@code Criteria} instance.
       * Regular attribute values will be replaced with those from the given instance.
       * Absent optional values will not replace present values.
       * @param instance The instance from which to copy values
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder from(Secret.Criteria instance) {
        Objects.requireNonNull(instance, "instance");
        @Nullable String idValue = instance.id();
        if (idValue != null) {
          id(idValue);
        }
        @Nullable String labelValue = instance.label();
        if (labelValue != null) {
          label(labelValue);
        }
        @Nullable String nameValue = instance.name();
        if (nameValue != null) {
          name(nameValue);
        }
        return this;
      }

      /**
       * Initializes the value for the {@link Secret.Criteria#id() id} attribute.
       * @param id The value for id (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder id(@Nullable String id) {
        this.id = id;
        return this;
      }

      /**
       * Initializes the value for the {@link Secret.Criteria#label() label} attribute.
       * @param label The value for label (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder label(@Nullable String label) {
        this.label = label;
        return this;
      }

      /**
       * Initializes the value for the {@link Secret.Criteria#name() name} attribute.
       * @param name The value for name (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder name(@Nullable String name) {
        this.name = name;
        return this;
      }

      /**
       * Builds a new {@link ImmutableSecret.Criteria Criteria}.
       * @return An immutable instance of Criteria
       * @throws java.lang.IllegalStateException if any required attributes are missing
       */
      public ImmutableSecret.Criteria build() {
        return new ImmutableSecret.Criteria(id, label, name);
      }
    }
  }
}
