package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Node}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNode.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNode implements Node {
  private final String id;
  private final Version version;
  private final Date createdAt;
  private final Date updatedAt;
  private final NodeSpec spec;
  private final NodeDescription description;
  private final NodeStatus status;
  private final @Nullable ManagerStatus managerStatus;

  private ImmutableNode(
      String id,
      Version version,
      Date createdAt,
      Date updatedAt,
      NodeSpec spec,
      NodeDescription description,
      NodeStatus status,
      @Nullable ManagerStatus managerStatus) {
    this.id = id;
    this.version = version;
    this.createdAt = createdAt;
    this.updatedAt = updatedAt;
    this.spec = spec;
    this.description = description;
    this.status = status;
    this.managerStatus = managerStatus;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("ID")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code version} attribute
   */
  @JsonProperty("Version")
  @Override
  public Version version() {
    return version;
  }

  /**
   * @return The value of the {@code createdAt} attribute
   */
  @JsonProperty("CreatedAt")
  @Override
  public Date createdAt() {
    return createdAt;
  }

  /**
   * @return The value of the {@code updatedAt} attribute
   */
  @JsonProperty("UpdatedAt")
  @Override
  public Date updatedAt() {
    return updatedAt;
  }

  /**
   * @return The value of the {@code spec} attribute
   */
  @JsonProperty("Spec")
  @Override
  public NodeSpec spec() {
    return spec;
  }

  /**
   * @return The value of the {@code description} attribute
   */
  @JsonProperty("Description")
  @Override
  public NodeDescription description() {
    return description;
  }

  /**
   * @return The value of the {@code status} attribute
   */
  @JsonProperty("Status")
  @Override
  public NodeStatus status() {
    return status;
  }

  /**
   * @return The value of the {@code managerStatus} attribute
   */
  @JsonProperty("ManagerStatus")
  @Override
  public @Nullable ManagerStatus managerStatus() {
    return managerStatus;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableNode(
        newValue,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.spec,
        this.description,
        this.status,
        this.managerStatus);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#version() version} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for version
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withVersion(Version value) {
    if (this.version == value) return this;
    Version newValue = Objects.requireNonNull(value, "version");
    return new ImmutableNode(
        this.id,
        newValue,
        this.createdAt,
        this.updatedAt,
        this.spec,
        this.description,
        this.status,
        this.managerStatus);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#createdAt() createdAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for createdAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withCreatedAt(Date value) {
    if (this.createdAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "createdAt");
    return new ImmutableNode(
        this.id,
        this.version,
        newValue,
        this.updatedAt,
        this.spec,
        this.description,
        this.status,
        this.managerStatus);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#updatedAt() updatedAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updatedAt
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withUpdatedAt(Date value) {
    if (this.updatedAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "updatedAt");
    return new ImmutableNode(
        this.id,
        this.version,
        this.createdAt,
        newValue,
        this.spec,
        this.description,
        this.status,
        this.managerStatus);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#spec() spec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for spec
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withSpec(NodeSpec value) {
    if (this.spec == value) return this;
    NodeSpec newValue = Objects.requireNonNull(value, "spec");
    return new ImmutableNode(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        newValue,
        this.description,
        this.status,
        this.managerStatus);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#description() description} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for description
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withDescription(NodeDescription value) {
    if (this.description == value) return this;
    NodeDescription newValue = Objects.requireNonNull(value, "description");
    return new ImmutableNode(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.spec,
        newValue,
        this.status,
        this.managerStatus);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#status() status} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for status
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withStatus(NodeStatus value) {
    if (this.status == value) return this;
    NodeStatus newValue = Objects.requireNonNull(value, "status");
    return new ImmutableNode(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.spec,
        this.description,
        newValue,
        this.managerStatus);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Node#managerStatus() managerStatus} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for managerStatus (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNode withManagerStatus(@Nullable ManagerStatus value) {
    if (this.managerStatus == value) return this;
    return new ImmutableNode(
        this.id,
        this.version,
        this.createdAt,
        this.updatedAt,
        this.spec,
        this.description,
        this.status,
        value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNode} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNode
        && equalTo((ImmutableNode) another);
  }

  private boolean equalTo(ImmutableNode another) {
    return id.equals(another.id)
        && version.equals(another.version)
        && createdAt.equals(another.createdAt)
        && updatedAt.equals(another.updatedAt)
        && spec.equals(another.spec)
        && description.equals(another.description)
        && status.equals(another.status)
        && Objects.equals(managerStatus, another.managerStatus);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code version}, {@code createdAt}, {@code updatedAt}, {@code spec}, {@code description}, {@code status}, {@code managerStatus}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + version.hashCode();
    h += (h << 5) + createdAt.hashCode();
    h += (h << 5) + updatedAt.hashCode();
    h += (h << 5) + spec.hashCode();
    h += (h << 5) + description.hashCode();
    h += (h << 5) + status.hashCode();
    h += (h << 5) + Objects.hashCode(managerStatus);
    return h;
  }

  /**
   * Prints the immutable value {@code Node} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Node{"
        + "id=" + id
        + ", version=" + version
        + ", createdAt=" + createdAt
        + ", updatedAt=" + updatedAt
        + ", spec=" + spec
        + ", description=" + description
        + ", status=" + status
        + ", managerStatus=" + managerStatus
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Node} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Node instance
   */
  public static ImmutableNode copyOf(Node instance) {
    if (instance instanceof ImmutableNode) {
      return (ImmutableNode) instance;
    }
    return ImmutableNode.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNode ImmutableNode}.
   * <pre>
   * ImmutableNode.builder()
   *    .id(String) // required {@link Node#id() id}
   *    .version(org.mandas.docker.client.messages.swarm.Version) // required {@link Node#version() version}
   *    .createdAt(Date) // required {@link Node#createdAt() createdAt}
   *    .updatedAt(Date) // required {@link Node#updatedAt() updatedAt}
   *    .spec(org.mandas.docker.client.messages.swarm.NodeSpec) // required {@link Node#spec() spec}
   *    .description(org.mandas.docker.client.messages.swarm.NodeDescription) // required {@link Node#description() description}
   *    .status(org.mandas.docker.client.messages.swarm.NodeStatus) // required {@link Node#status() status}
   *    .managerStatus(org.mandas.docker.client.messages.swarm.ManagerStatus | null) // nullable {@link Node#managerStatus() managerStatus}
   *    .build();
   * </pre>
   * @return A new ImmutableNode builder
   */
  public static ImmutableNode.Builder builder() {
    return new ImmutableNode.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNode ImmutableNode}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private static final long INIT_BIT_VERSION = 0x2L;
    private static final long INIT_BIT_CREATED_AT = 0x4L;
    private static final long INIT_BIT_UPDATED_AT = 0x8L;
    private static final long INIT_BIT_SPEC = 0x10L;
    private static final long INIT_BIT_DESCRIPTION = 0x20L;
    private static final long INIT_BIT_STATUS = 0x40L;
    private long initBits = 0x7fL;

    private String id;
    private Version version;
    private Date createdAt;
    private Date updatedAt;
    private NodeSpec spec;
    private NodeDescription description;
    private NodeStatus status;
    private ManagerStatus managerStatus;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Node} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Node instance) {
      Objects.requireNonNull(instance, "instance");
      id(instance.id());
      version(instance.version());
      createdAt(instance.createdAt());
      updatedAt(instance.updatedAt());
      spec(instance.spec());
      description(instance.description());
      status(instance.status());
      @Nullable ManagerStatus managerStatusValue = instance.managerStatus();
      if (managerStatusValue != null) {
        managerStatus(managerStatusValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Node#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ID")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link Node#version() version} attribute.
     * @param version The value for version 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Version")
    public final Builder version(Version version) {
      this.version = Objects.requireNonNull(version, "version");
      initBits &= ~INIT_BIT_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Node#createdAt() createdAt} attribute.
     * @param createdAt The value for createdAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CreatedAt")
    public final Builder createdAt(Date createdAt) {
      this.createdAt = Objects.requireNonNull(createdAt, "createdAt");
      initBits &= ~INIT_BIT_CREATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Node#updatedAt() updatedAt} attribute.
     * @param updatedAt The value for updatedAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UpdatedAt")
    public final Builder updatedAt(Date updatedAt) {
      this.updatedAt = Objects.requireNonNull(updatedAt, "updatedAt");
      initBits &= ~INIT_BIT_UPDATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Node#spec() spec} attribute.
     * @param spec The value for spec 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Spec")
    public final Builder spec(NodeSpec spec) {
      this.spec = Objects.requireNonNull(spec, "spec");
      initBits &= ~INIT_BIT_SPEC;
      return this;
    }

    /**
     * Initializes the value for the {@link Node#description() description} attribute.
     * @param description The value for description 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Description")
    public final Builder description(NodeDescription description) {
      this.description = Objects.requireNonNull(description, "description");
      initBits &= ~INIT_BIT_DESCRIPTION;
      return this;
    }

    /**
     * Initializes the value for the {@link Node#status() status} attribute.
     * @param status The value for status 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Status")
    public final Builder status(NodeStatus status) {
      this.status = Objects.requireNonNull(status, "status");
      initBits &= ~INIT_BIT_STATUS;
      return this;
    }

    /**
     * Initializes the value for the {@link Node#managerStatus() managerStatus} attribute.
     * @param managerStatus The value for managerStatus (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ManagerStatus")
    public final Builder managerStatus(@Nullable ManagerStatus managerStatus) {
      this.managerStatus = managerStatus;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNode ImmutableNode}.
     * @return An immutable instance of Node
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNode build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNode(id, version, createdAt, updatedAt, spec, description, status, managerStatus);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_VERSION) != 0) attributes.add("version");
      if ((initBits & INIT_BIT_CREATED_AT) != 0) attributes.add("createdAt");
      if ((initBits & INIT_BIT_UPDATED_AT) != 0) attributes.add("updatedAt");
      if ((initBits & INIT_BIT_SPEC) != 0) attributes.add("spec");
      if ((initBits & INIT_BIT_DESCRIPTION) != 0) attributes.add("description");
      if ((initBits & INIT_BIT_STATUS) != 0) attributes.add("status");
      return "Cannot build Node, some of required attributes are not set " + attributes;
    }
  }

  /**
   * Immutable implementation of {@link Node.Criteria}.
   * <p>
   * Use the builder to create immutable instances:
   * {@code ImmutableNode.Criteria.builder()}.
   */
  static final class Criteria implements Node.Criteria {
    private final @Nullable String nodeId;
    private final @Nullable String label;
    private final @Nullable String membership;
    private final @Nullable String nodeName;
    private final @Nullable String nodeRole;

    private Criteria(
        @Nullable String nodeId,
        @Nullable String label,
        @Nullable String membership,
        @Nullable String nodeName,
        @Nullable String nodeRole) {
      this.nodeId = nodeId;
      this.label = label;
      this.membership = membership;
      this.nodeName = nodeName;
      this.nodeRole = nodeRole;
    }

    /**
     * @return The value of the {@code nodeId} attribute
     */
    @JsonProperty("nodeId")
    @Override
    public @Nullable String nodeId() {
      return nodeId;
    }

    /**
     * @return The value of the {@code label} attribute
     */
    @JsonProperty("label")
    @Override
    public @Nullable String label() {
      return label;
    }

    /**
     * @return The value of the {@code membership} attribute
     */
    @JsonProperty("membership")
    @Override
    public @Nullable String membership() {
      return membership;
    }

    /**
     * @return The value of the {@code nodeName} attribute
     */
    @JsonProperty("nodeName")
    @Override
    public @Nullable String nodeName() {
      return nodeName;
    }

    /**
     * @return The value of the {@code nodeRole} attribute
     */
    @JsonProperty("nodeRole")
    @Override
    public @Nullable String nodeRole() {
      return nodeRole;
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Node.Criteria#nodeId() nodeId} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for nodeId (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableNode.Criteria withNodeId(@Nullable String value) {
      if (Objects.equals(this.nodeId, value)) return this;
      return new ImmutableNode.Criteria(value, this.label, this.membership, this.nodeName, this.nodeRole);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Node.Criteria#label() label} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for label (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableNode.Criteria withLabel(@Nullable String value) {
      if (Objects.equals(this.label, value)) return this;
      return new ImmutableNode.Criteria(this.nodeId, value, this.membership, this.nodeName, this.nodeRole);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Node.Criteria#membership() membership} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for membership (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableNode.Criteria withMembership(@Nullable String value) {
      if (Objects.equals(this.membership, value)) return this;
      return new ImmutableNode.Criteria(this.nodeId, this.label, value, this.nodeName, this.nodeRole);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Node.Criteria#nodeName() nodeName} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for nodeName (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableNode.Criteria withNodeName(@Nullable String value) {
      if (Objects.equals(this.nodeName, value)) return this;
      return new ImmutableNode.Criteria(this.nodeId, this.label, this.membership, value, this.nodeRole);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Node.Criteria#nodeRole() nodeRole} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for nodeRole (can be {@code null})
     * @return A modified copy of the {@code this} object
     */
    public final ImmutableNode.Criteria withNodeRole(@Nullable String value) {
      if (Objects.equals(this.nodeRole, value)) return this;
      return new ImmutableNode.Criteria(this.nodeId, this.label, this.membership, this.nodeName, value);
    }

    /**
     * This instance is equal to all instances of {@code Criteria} that have equal attribute values.
     * @return {@code true} if {@code this} is equal to {@code another} instance
     */
    @Override
    public boolean equals(Object another) {
      if (this == another) return true;
      return another instanceof ImmutableNode.Criteria
          && equalTo((ImmutableNode.Criteria) another);
    }

    private boolean equalTo(ImmutableNode.Criteria another) {
      return Objects.equals(nodeId, another.nodeId)
          && Objects.equals(label, another.label)
          && Objects.equals(membership, another.membership)
          && Objects.equals(nodeName, another.nodeName)
          && Objects.equals(nodeRole, another.nodeRole);
    }

    /**
     * Computes a hash code from attributes: {@code nodeId}, {@code label}, {@code membership}, {@code nodeName}, {@code nodeRole}.
     * @return hashCode value
     */
    @Override
    public int hashCode() {
      int h = 5381;
      h += (h << 5) + Objects.hashCode(nodeId);
      h += (h << 5) + Objects.hashCode(label);
      h += (h << 5) + Objects.hashCode(membership);
      h += (h << 5) + Objects.hashCode(nodeName);
      h += (h << 5) + Objects.hashCode(nodeRole);
      return h;
    }

    /**
     * Prints the immutable value {@code Criteria} with attribute values.
     * @return A string representation of the value
     */
    @Override
    public String toString() {
      return "Criteria{"
          + "nodeId=" + nodeId
          + ", label=" + label
          + ", membership=" + membership
          + ", nodeName=" + nodeName
          + ", nodeRole=" + nodeRole
          + "}";
    }

    /**
     * Creates an immutable copy of a {@link Node.Criteria} value.
     * Uses accessors to get values to initialize the new immutable instance.
     * If an instance is already immutable, it is returned as is.
     * @param instance The instance to copy
     * @return A copied immutable Criteria instance
     */
    public static ImmutableNode.Criteria copyOf(Node.Criteria instance) {
      if (instance instanceof ImmutableNode.Criteria) {
        return (ImmutableNode.Criteria) instance;
      }
      return ImmutableNode.Criteria.builder()
          .from(instance)
          .build();
    }

    /**
     * Creates a builder for {@link ImmutableNode.Criteria Criteria}.
     * <pre>
     * ImmutableNode.Criteria.builder()
     *    .nodeId(String | null) // nullable {@link Node.Criteria#nodeId() nodeId}
     *    .label(String | null) // nullable {@link Node.Criteria#label() label}
     *    .membership(String | null) // nullable {@link Node.Criteria#membership() membership}
     *    .nodeName(String | null) // nullable {@link Node.Criteria#nodeName() nodeName}
     *    .nodeRole(String | null) // nullable {@link Node.Criteria#nodeRole() nodeRole}
     *    .build();
     * </pre>
     * @return A new Criteria builder
     */
    public static ImmutableNode.Criteria.Builder builder() {
      return new ImmutableNode.Criteria.Builder();
    }

    /**
     * Builds instances of type {@link ImmutableNode.Criteria Criteria}.
     * Initialize attributes and then invoke the {@link #build()} method to create an
     * immutable instance.
     * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
     * but instead used immediately to create instances.</em>
     */
    static final class Builder implements Node.Criteria.Builder {
      private String nodeId;
      private String label;
      private String membership;
      private String nodeName;
      private String nodeRole;

      private Builder() {
      }

      /**
       * Fill a builder with attribute values from the provided {@code Criteria} instance.
       * Regular attribute values will be replaced with those from the given instance.
       * Absent optional values will not replace present values.
       * @param instance The instance from which to copy values
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder from(Node.Criteria instance) {
        Objects.requireNonNull(instance, "instance");
        @Nullable String nodeIdValue = instance.nodeId();
        if (nodeIdValue != null) {
          nodeId(nodeIdValue);
        }
        @Nullable String labelValue = instance.label();
        if (labelValue != null) {
          label(labelValue);
        }
        @Nullable String membershipValue = instance.membership();
        if (membershipValue != null) {
          membership(membershipValue);
        }
        @Nullable String nodeNameValue = instance.nodeName();
        if (nodeNameValue != null) {
          nodeName(nodeNameValue);
        }
        @Nullable String nodeRoleValue = instance.nodeRole();
        if (nodeRoleValue != null) {
          nodeRole(nodeRoleValue);
        }
        return this;
      }

      /**
       * Initializes the value for the {@link Node.Criteria#nodeId() nodeId} attribute.
       * @param nodeId The value for nodeId (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("nodeId")
      public final Builder nodeId(@Nullable String nodeId) {
        this.nodeId = nodeId;
        return this;
      }

      /**
       * Initializes the value for the {@link Node.Criteria#label() label} attribute.
       * @param label The value for label (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("label")
      public final Builder label(@Nullable String label) {
        this.label = label;
        return this;
      }

      /**
       * Initializes the value for the {@link Node.Criteria#membership() membership} attribute.
       * @param membership The value for membership (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("membership")
      public final Builder membership(@Nullable String membership) {
        this.membership = membership;
        return this;
      }

      /**
       * Initializes the value for the {@link Node.Criteria#nodeName() nodeName} attribute.
       * @param nodeName The value for nodeName (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("nodeName")
      public final Builder nodeName(@Nullable String nodeName) {
        this.nodeName = nodeName;
        return this;
      }

      /**
       * Initializes the value for the {@link Node.Criteria#nodeRole() nodeRole} attribute.
       * @param nodeRole The value for nodeRole (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      @JsonProperty("nodeRole")
      public final Builder nodeRole(@Nullable String nodeRole) {
        this.nodeRole = nodeRole;
        return this;
      }

      /**
       * Builds a new {@link ImmutableNode.Criteria Criteria}.
       * @return An immutable instance of Criteria
       * @throws java.lang.IllegalStateException if any required attributes are missing
       */
      public ImmutableNode.Criteria build() {
        return new ImmutableNode.Criteria(nodeId, label, membership, nodeName, nodeRole);
      }
    }
  }
}
