/* Copyright (c) 2018 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util.primitive;

import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.HashSet;
import java.util.NoSuchElementException;
import java.util.Random;
import java.util.Set;

import javax.annotation.Generated;

import org.junit.Assert;
import org.junit.Test;
import org.libj.util.CollectionUtil;

@SuppressWarnings("all")
@Generated(value="Autogenerated by OpenJAX CodeGen Template (0.2.5)", date="2020-05-23")
public class HashDoubleSetTest extends PrimitiveCollectionTest {
  private static final int INITIAL_CAPACITY = 64;

  private final HashDoubleSet testSet = new HashDoubleSet(INITIAL_CAPACITY);

  @Test
  public void initiallyContainsNoElements() {
    for (double i = 0; i < 100; ++i)
      assertFalse(testSet.contains(i));
  }

  @Test
  public void initiallyContainsNoBoxedElements() {
    for (double i = 0; i < 100; ++i)
      assertFalse(testSet.contains(Double.valueOf(i)));
  }

  @Test
  public void containsAddedElement() {
    assertTrue(testSet.add((double)1));
    assertTrue(testSet.contains((double)1));
  }

  @Test
  public void addingAnElementTwiceDoesNothing() {
    assertTrue(testSet.add((double)1));
    assertFalse(testSet.add((double)1));
  }

  @Test
  public void containsAddedBoxedElements() {
    assertTrue(testSet.add((double)1));
    assertTrue(testSet.add(Double.valueOf((double)2)));

    assertTrue(testSet.contains(Double.valueOf((double)1)));
    assertTrue(testSet.contains((double)2));
  }

  @Test
  public void removingAnElementFromAnEmptyListDoesNothing() {
    assertFalse(testSet.remove((double)0));
  }

  @Test
  public void removingAPresentElementRemovesIt() {
    assertTrue(testSet.add((double)1));
    assertTrue(testSet.remove((double)1));
    assertFalse(testSet.contains((double)1));
  }

  @Test
  public void sizeIsInitiallyZero() {
    assertEquals(0, testSet.size());
  }

  @Test
  public void sizeIncrementsWithNumberOfAddedElements() {
    addTwoElements(testSet);

    assertEquals(2, testSet.size());
  }

  @Test
  public void sizeContainsNumberOfNewElements() {
    testSet.add((double)1);
    testSet.add((double)1);

    assertEquals(1, testSet.size());
  }

  @Test
  public void iteratorsListElements() {
    addTwoElements(testSet);

    assertIteratorHasElements();
  }

  @Test
  public void iteratorsStartFromTheBeginningEveryTime() {
    iteratorsListElements();

    assertIteratorHasElements();
  }

  @Test
  public void iteratorsListElementsWithoutHasNext() {
    addTwoElements(testSet);

    assertIteratorHasElementsWithoutHasNext();
  }

  @Test
  public void iteratorsStartFromTheBeginningEveryTimeWithoutHasNext() {
    iteratorsListElementsWithoutHasNext();

    assertIteratorHasElementsWithoutHasNext();
  }

  @Test(expected = NoSuchElementException.class)
  public void iteratorsThrowNoSuchElementException() {
    addTwoElements(testSet);

    exhaustIterator();
  }

  @Test(expected = NoSuchElementException.class)
  public void iteratorsThrowNoSuchElementExceptionFromTheBeginningEveryTime() {
    addTwoElements(testSet);

    try {
      exhaustIterator();
    }
    catch (final NoSuchElementException ignore) {
    }

    exhaustIterator();
  }

  @Test
  public void iteratorHasNoElements() {
    assertFalse(testSet.iterator().hasNext());
  }

  @Test(expected = NoSuchElementException.class)
  public void iteratorThrowExceptionForEmptySet() {
    testSet.iterator().next();
  }

  @Test
  public void clearRemovesAllElementsOfTheSet() {
    addTwoElements(testSet);

    testSet.clear();

    assertEquals(0, testSet.size());
    assertFalse(testSet.contains((double)1));
    assertFalse(testSet.contains((double)101));
  }

  @Test
  public void differenceReturnsNullIfBothSetsEqual() {
    addTwoElements(testSet);

    final HashDoubleSet other = new HashDoubleSet(100);
    addTwoElements(other);

    other.removeAll(testSet);
    assertEquals(0, other.size());
  }

  @Test
  public void differenceReturnsSetDifference() {
    addTwoElements(testSet);

    final HashDoubleSet other = new HashDoubleSet(100);
    other.add((double)1);

    testSet.removeAll(other);
    assertTrue(testSet.contains((double)101));
  }

  @Test
  public void copiesOtherHashDoubleSet() {
    addTwoElements(testSet);

    final HashDoubleSet other = new HashDoubleSet(testSet);
    assertContainsElements(other);
  }

  @Test
  public void twoEmptySetsAreEqual() {
    Assert.assertEquals(testSet, new HashDoubleSet(100));
  }

  @Test
  public void setsWithTheSameValuesAreEqual() {
    final HashDoubleSet that = new HashDoubleSet(100);

    addTwoElements(testSet);
    addTwoElements(that);

    Assert.assertEquals(testSet, that);
  }

  @Test
  public void setsWithTheDifferentSizesAreNotEqual() {
    final HashDoubleSet that = new HashDoubleSet(100);

    addTwoElements(testSet);
    that.add((double)101);

    assertNotEquals(testSet, that);
  }

  @Test
  public void setsWithTheDifferentValuesAreNotEqual() {
    final HashDoubleSet that = new HashDoubleSet(100);

    addTwoElements(testSet);
    that.add((double)2);
    that.add((double)101);

    assertNotEquals(testSet, that);
  }

  @Test
  public void twoEmptySetsHaveTheSameHashcode() {
    assertEquals(testSet.hashCode(), new HashDoubleSet(100).hashCode());
  }

  @Test
  public void setsWithTheSameValuesHaveTheSameHashcode() {
    final HashDoubleSet other = new HashDoubleSet(100);

    addTwoElements(testSet);
    addTwoElements(other);

    assertEquals(testSet.hashCode(), other.hashCode());
  }

  @Test
  public void reducesSizeWhenElementRemoved() {
    addTwoElements(testSet);
    testSet.remove((double)101);

    assertEquals(1, testSet.size());
  }

  @Test(expected = NullPointerException.class)
  public void toArrayThrowsNullPointerExceptionForNullArgument() {
    final Double[] a = null;
    testSet.toArray(a);
  }

  @Test
  public void toArrayCopiesElementsDoubleoSufficientlySizedArray() {
    addTwoElements(testSet);
    final Double[] result = testSet.toArray(new Double[testSet.size()]);

    assertArrayContainingElements(result);
  }

  @Test
  public void toArrayCopiesElementsDoubleoNewArray() {
    addTwoElements(testSet);
    final Double[] result = testSet.toArray(new Double[testSet.size()]);

    assertArrayContainingElements(result);
  }

  @Test
  public void toArraySupportsEmptyCollection() {
    final Double[] result = testSet.toArray(new Double[testSet.size()]);

    Assert.assertArrayEquals(result, new Double[] {});
  }

  // Test case from usage bug.
  @Test
  public void chainCompactionShouldNotCauseElementsToBeMovedBeforeTheirHash() {
    final HashDoubleSet requiredFields = new HashDoubleSet(14);

    requiredFields.add((double)8);
    requiredFields.add((double)9);
    requiredFields.add((double)35);
    requiredFields.add((double)49);
    requiredFields.add((double)56);

    assertTrue("Failed to remove 8", requiredFields.remove((double)8));
    assertTrue("Failed to remove 9", requiredFields.remove((double)9));

    assertTrue(requiredFields.containsAll(Arrays.asList((double)35, (double)49, (double)56)));
  }

  @Test
  public void shouldResizeWhenItHitsCapacity() {
    for (double i = 0; i < 2 * INITIAL_CAPACITY - 1; ++i)
      assertTrue(testSet.add(i));

    for (double i = 0; i < 2 * INITIAL_CAPACITY - 1; ++i)
      assertTrue(testSet.contains(i));
  }

  @Test
  public void containsEmptySet() {
    assertTrue(testSet.containsAll(new HashDoubleSet(100)));
  }

  @Test
  public void containsSubset() {
    addTwoElements(testSet);

    final HashDoubleSet subset = new HashDoubleSet(100);
    subset.add((double)1);

    assertTrue(testSet.containsAll(subset));
  }

  @Test
  public void doesNotContainDisjointSet() {
    addTwoElements(testSet);

    final HashDoubleSet disjoint = new HashDoubleSet(100);
    disjoint.add((double)1);
    disjoint.add((double)102);

    assertFalse(testSet.containsAll(disjoint));
  }

  @Test
  public void doesNotContainSuperset() {
    addTwoElements(testSet);

    final HashDoubleSet superset = new HashDoubleSet(100);
    addTwoElements(superset);
    superset.add((double)15);

    assertFalse(testSet.containsAll(superset));
  }

  @Test
  public void addingEmptySetDoesNothing() {
    addTwoElements(testSet);

    assertFalse(testSet.addAll(new HashDoubleSet(100)));
    assertFalse(testSet.addAll(new HashSet<>()));
    assertContainsElements(testSet);
  }

  @Test
  public void containsValuesAddedFromDisjointSetPrimitive() {
    addTwoElements(testSet);

    final HashDoubleSet disjoint = new HashDoubleSet(100);

    disjoint.add((double)2);
    disjoint.add((double)102);

    assertTrue(testSet.addAll(disjoint));
    assertTrue(testSet.contains((double)1));
    assertTrue(testSet.contains((double)101));
    assertTrue(testSet.containsAll(disjoint));
  }

  @Test
  public void containsValuesAddedFromDisjointSet() {
    addTwoElements(testSet);

    final HashSet<Double> disjoint = new HashSet<>();

    disjoint.add((double)2);
    disjoint.add((double)102);

    assertTrue(testSet.addAll(disjoint));
    assertTrue(testSet.contains((double)1));
    assertTrue(testSet.contains((double)101));
    assertTrue(testSet.containsAll(disjoint));
  }

  @Test
  public void containsValuesAddedFromDoubleersectingSetPrimitive() {
    addTwoElements(testSet);

    final HashDoubleSet intersecting = new HashDoubleSet(100);
    intersecting.add((double)1);
    intersecting.add((double)102);

    assertTrue(testSet.addAll(intersecting));
    assertTrue(testSet.contains((double)1));
    assertTrue(testSet.contains((double)101));
    assertTrue(testSet.containsAll(intersecting));
  }

  @Test
  public void containsValuesAddedFromDoubleersectingSet() {
    addTwoElements(testSet);

    final HashSet<Double> intersecting = new HashSet<>();

    intersecting.add((double)1);
    intersecting.add((double)102);

    assertTrue(testSet.addAll(intersecting));
    assertTrue(testSet.contains((double)1));
    assertTrue(testSet.contains((double)101));
    assertTrue(testSet.containsAll(intersecting));
  }

  @Test
  public void removingEmptySetDoesNothing() {
    addTwoElements(testSet);

    assertFalse(testSet.removeAll(new HashDoubleSet(100)));
    assertFalse(testSet.removeAll(new HashSet<>()));
    assertContainsElements(testSet);
  }

  @Test
  public void removingDisjointSetDoesNothing() {
    addTwoElements(testSet);

    final HashDoubleSet disjoint = new HashDoubleSet(100);
    disjoint.add((double)2);
    disjoint.add((double)102);

    assertFalse(testSet.removeAll(disjoint));
    assertFalse(testSet.removeAll(new HashSet<>()));
    assertContainsElements(testSet);
  }

  @Test
  public void doesNotContainRemovedDoubleersectingSetPrimitive() {
    addTwoElements(testSet);

    final HashDoubleSet intersecting = new HashDoubleSet(100);

    intersecting.add((double)1);
    intersecting.add((double)102);

    assertTrue(testSet.removeAll(intersecting));
    assertTrue(testSet.contains((double)101));
    assertFalse(testSet.containsAll(intersecting));
  }

  @Test
  public void doesNotContainRemovedDoubleersectingSet() {
    addTwoElements(testSet);

    final HashSet<Double> intersecting = new HashSet<>();
    intersecting.add((double)1);
    intersecting.add((double)102);

    assertTrue(testSet.removeAll(intersecting));
    assertTrue(testSet.contains((double)101));
    assertFalse(testSet.containsAll(intersecting));
  }

  @Test
  public void isEmptyAfterRemovingEqualSetPrimitive() {
    addTwoElements(testSet);

    final HashDoubleSet equal = new HashDoubleSet(100);
    addTwoElements(equal);

    assertTrue(testSet.removeAll(equal));
    assertTrue(testSet.isEmpty());
  }

  @Test
  public void isEmptyAfterRemovingEqualSet() {
    addTwoElements(testSet);

    final HashSet<Double> equal = new HashSet<>();
    addTwoElements(equal);

    assertTrue(testSet.removeAll(equal));
    assertTrue(testSet.isEmpty());
  }

  @Test
  public void removeElementsFromIterator() {
    addTwoElements(testSet);

    final DoubleIterator iterator = testSet.iterator();
    while (iterator.hasNext())
      if (iterator.next() == 1)
        iterator.remove();

    assertEquals(1, testSet.size());
    assertTrue(testSet.contains((double)101));
  }

  @Test
  public void shouldNotContainMissingValueInitially() {
    assertFalse(testSet.contains(HashDoubleSet.NULL));
  }

  @Test
  public void shouldAllowMissingValue() {
    assertTrue(testSet.add(HashDoubleSet.NULL));
    assertTrue(testSet.contains(HashDoubleSet.NULL));
    assertFalse(testSet.add(HashDoubleSet.NULL));
  }

  @Test
  public void shouldAllowRemovalOfMissingValue() {
    assertTrue(testSet.add(HashDoubleSet.NULL));
    assertTrue(testSet.remove(HashDoubleSet.NULL));
    assertFalse(testSet.contains(HashDoubleSet.NULL));
    assertFalse(testSet.remove(HashDoubleSet.NULL));
  }

  @Test
  public void sizeAccountsForMissingValue() {
    testSet.add((double)1);
    testSet.add(HashDoubleSet.NULL);

    assertEquals(2, testSet.size());
  }

  @Test
  public void toArrayCopiesElementsDoubleoNewArrayIncludingMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashDoubleSet.NULL);

    final Double[] result = testSet.toArray(new Double[testSet.size()]);
    assertTrue(Arrays.asList(result).containsAll(Arrays.asList((double)1, (double)101, HashDoubleSet.NULL)));
  }

  @Test
  public void toObjectArrayCopiesElementsDoubleoNewArrayIncludingMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashDoubleSet.NULL);

    final double[] result = testSet.toArray();
    Arrays.sort(result);
    assertArrayEquals(new double[] {HashDoubleSet.NULL, 1, 101}, result);
  }

  @Test
  public void equalsAccountsForMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashDoubleSet.NULL);

    final HashDoubleSet other = new HashDoubleSet(100);
    addTwoElements(other);

    assertNotEquals(testSet, other);

    other.add(HashDoubleSet.NULL);
    Assert.assertEquals(testSet, other);

    testSet.remove(HashDoubleSet.NULL);

    assertNotEquals(testSet, other);
  }

  @Test
  public void consecutiveValuesShouldBeCorrectlyStored() {
    for (double i = 0; i < 127; ++i)
      testSet.add(i);

    assertEquals(127, testSet.size());

    int distinctElements = 0;
    for (final DoubleIterator i = testSet.iterator(); i.hasNext(); i.next())
      ++distinctElements;

    assertEquals(distinctElements, 127);
  }

  @Test
  public void hashCodeAccountsForMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashDoubleSet.NULL);

    final HashDoubleSet other = new HashDoubleSet(100);
    addTwoElements(other);

    other.add(HashDoubleSet.NULL);
    assertEquals(testSet.hashCode(), other.hashCode());
  }

  @Test
  public void iteratorAccountsForMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashDoubleSet.NULL);

    int missingValueCount = 0;
    final DoubleIterator iterator = testSet.iterator();
    while (iterator.hasNext())
      if (iterator.next() == HashDoubleSet.NULL)
        ++missingValueCount;

    assertEquals(1, missingValueCount);
  }

  @Test
  public void iteratorCanRemoveMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashDoubleSet.NULL);

    final DoubleIterator iterator = testSet.iterator();
    while (iterator.hasNext())
      if (iterator.next() == HashDoubleSet.NULL)
        iterator.remove();

    assertFalse(testSet.contains(HashDoubleSet.NULL));
  }

  @Test
  public void shouldGenerateStringRepresentation() {
    final double[] testEntries = {(double)65, (double)68, (double)83, (double)104, (double)111, (double)75, (double)83, (double)97};
    for (final double testEntry : testEntries)
      testSet.add(testEntry);

    final String string = testSet.toString();
    final String[] parts = string.substring(1, string.length() - 1).replace(" ", "").split(",");
    final HashSet<String> strings = CollectionUtil.asCollection(new HashSet<>(testSet.size()), parts);
    for (final double testEntry : testEntries)
      assertTrue(Arrays.toString(parts), strings.contains(String.valueOf(testEntry)));
  }

  @Test
  public void shouldRemoveMissingValueWhenCleared() {
    assertTrue(testSet.add(HashDoubleSet.NULL));

    testSet.clear();

    assertFalse(testSet.contains(HashDoubleSet.NULL));
  }

  @Test
  public void shouldHaveCompatibleEqualsAndHashcode() {
    final HashSet<Double> compatibleSet = new HashSet<>();
    final long seed = System.nanoTime();
    final Random r = new Random(seed);
    for (int i = 0; i < 1024; ++i) {
      final double value = (double)r.nextInt();
      compatibleSet.add(value);
      testSet.add(value);
    }

    if (r.nextBoolean()) {
      compatibleSet.add(HashDoubleSet.NULL);
      testSet.add(HashDoubleSet.NULL);
    }

    assertTrue("Fail with seed:" + seed, testSet.size() == compatibleSet.size() && testSet.containsAll(compatibleSet));
    Assert.assertEquals("Fail with seed:" + seed, compatibleSet.hashCode(), testSet.hashCode());
  }

  private static void addTwoElements(final HashDoubleSet obj) {
    obj.add((double)1);
    obj.add((double)101);
  }

  private static void addTwoElements(final HashSet<Double> obj) {
    obj.add((double)1);
    obj.add((double)101);
  }

  private void assertIteratorHasElements() {
    final DoubleIterator iterator = testSet.iterator();

    final Set<Double> values = new HashSet<>();

    assertTrue(iterator.hasNext());
    values.add(iterator.next());
    assertTrue(iterator.hasNext());
    values.add(iterator.next());
    assertFalse(iterator.hasNext());

    assertContainsElements(values);
  }

  private void assertIteratorHasElementsWithoutHasNext() {
    final DoubleIterator iterator = testSet.iterator();
    final Set<Double> values = new HashSet<>();

    values.add(iterator.next());
    values.add(iterator.next());

    assertContainsElements(values);
  }

  private static void assertArrayContainingElements(final Double[] result) {
    assertTrue(Arrays.asList(result).containsAll(Arrays.asList((double)1, (double)101)));
  }

  private static void assertContainsElements(final Set<Double> other) {
    assertTrue(other.containsAll(Arrays.asList((double)1, (double)101)));
  }

  private static void assertContainsElements(final DoubleSet other) {
    assertTrue(other.containsAll(Arrays.asList((double)1, (double)101)));
  }

  private void exhaustIterator() {
    final DoubleIterator iterator = testSet.iterator();
    iterator.next();
    iterator.next();
    iterator.next();
  }
}