/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.wait;

import static org.kuali.common.util.base.Precondition.checkMin;

public final class WaitResult {

	private final long start;
	private final long stop;
	private final long elapsed;

	public static WaitResult create(long start, long stop) {
		return builder(start, stop).build();
	}

	public static Builder builder(long start, long stop) {
		return new Builder(start, stop);
	}

	public static class Builder {

		private final long start;
		private final long stop;
		private final long elapsed;

		public Builder(long start, long stop) {
			this.start = start;
			this.stop = stop;
			this.elapsed = stop - start;
		}

		public WaitResult build() {
			WaitResult instance = new WaitResult(this);
			validate(instance);
			return instance;
		}

		private static void validate(WaitResult instance) {
			checkMin(instance.start, 0, "start");
			checkMin(instance.stop, instance.start, "stop");
			checkMin(instance.elapsed, 0, "elapsed");
		}

	}

	private WaitResult(Builder builder) {
		this.start = builder.start;
		this.stop = builder.stop;
		this.elapsed = builder.elapsed;
	}

	public long getStart() {
		return start;
	}

	public long getElapsed() {
		return elapsed;
	}

	public long getStop() {
		return stop;
	}

}
