/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.wait;

import static org.kuali.common.util.base.Precondition.checkMin;

import org.kuali.common.util.FormatUtils;

public final class WaitContext {

	private final long timeoutMillis;
	private final long sleepMillis;
	private final long initialPauseMillis;

	public static WaitContext create(long timeoutMillis) {
		return builder(timeoutMillis).build();
	}

	public static Builder builder(long timeoutMillis) {
		return new Builder(timeoutMillis);
	}

	public static class Builder implements org.apache.commons.lang3.builder.Builder<WaitContext> {

		// Required
		private final long timeoutMillis;

		// Optional
		private long sleepMillis = FormatUtils.getMillis("1s"); // 1 second
		private long initialPauseMillis = 0;

		public Builder(long timeoutMillis) {
			this.timeoutMillis = timeoutMillis;
		}

		public Builder initialPauseMillis(long initialPauseMillis) {
			this.initialPauseMillis = initialPauseMillis;
			return this;
		}

		public Builder sleepMillis(long sleepMillis) {
			this.sleepMillis = sleepMillis;
			return this;
		}

		@Override
		public WaitContext build() {
			WaitContext instance = new WaitContext(this);
			validate(instance);
			return instance;
		}

		private static void validate(WaitContext instance) {
			checkMin(instance.timeoutMillis, 0, "timeoutMillis");
			checkMin(instance.sleepMillis, 0, "sleepMillis");
			checkMin(instance.initialPauseMillis, 0, "initialPauseMillis");
		}

	}

	private WaitContext(Builder builder) {
		this.timeoutMillis = builder.timeoutMillis;
		this.sleepMillis = builder.sleepMillis;
		this.initialPauseMillis = builder.initialPauseMillis;
	}

	public long getTimeoutMillis() {
		return timeoutMillis;
	}

	public long getSleepMillis() {
		return sleepMillis;
	}

	public long getInitialPauseMillis() {
		return initialPauseMillis;
	}

}
