/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.base;

import static com.google.common.base.Preconditions.checkState;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.kuali.common.util.FormatUtils.getMillis;
import static org.kuali.common.util.base.Exceptions.illegalState;

import java.util.List;

import com.google.common.base.Optional;
import com.google.common.base.Stopwatch;

public class Threads {

    /**
     * Check to make sure elapsed time on the stopwatch has not exceeded {@code timeout} and then sleep for the indicated number of milliseconds
     *
     * @throws IllegalStateException
     *             if elapsed time on the stopwatch has exceeded {@code timeout} or if {@code InterruptedException} is thrown while sleeping
     *
     * @returns The number of milliseconds this method slept for
     */
    public static long checkedWait(Stopwatch sw, long timeout, long sleep) {
        checkState(sw.elapsed(MILLISECONDS) <= timeout, "wait timeout exceeded");
        return sleep(sleep);
    }

    /**
     * Sleep for {@code time} where time is in the format 15ms, 15s, 15m, 15h, 15d, 15y for 15 milliseconds, seconds, minutes, hours, days, and years, respectively. Note that years
     * is approximate as the logic always assumes exactly 365 days a year.
     *
     * @throws <code>IllegalStateException</code> if interrupted.
     */
    public static long sleep(String time) {
        return sleep(getMillis(time));
    }

    /**
     * Sleep for <code>millis</code> or zero, if {@code !millis.isPresent()}
     *
     * @throws <code>IllegalStateException</code> if interrupted.
     */
    public static long sleep(Optional<? extends Number> millis) {
        if (millis.isPresent()) {
            return sleep(millis.get().longValue());
        } else {
            return 0;
        }
    }

    /**
     * Sleep for <code>millis</code>
     *
     * @throws <code>IllegalStateException</code> if interrupted.
     */
    public static long sleep(long millis) {
        try {
            Thread.sleep(millis);
            return millis;
        } catch (InterruptedException e) {
            throw illegalState(e);
        }
    }

    /**
     * Start each thread
     */
    public static void start(List<Thread> threads) {
        for (Thread thread : threads) {
            thread.start();
        }
    }

    /**
     * Invoke join() on each thread
     *
     * @throws <code>IllegalStateException</code> if any thread gets interrupted.
     */
    public static void join(List<Thread> threads) {
        for (Thread thread : threads) {
            try {
                thread.join();
            } catch (InterruptedException e) {
                throw illegalState(e, "unexpected interruption [thread id:%s] [thread name:%s]", thread.getId(), thread.getName());
            }
        }
    }
}
