/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;

import java.util.Objects;

import org.kuali.common.util.equality.Equatable;

public class ObjectUtils {

	/**
	 * <p>
	 * This method returns <code>true</code> if the <code>toString()</code> methods on both objects return matching strings AND both objects are the exact same runtime type.
	 * </p>
	 * 
	 * <p>
	 * Returns <code>true</code> immediately if <code>reference == other</code> (ie they are the same object).
	 * </p>
	 * 
	 * <p>
	 * Returns <code>false</code> immediately if <code>other==null</code> or is a different runtime type than <code>reference</code>.
	 * </p>
	 * 
	 * <p>
	 * If neither one is <code>null</code>, and both are the exact same runtime type, then compare the <code>toString()</code> methods
	 * </p>
	 * 
	 * @param reference
	 *            The object <code>other</code> is being compared to.
	 * @param other
	 *            The object being examined for equality with <code>reference</code>.
	 * 
	 * @throws NullPointerException
	 *             If <code>reference</cod> is <code>null</code> or <code>reference.toString()</code> returns <code>null</code>
	 */
	public static boolean equalByToString(Object reference, Object other) {
		checkNotNull(reference);

		// They are the same object
		if (reference == other) {
			return true;
		}

		if (notEqual(reference, other)) {
			return false; // Don't bother comparing the toString() methods
		} else {
			return reference.toString().equals(other.toString());
		}
	}

	/**
	 * Return true if {@code reference} and {@code object} are not null, the exact same runtime type, and {@code reference.compareTo(object) == 0}
	 * 
	 * @throws NullPointerException
	 *             If {@code reference} is {@code null}
	 */
	public static <T extends Comparable<? super T>> boolean equalByComparison(T reference, Object object) {
		if (notEqual(reference, object)) {
			return false;
		} else {
			// notEqual() guarantees that reference and object are the exact same runtime type
			@SuppressWarnings("unchecked")
			T other = (T) object;
			return reference.compareTo(other) == 0;
		}

	}

	/**
	 * <p>
	 * Return true if <code>reference</code> is definitely not equal to <code>other</code>.
	 * </p>
	 * 
	 * <p>
	 * If <code>other</code> is null <b>OR</b> a different runtime type than <code>reference</code>, return true
	 * </p>
	 * 
	 * @param reference
	 *            The object <code>other</code> is being compared to.
	 * @param other
	 *            The object being examined for equality with <code>reference</code>.
	 * 
	 * @throws NullPointerException
	 *             If <code>reference</cod> is <code>null</code>
	 */
	public static boolean notEqual(Object reference, Object other) {
		checkNotNull(reference);
		if (other == null) {
			return true;
		} else {
			return !reference.getClass().equals(other.getClass());
		}
	}

	public static <T extends Equatable> boolean equal(T reference, Object other) {
		checkNotNull(reference);
		if (reference == other) {
			return true;
		} else if (notEqual(reference, other)) {
			return false;
		} else {
			return equal(reference, (Equatable) other);
		}
	}

	private static boolean equal(Equatable reference, Equatable other) {
		Object[] values1 = reference.getEqualityValues();
		Object[] values2 = other.getEqualityValues();
		checkState(values1.length == values2.length, "value arrays must be the same. values1.length=%s values2.length=%s", values1.length, values2.length);
		for (int i = 0; i < values1.length; i++) {
			if (!Objects.equals(values1[i], values2[i])) {
				return false;
			}
		}
		return true;
	}

	/**
	 * @deprecated Use equalByToString() instead
	 */
	@Deprecated
	public static boolean equalsByToString(Object reference, Object other) {
		return equalByToString(reference, other);
	}

}
