/**
 * Copyright (c) 2009-2011, Lukas Eder, lukas.eder@gmail.com
 * All rights reserved.
 *
 * This software is licensed to you under the Apache License, Version 2.0
 * (the "License"); You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * . Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * . Neither the name "jOOQ" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.jooq.util;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class DefaultForeignKeyDefinition extends AbstractDefinition implements ForeignKeyDefinition {

    private final List<ColumnDefinition> keyColumns;
    private final TableDefinition        table;
    private final UniqueKeyDefinition    uniqueKey;

    public DefaultForeignKeyDefinition(Database database, String name, TableDefinition table,
        UniqueKeyDefinition uniqueKey) {

        super(database, name, null);

        this.keyColumns = new ArrayList<ColumnDefinition>();
        this.table = table;
        this.uniqueKey = uniqueKey;
    }

    @Override
    public TableDefinition getKeyTableDefinition() {
        return table;
    }

    @Override
    public List<ColumnDefinition> getKeyColumns() {
        return keyColumns;
    }

    @Override
    public UniqueKeyDefinition getReferencedKey() {
        return uniqueKey;
    }

    @Override
    public TableDefinition getReferencedTable() {
        return uniqueKey.getTable();
    }

    @Override
    public List<ColumnDefinition> getReferencedColumns() {
        return uniqueKey.getKeyColumns();
    }

    @Override
    public int countSimilarReferences() {
        Set<String> keys = new HashSet<String>();

        for (ColumnDefinition column : table.getColumns()) {
            ForeignKeyDefinition key = getDatabase().getRelations().getForeignKey(column);

            if (key != null) {
                if (key.getReferencedTable().equals(getReferencedTable())) {
                    keys.add(key.getName());
                }
            }
        }

        return keys.size();
    }
}
