/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.internal.apitool;

import org.jooby.Jooby;
import org.jooby.Mutant;
import org.jooby.Request;
import org.jooby.Response;
import org.jooby.Route;
import org.jooby.Router;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.AbstractInsnNode;
import org.objectweb.asm.tree.MethodInsnNode;
import org.objectweb.asm.tree.MethodNode;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

class Filters {

  private static List<Signature> PARAMS = params();

  public static Predicate<MethodInsnNode> joobyRun(final ClassLoader loader) {
    return call(loader, "org.jooby.JoobyKt", "run", "kotlin.jvm.functions.Function0",
        String.class.getName() + "[]").or(
        call(loader, "org.jooby.Jooby", "run", Supplier.class, String.class.getName() + "[]"));

  }

  public static <T> Predicate<T> is(final Class<? extends T> type) {
    return type::isInstance;
  }

  @SuppressWarnings("unchecked")
  public static <T> Predicate<T> and(
      final Predicate<? super T>... predicates) {
    return n -> Arrays.asList(predicates).stream()
        .reduce(true, (value, predicate) -> value && predicate.test(n), (l, r) -> l && r);
  }

  public static <T extends AbstractInsnNode> Predicate<T> opcode(final int opcode) {
    return it -> it.getOpcode() == opcode;
  }

  public static Predicate<MethodInsnNode> call(final Class owner, final String name,
      final Object... args) {
    return call(owner.getClassLoader(), Type.getInternalName(owner), name, args);
  }

  @SuppressWarnings("rawtypes")
  public static Predicate<MethodInsnNode> call(final ClassLoader loader, final String owner,
      final String name,
      final Object... args) {
    return is(MethodInsnNode.class).and(m -> {
      return new Signature(loader, owner, name, args).matches(m);
    });
  }

  public static Predicate<MethodNode> access(final int access) {
    return is(MethodNode.class).and(m -> {
      return (m.access & access) != 0;
    });
  }

  public static Predicate<MethodNode> kotlinRouteHandler() {
    return is(MethodNode.class).and(m -> {
      return (m.name.equals("invoke") || m.name.equals("handle")) && (
          (m.access & Opcodes.ACC_SYNTHETIC) == 0);
    });
  }

  public static Predicate<MethodInsnNode> mount(final ClassLoader loader, final String owner) {
    Signature use1 = new Signature(loader, owner, "use", Jooby.class);
    Signature use2 = new Signature(loader, owner, "use", String.class
        , Jooby.class);
    return is(MethodInsnNode.class).and(m -> {
      return use1.matches(m) || use2.matches(m);
    });
  }

  public static Predicate<MethodInsnNode> use(final ClassLoader loader, final String owner) {
    Signature use = new Signature(loader, owner, "use", Class.class);
    Signature usepath = new Signature(loader, owner, "use", String.class, Class.class);
    Signature kuse = new Signature(loader, owner, "use", "kotlin.reflect.KClass");
    Signature kusepath = new Signature(loader, owner, "use", String.class, "kotlin.reflect.KClass");
    return is(MethodInsnNode.class)
        .and(m -> use.matches(m) || usepath.matches(m) || kuse.matches(m) || kusepath.matches(m));
  }

  public static Predicate<MethodInsnNode> path(final ClassLoader loader, final String owner) {
    Signature path = new Signature(loader, owner, "path", String.class, Runnable.class);
    return is(MethodInsnNode.class).and(m -> path.matches(m));
  }

  public static Predicate<MethodNode> methodName(final String name) {
    return is(MethodNode.class).and(m -> m.name.equals(name));
  }

  public static Predicate<MethodInsnNode> sendObject() {
    return call(Response.class, "send", Object.class.getName());
  }

  @SuppressWarnings("rawtypes")
  public static Predicate<MethodNode> method(final String name, final Object... args) {
    return is(MethodNode.class).and(m -> {
      if (name.equals(m.name)) {
        Type[] params = Type.getArgumentTypes(m.desc);
        if (params.length == args.length) {
          for (int i = 0; i < args.length; i++) {
            Object param = params[i].getClassName();
            Object arg = args[i];
            if (arg instanceof Class) {
              arg = ((Class) arg).getName();
            }
            if (!param.equals(arg)) {
              return false;
            }
          }
          return true;
        }
      }
      return false;
    });
  }

  public static Predicate<MethodInsnNode> scriptRoute(ClassLoader loader) {
    List<Signature> routes = collectScriptRoutes(loader);
    return is(MethodInsnNode.class).and(m -> {
      Signature signature = new Signature(loader, m);
      return routes.stream()
          .filter(signature::matches)
          .findFirst()
          .isPresent();
    });
  }

  public static Predicate<MethodInsnNode> getOrCreateKotlinClass() {
    return m -> {
      return m.owner.equals("kotlin/jvm/internal/Reflection") && m.name
          .equals("getOrCreateKotlinClass");
    };
  }

  public static Predicate<MethodInsnNode> mutantToSomething() {
    return m -> {
      if (m.owner.equals(Mutant.class.getName().replace('.', '/')) && m.name.startsWith("to")) {
        return true;
      }
      return false;
    };
  }

  public static Predicate<MethodInsnNode> mutantValue() {
    return m -> {
      if (m.owner.equals(Mutant.class.getName().replace('.', '/'))) {
        return m.name.equals("value") || m.name.endsWith("Value");
      }
      // property value
      if (m.owner.equals("org/jooby/JoobyKt")) {
        if (m.name.startsWith("get") && m.name.endsWith("Value")) {
          Type[] args = Type.getArgumentTypes(m.desc);
          if (args.length >= 1) {
            return args[0].getClassName().equals(Mutant.class.getName());
          }
        }
      }
      return false;
    };
  }

  public static Predicate<MethodInsnNode> param(final ClassLoader loader) {
    return is(MethodInsnNode.class).and(m -> {
      Signature signature = new Signature(loader, m);
      return PARAMS.stream()
          .filter(signature::matches)
          .findFirst()
          .isPresent();
    });
  }

  public static Predicate<MethodInsnNode> file() {
    return is(MethodInsnNode.class).and(m -> {
      if ((m.name.equals("file") || m.name.equals("files")) && m.owner
          .equals(Request.class.getName().replace(".", "/"))) {
        return true;
      }
      return false;
    });
  }

  private static List<Signature> params() {
    Predicate<Method> args = argument(String.class)
        .or(argument(String.class, String[].class))
        .or(argument(Class.class));

    List<Signature> signatures = signatures(Request.class, m -> {
      String name = m.getName();
      switch (name) {
        case "param":
        case "header":
        case "file":
        case "files":
        case "params":
        case "form":
          return args.test(m);
        case "body":
          return m.getParameterCount() == 0 || args.test(m);

      }
      return false;
    });

    return signatures;
  }

  private static List<Signature> collectScriptRoutes(ClassLoader loader) {

    Function<String, Optional<Class<?>>> loadClass = name -> {
      try {
        return Optional.of(loader.loadClass(name));
      } catch (ClassNotFoundException e) {
        return Optional.empty();
      }
    };
    Predicate<Method> route = argument(Route.ZeroArgHandler.class)
        .or(argument(Route.OneArgHandler.class))
        .or(argument(Route.Handler.class))
        .or(argument(Route.Filter.class))
        //
        .or(argument(String.class, Route.ZeroArgHandler.class))
        .or(argument(String.class, Route.OneArgHandler.class))
        .or(argument(String.class, Route.Handler.class))
        .or(argument(String.class, Route.Filter.class))
        //
        .or(argument(String.class, String.class, Route.ZeroArgHandler.class))
        .or(argument(String.class, String.class, Route.OneArgHandler.class))
        .or(argument(String.class, String.class, Route.Handler.class))
        .or(argument(String.class, String.class, Route.Filter.class))
        //
        .or(argument(String.class, String.class, String.class, Route.ZeroArgHandler.class))
        .or(argument(String.class, String.class, String.class, Route.OneArgHandler.class))
        .or(argument(String.class, String.class, String.class, Route.Handler.class))
        .or(argument(String.class, String.class, String.class, Route.Filter.class));

    Predicate<Method> krouteArg = argument("kotlin.jvm.functions.Function1")
        .or(argument("kotlin.jvm.functions.Function2"))
        .or(argument("kotlin.jvm.functions.Function3"))
        //
        .or(argument(String.class, "kotlin.jvm.functions.Function1"))
        .or(argument(String.class, "kotlin.jvm.functions.Function2"))
        .or(argument(String.class, "kotlin.jvm.functions.Function3"))
        //
        .or(argument(String.class, String.class, "kotlin.jvm.functions.Function1"))
        .or(argument(String.class, String.class, "kotlin.jvm.functions.Function2"))
        .or(argument(String.class, String.class, "kotlin.jvm.functions.Function3"))
        // default method: get$default
        .or(argument("org.jooby.Kooby", String.class, "kotlin.jvm.functions.Function1",
            int.class, Object.class))
        .or(argument("org.jooby.Kooby", String.class, "kotlin.jvm.functions.Function2",
            int.class, Object.class))
        .or(argument("org.jooby.Kooby", String.class, "kotlin.jvm.functions.Function3",
            int.class, Object.class));

    Predicate<Method> kotlinScriptRoute = m -> {
      String name = m.getName().replace("$default", "").toUpperCase();
      if (Route.METHODS.contains(name) || name.equals("ALL")) {
        return krouteArg.test(m);
      }
      return false;
    };

    List<Signature> signatures = signatures(Router.class, route);
    loadClass.apply("org.jooby.Kooby").ifPresent(c -> {
      signatures.addAll(signatures(c, kotlinScriptRoute));
    });

    loadClass.apply("org.jooby.KRouteGroup").ifPresent(c -> {
      signatures.addAll(signatures(c, kotlinScriptRoute));
    });

    return signatures;
  }

  private static Predicate<Method> argument(final Object... types) {
    return m -> {
      Class<?>[] params = m.getParameterTypes();
      if (params.length == types.length) {
        for (int i = 0; i < types.length; i++) {
          Object type = types[i];
          Object it;
          if (type instanceof String) {
            it = params[i].getName();
          } else {
            it = params[i];
          }
          if (!it.equals(type)) {
            return false;
          }
        }
        return true;
      }
      return false;
    };
  }

  @SuppressWarnings("rawtypes")
  private static List<Signature> signatures(final Class owner, final Predicate<Method> filter) {
    Method[] methods = owner.getDeclaredMethods();
    List<Signature> result = new ArrayList<>();
    for (Method method : methods) {
      String name = method.getName();
      if (!name.contains("deferred") && filter.test(method)) {
        result.add(new Signature(method));
      }

    }
    return result;
  }

}
