/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.internal.apitool;

import com.google.common.base.Strings;
import org.antlr.v4.runtime.ANTLRErrorListener;
import org.antlr.v4.runtime.BaseErrorListener;
import org.antlr.v4.runtime.CharStream;
import org.antlr.v4.runtime.CharStreams;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.RecognitionException;
import org.antlr.v4.runtime.Recognizer;
import org.antlr.v4.runtime.TokenStream;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeWalker;
import org.antlr.v4.runtime.tree.TerminalNode;
import org.jooby.Route;
import static org.jooby.Route.normalize;
import org.jooby.Status;
import org.jooby.internal.apitool.javadoc.FuzzyDocBaseListener;
import org.jooby.internal.apitool.javadoc.FuzzyDocLexer;
import org.jooby.internal.apitool.javadoc.FuzzyDocParser;
import org.jooby.internal.apitool.javadoc.JavadocLexer;
import org.jooby.internal.apitool.javadoc.JavadocParser;
import org.jooby.internal.apitool.javadoc.JavadocParserBaseListener;
import org.jooby.internal.apitool.javadoc.StatusCodeBaseListener;
import org.jooby.internal.apitool.javadoc.StatusCodeLexer;
import org.jooby.internal.apitool.javadoc.StatusCodeParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * Not thread safe due to internal cache.
 */
class DocParser {

  private static class DocCollector extends FuzzyDocBaseListener {
    LinkedList<String> prefix = new LinkedList<>();

    LinkedList<Boolean> insideRoute = new LinkedList<>();

    List<DocItem> doc;

    LinkedList<String> summary = new LinkedList<>();

    Path file;

    public List<DocItem> collect(Path file) throws IOException {
      this.file = file;
      doc = new ArrayList<>();
      CharStream input = CharStreams.fromPath(file, StandardCharsets.UTF_8);
      FuzzyDocLexer lexer = new FuzzyDocLexer(input);
      ANTLRErrorListener error = errorListener(LoggerFactory.getLogger(DocCollector.class), file);
      lexer.removeErrorListeners();
      lexer.addErrorListener(error);
      TokenStream tokens = new CommonTokenStream(lexer);
      FuzzyDocParser parser = new FuzzyDocParser(tokens);
      parser.removeErrorListeners();
      parser.addErrorListener(error);
      ParseTreeWalker.DEFAULT.walk(this, parser.source());
      return doc;
    }

    private static ANTLRErrorListener errorListener(final Logger log, Path file) {
      return new BaseErrorListener() {
        @Override
        public void syntaxError(final Recognizer<?, ?> recognizer, final Object offendingSymbol,
            final int line,
            final int charPositionInLine, final String msg, final RecognitionException e) {
          log.debug("{}:{}:{} {}", file, line, charPositionInLine, msg);
        }
      };
    }

    /**
     * <pre>{@code
     *   use("/api/pattern");
     * }</>
     * @param ctx
     */
    @Override public void enterUse(final FuzzyDocParser.UseContext ctx) {
      if (ctx.comma.size() > 0) {
        String comment = ctx.doc.getText();
        String method = Optional.ofNullable(ctx.method).map(it -> str(it.getText())).orElse("*");
        String pattern = Optional.ofNullable(ctx.pattern).map(it -> str(it.getText()))
            .orElse("*");
        doc.add(
            doc(method, normalize(pattern), file, summary(), comment));
      } else {
        this.prefix.addLast(Route.normalize(str(ctx.pattern.getText())));
        popInsideRoute();
        this.summary.addLast(cleanJavadoc(file, ctx.doc.getText()));
      }
    }

    @Override public void enterPath(FuzzyDocParser.PathContext ctx) {
      this.prefix.addLast(Route.normalize(str(ctx.pattern.getText())));
      this.insideRoute.addLast(true);
      if (ctx.doc != null) {
        this.summary.addLast(cleanJavadoc(file, ctx.doc.getText()));
      }
    }

    @Override public void exitPath(FuzzyDocParser.PathContext ctx) {
      popPrefix();
      popSummary();
      popInsideRoute();
    }

    /**
     * <pre>{@code
     *   route("/api/pattern");
     * }</>
     * @param ctx
     */
    @Override public void enterRoute(final FuzzyDocParser.RouteContext ctx) {
      this.prefix.addLast(Route.normalize(str(ctx.pattern.getText())));
      this.insideRoute.addLast(true);
      this.summary.addLast(cleanJavadoc(file, ctx.doc.getText()));
    }

    @Override public void exitRoute(final FuzzyDocParser.RouteContext ctx) {
      popPrefix();
      popSummary();
      popInsideRoute();
    }

    @Override public void enterScript(final FuzzyDocParser.ScriptContext ctx) {
      if (ctx.dot == null && !insideRoute()) {
        /**
         * reset prefix when '.' is missing:
         *
         * use("prefix")
         *   .get()
         *
         * vs
         *
         * get("pattern")
         *
         * but ignore dot when route(prefix) is present (kotlin)
         */
        popPrefix();
        popSummary();
      }
      String comment = Optional.ofNullable(ctx.doc).map(it -> it.getText()).orElse("");
      if (ctx.method != null) {
        String method = ctx.method.getText();
        String pattern =
            prefix.stream().collect(Collectors.joining("")) + "/" + Optional.ofNullable(ctx.pattern)
                .map(it -> str(it.getText()))
                .orElse("/");
        doc.add(doc(method, normalize(pattern), file, summary(), comment));
      }
    }

    private void popSummary() {
      if (summary.size() > 0) {
        summary.removeLast();
      }
    }

    private boolean insideRoute() {
      return insideRoute.isEmpty() ? false : insideRoute.getLast();
    }

    private void popInsideRoute() {
      if (insideRoute.size() > 0) {
        insideRoute.removeLast();
      }
    }

    private String summary() {
      return summary.stream().collect(Collectors.joining());
    }

    @Override public void enterClazz(final FuzzyDocParser.ClazzContext ctx) {
      String pattern = pattern(ctx.annotations);
      if (pattern != null) {
        /** FuzzyJavaDoc should be better... sometimes reports a method as class elements due to
         * annotations. We hack it and test if the current context has the `class` keyword, if
         * keyword is missing we treat this element as a MVC method.
         */
        boolean isClass = ctx.isClass != null;
        if (isClass) {
          this.prefix.addLast(normalize(pattern));
          if (ctx.doc != null) {
            this.summary.addLast(cleanJavadoc(file, ctx.doc.getText()));
          }
        } else {
          popPrefix();
          popSummary();
          List<String> methods = methods(ctx.annotations);
          if (ctx.doc != null) {
            String comment = ctx.doc.getText();
            if (methods.size() == 0) {
              doc.add(doc("get", normalize(pattern), file, summary(), comment));
            } else {
              methods.stream()
                  .forEach(it -> doc.add(doc(it, normalize(pattern), file, summary(), comment)));
            }
          }
        }
      } else {
        popPrefix();
      }
    }

    @Override public void exitClazz(final FuzzyDocParser.ClazzContext ctx) {
      popPrefix();
    }

    private void popPrefix() {
      if (prefix.size() > 0) {
        prefix.removeLast();
      }
    }

    private String pattern(List<FuzzyDocParser.AnnotationContext> annotations) {
      return annotations.stream()
          .filter(it -> it.name.getText().equals("@Path"))
          .findFirst()
          .map(it -> Optional.ofNullable(it.value).map(e -> str(e.getText())).orElse("/"))
          .orElse(null);
    }

    private List<String> methods(List<FuzzyDocParser.AnnotationContext> annotations) {
      return annotations.stream()
          .filter(it -> Route.METHODS.contains(it.name.getText().substring(1)))
          .map(it -> it.name.getText().substring(1))
          .collect(Collectors.toList());
    }

    @Override public void enterMvcRoute(final FuzzyDocParser.MvcRouteContext ctx) {
      String path = pattern(ctx.annotations);
      List<String> methods = methods(ctx.annotations);
      if (methods.isEmpty() && path == null) {
        // not an mvc method
        return;
      }
      String comment = ctx.doc.getText();
      String pattern =
          prefix.stream().collect(Collectors.joining()) + "/" + Optional.ofNullable(path)
              .orElse("/");
      if (methods.size() == 0) {
        doc.add(doc("get", normalize(pattern), file, summary(), comment));
      } else {
        methods.stream()
            .forEach(method -> doc.add(doc(method, normalize(pattern), file, summary(), comment)));
      }
    }

    private static JavadocParser.DocumentationContext parseJavadoc(Path file, String doc) {
      ANTLRErrorListener error = errorListener(LoggerFactory.getLogger(DocCollector.class), file);
      CharStream stream = CharStreams.fromString(doc == null ? "" : doc);
      JavadocLexer lexer = new JavadocLexer(stream);
      lexer.removeErrorListeners();
      lexer.addErrorListener(error);
      CommonTokenStream tokens = new CommonTokenStream(lexer);
      JavadocParser parser = new JavadocParser(tokens);
      parser.removeErrorListeners();
      parser.addErrorListener(error);
      return parser.documentation();
    }

    private static String cleanJavadoc(Path file, String doc) {
      return cleanJavadoc(parseJavadoc(file, doc));
    }

    private static String cleanJavadoc(JavadocParser.DocumentationContext ctx) {
      StringBuilder buff = new StringBuilder();
      JavadocParserBaseListener listener = new JavadocParserBaseListener() {
        @Override
        public void enterDescriptionLine(final JavadocParser.DescriptionLineContext ctx) {
          buff.append(ctx.getText()).append("\n");
        }
      };
      ParseTreeWalker.DEFAULT.walk(listener, ctx);
      return buff.toString();
    }

    private static DocItem doc(String method, String pattern, Path file, String summary,
        String doc) {
      Map<String, String> params = new LinkedHashMap<>();
      StringBuilder returns = new StringBuilder();
      StringBuilder text = new StringBuilder();
      JavadocParserBaseListener listener = new JavadocParserBaseListener() {

        @Override
        public void enterDescriptionLine(final JavadocParser.DescriptionLineContext ctx) {
          text.append(ctx.getText()).append("\n");
        }

        @Override public void enterBlockTag(final JavadocParser.BlockTagContext ctx) {
          String tag = ctx.blockTagName().NAME().getText();
          List<TerminalNode> terminals = ctx.blockTagContent().stream()
              .flatMap(it -> terminals(it).stream())
              .collect(Collectors.toList());
          boolean param = false;
          if (tag.equals("param") && terminals.size() > 0) {
            tag = terminals.get(0).getText();
            param = true;
          }
          String desc = terminals.stream()
              .skip(param ? 1 : 0)
              .map(TerminalNode::getText)
              .filter(it -> !it.trim().equals("*"))
              .collect(Collectors.joining())
              .trim();
          if (param) {
            params.put(tag, Strings.emptyToNull(desc));
          } else if (tag.equals("return")) {
            returns.append(desc);
          }
        }
      };
      ParseTreeWalker.DEFAULT.walk(listener, parseJavadoc(file, doc));
      return new DocItem(method, pattern, summary, text.toString(), params, returns.toString(),
          statusCodes(returns.toString()));
    }

    static Map<Integer, String> statusCodes(String text) {
      if (text.length() == 0) {
        return new LinkedHashMap<>();
      }
      CharStream input = CharStreams.fromString(text);
      StatusCodeLexer lexer = new StatusCodeLexer(input);
      CommonTokenStream tokens = new CommonTokenStream(lexer);
      StatusCodeParser parser = new StatusCodeParser(tokens);
      Map<Integer, String> statusCodes = new LinkedHashMap<>();
      StatusCodeBaseListener listener = new StatusCodeBaseListener() {
        @Override public void enterHtmlCode(final StatusCodeParser.HtmlCodeContext ctx) {
          parse(ctx.statusCode.getText()
              .replace("<code>", "")
              .replace("</code>", ""));
        }

        @Override public void enterJsonCode(final StatusCodeParser.JsonCodeContext ctx) {
          String json = ctx.statusCode.getText();
          parse(json.substring(1, json.length() - 1));
        }

        @Override public void enterMarkdownCode(final StatusCodeParser.MarkdownCodeContext ctx) {
          String md = ctx.statusCode.getText();
          parse(md.substring(1, md.length() - 1));
        }

        private void parse(final String text) {
          Pattern pattern = Pattern.compile("(\\d+)\\s*([:=]\\s*((.*)))?");
          Matcher matcher = pattern.matcher(text);
          if (matcher.find()) {
            try {
              int statusCode = Integer.parseInt(matcher.group(1).trim());
              String desc = Optional.ofNullable(matcher.group(3))
                  .orElse(Status.valueOf(statusCode).reason());
              statusCodes.put(statusCode, desc);
            } catch (IllegalArgumentException x) {
              LoggerFactory.getLogger(DocParser.class).debug("Ignoring status code", x);
            }
          }
        }
      };

      ParseTreeWalker.DEFAULT.walk(listener, parser.text());
      return statusCodes;
    }

    static List<TerminalNode> terminals(ParseTree context) {
      List<TerminalNode> terminals = new ArrayList<>();
      for (int i = 0; i < context.getChildCount(); i++) {
        if (context.getChild(i) instanceof TerminalNode) {
          terminals.add((TerminalNode) context.getChild(i));
        } else {
          terminals.addAll(terminals(context.getChild(i)));
        }
      }
      return terminals;
    }

    private String str(final String text) {
      return text.substring(1, text.length() - 1);
    }
  }

  private final Logger log = LoggerFactory.getLogger(getClass());

  private final Path basedir;

  private final Map<String, List<DocItem>> javadocCache = new HashMap<>();

  public DocParser(Path basedir) {
    this.basedir = basedir;
  }

  private List<DocItem> javadoc(Path file) throws IOException {
    return new DocCollector().collect(file);
  }

  public Optional<DocItem> pop(final String classname, String method, String pattern) {
    try {
      List<DocItem> javadoc = javadoc(classname);
      int pos = IntStream.range(0, javadoc.size())
          .filter(i -> javadoc.get(i).matches(method, pattern))
          .findFirst()
          .orElse(-1);
      if (pos >= 0) {
        return Optional.ofNullable(javadoc.remove(pos));
      }
    } catch (Exception x) {
      log.debug("Parsing of javadoc resulted in exception: {}", classname, x);
    }
    return Optional.empty();
  }

  private List<DocItem> javadoc(final String classname) throws IOException {
    int i = classname.indexOf("$");
    String toplevelClass = classname;
    if (i > 0) {
      toplevelClass = classname.substring(0, i);
    }

    List<DocItem> doc = javadocCache.get(toplevelClass);
    if (doc != null) {
      return doc;
    }

    Function<String, Path> path = (name) -> Arrays.asList(name.split("\\."))
        .stream()
        .reduce(basedir, (base, segment) ->
            basedir.equals(base) ? Paths.get(segment) : base.resolve(segment), Path::resolve);

    Path filename = path.apply(toplevelClass);
    Path location = Files.walk(basedir)
        .filter(it -> {
          if (Files.isRegularFile(it)) {
            String java = filename.toString() + ".java";
            String kt = filename.toString() + ".kt";
            return it.toString().endsWith(java) || it.toString().endsWith(kt);
          }
          return false;
        })
        .findFirst()
        .orElse(null);
    if (location == null) {
      throw new FileNotFoundException(filename.toString());
    }
    doc = javadoc(location);
    javadocCache.put(toplevelClass, doc);
    return doc;
  }

}
