/*
 *   Copyright (C) 2005 Christian Schulte <cs@schulte.it>
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *     o Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *     o Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *   INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 *   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *   NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *   THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *   $JOMC: AbstractClasspathValidateMojo.java 5135 2016-04-08 13:53:07Z schulte $
 *
 */
package org.jomc.mojo;

import java.util.logging.Level;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.util.JAXBSource;
import javax.xml.transform.Source;
import org.apache.maven.plugin.MojoExecutionException;
import org.jomc.modlet.ModelContext;
import org.jomc.modlet.ModelValidationReport;
import org.jomc.modlet.ObjectFactory;
import org.jomc.tools.ClassFileProcessor;

/**
 * Base class for validating class path class file model objects.
 *
 * @author <a href="mailto:cs@schulte.it">Christian Schulte</a>
 * @version $JOMC: AbstractClasspathValidateMojo.java 5135 2016-04-08 13:53:07Z schulte $
 * @since 1.1
 */
public abstract class AbstractClasspathValidateMojo extends AbstractJomcMojo
{

    /**
     * Constant for the name of the tool backing the mojo.
     */
    private static final String TOOLNAME = "ClassFileProcessor";

    /**
     * Creates a new {@code AbstractClasspathValidateMojo} instance.
     */
    public AbstractClasspathValidateMojo()
    {
        super();
    }

    @Override
    protected final void executeTool() throws Exception
    {
        this.logSeparator();
        this.logProcessingModel( TOOLNAME, this.getModel() );

        final ModelContext context = this.createModelContext( this.getClasspathClassLoader() );
        final ClassFileProcessor tool = this.createClassFileProcessor( context );
        final JAXBContext jaxbContext = context.createContext( this.getModel() );
        final Source source = new JAXBSource( jaxbContext, new ObjectFactory().createModel( tool.getModel() ) );
        ModelValidationReport validationReport = context.validateModel( this.getModel(), source );

        this.log( context, validationReport.isModelValid() ? Level.INFO : Level.SEVERE, validationReport );

        if ( validationReport.isModelValid() )
        {
            validationReport = tool.validateModelObjects( context );

            if ( validationReport != null )
            {
                this.log( context, validationReport.isModelValid() ? Level.INFO : Level.SEVERE, validationReport );

                if ( !validationReport.isModelValid() )
                {
                    throw new MojoExecutionException( Messages.getMessage( "classFileValidationFailure" ) );
                }
            }

            this.logToolSuccess( TOOLNAME );
        }
        else
        {
            throw new MojoExecutionException( Messages.getMessage( "classFileValidationFailure" ) );
        }
    }

    /**
     * Gets the class loader to validate class path model objects of.
     *
     * @return The class loader to validate class path model objects of.
     *
     * @throws MojoExecutionException if getting the class loader fails.
     */
    protected abstract ClassLoader getClasspathClassLoader() throws MojoExecutionException;

}
