/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.runtime;

//$Id: DelegatingExecutionContext.java 1941 2008-08-20 14:07:10Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.jboss.bpm.model.Activity;
import org.jboss.bpm.model.FlowObject;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.Property;
import org.jboss.bpm.runtime.ExecutionContext;

/**
 * An ExecutionContext that delegates to the current {@link Activity} or {@link Process} for property rerieval.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 15-Aug-2008
 */
public class DelegatingExecutionContext implements ExecutionContext
{
  private FlowObject flowObject;
  private ExecutionContext delegate;
  private String activityPrefix;
  private String procPrefix;
  private Process proc;

  DelegatingExecutionContext(FlowObject flowObject, ExecutionContext delegate)
  {
    this.flowObject = flowObject;
    this.delegate = delegate;

    proc = flowObject.getProcess();
    procPrefix = proc.getName() + ".";
    activityPrefix = procPrefix + flowObject.getName() + ".";
  }

  public Object getAttachment(String name)
  {
    Object value = null;

    // Get activity property
    if (flowObject instanceof Activity && name.startsWith(activityPrefix))
    {
      Activity activity = (Activity)flowObject;
      String key = name.substring(activityPrefix.length());
      Property prop = activity.getProperty(key);
      if (prop != null)
      {
        value = prop.getValue();
      }
    }

    // Get process property
    else if (name.startsWith(procPrefix))
    {
      String key = name.substring(procPrefix.length());
      Property prop = proc.getProperty(key);
      if (prop != null)
      {
        value = prop.getValue();
      }
    }

    // Fallback to execution context
    if (value == null)
    {
      value = delegate.getAttachment(name);
    }

    return value;
  }

  public Collection<Key> getAttachmentKeys()
  {
    Collection<Key> keys = new ArrayList<Key>(delegate.getAttachmentKeys());

    // Add activity property keys
    if (flowObject instanceof Activity)
    {
      Activity activity = (Activity)flowObject;
      for (Property prop : activity.getProperties())
      {
        Key key = new Key(null, activityPrefix + prop.getName());
        keys.add(key);
      }
    }

    // Add process property keys
    for (Property prop : proc.getProperties())
    {
      Key key = new Key(null, procPrefix + prop.getName());
      keys.add(key);
    }

    return keys;
  }

  public <T> T getAttachment(Class<T> clazz)
  {
    return delegate.getAttachment(clazz);
  }

  public <T> T getAttachment(Class<T> clazz, String name)
  {
    return delegate.getAttachment(clazz, name);
  }

  public <T> T addAttachment(Class<T> clazz, Object value)
  {
    return delegate.addAttachment(clazz, value);
  }

  public <T> T addAttachment(Class<T> clazz, String name, Object value)
  {
    validateAttachmentKey(name);
    return delegate.addAttachment(clazz, name, value);
  }

  public Object addAttachment(String name, Object value)
  {
    validateAttachmentKey(name);
    return delegate.addAttachment(name, value);
  }

  public <T> T removeAttachment(Class<T> clazz)
  {
    return delegate.removeAttachment(clazz);
  }

  public <T> T removeAttachment(Class<T> clazz, String name)
  {
    validateAttachmentKey(name);
    return delegate.removeAttachment(clazz, name);
  }

  public Object removeAttachment(String name)
  {
    validateAttachmentKey(name);
    return delegate.removeAttachment(name);
  }

  private void validateAttachmentKey(String name)
  {
    if (name.startsWith(procPrefix) || name.startsWith(activityPrefix))
      throw new IllegalArgumentException("Invalid attachment key: " + name);
  }

  public String toString()
  {
    Map<Key,Object> combined = new HashMap<Key,Object>();
    for (Key key : getAttachmentKeys())
    {
      if (key.getClassPart() != null)
        combined.put(key, getAttachment(key.getClassPart(), key.getNamePart()));
      else
        combined.put(key, getAttachment(key.getNamePart()));
    }
    return combined.toString();
  }
}