/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: TokenExecutorImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.model.ConnectingObject;
import org.jboss.bpm.model.FlowObject;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.Process.ProcessStatus;
import org.jboss.bpm.ri.runtime.DelegatingToken;
import org.jboss.bpm.ri.runtime.MutableToken;
import org.jboss.bpm.ri.runtime.RuntimeProcess;
import org.jboss.bpm.runtime.FlowHandler;
import org.jboss.bpm.runtime.HandlerSupport;
import org.jboss.bpm.runtime.SignalHandler;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;
import org.jboss.bpm.runtime.Token.TokenStatus;

/**
 * The {@link FlowHandler} invokes the TokenExecutor to schedule {@link ConnectingObject} objects together with their
 * associated {@link Token}.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public class TokenExecutorImpl implements TokenExecutor
{
  // provide logging
  private static final Log log = LogFactory.getLog(TokenExecutorImpl.class);

  private RuntimeProcess rtProc;
  private ExecutorService executor = Executors.newCachedThreadPool();
  private Map<String, RunnableToken> runnableTokens = new HashMap<String, RunnableToken>();

  public TokenExecutorImpl(RuntimeProcess rtProc)
  {
    this.rtProc = rtProc;
  }

  public Set<Token> getRunnableTokens()
  {
    synchronized (runnableTokens)
    {
      Set<Token> tokenSet = new HashSet<Token>();
      for (RunnableToken rt : runnableTokens.values())
        tokenSet.add(rt.getToken());

      return Collections.unmodifiableSet(tokenSet);
    }
  }

  public boolean hasRunnableTokens()
  {
    synchronized (runnableTokens)
    {
      return runnableTokens.size() > 0;
    }
  }

  public void create(Token token, SequenceFlow initialFlow)
  {
    synchronized (runnableTokens)
    {
      MutableToken mutableToken = (MutableToken)token;
      mutableToken.setTokenStatus(TokenStatus.Created);
      mutableToken.setFlow(initialFlow);

      log.debug("Create Token: " + token);

      RunnableToken rtToken = new RunnableToken(rtProc, mutableToken);
      runnableTokens.put(token.getTokenID(), rtToken);
    }
  }

  public void start(Token token)
  {
    synchronized (runnableTokens)
    {
      ProcessStatus procStatus = getProcess(token).getProcessStatus();
      if (procStatus != ProcessStatus.Ready && procStatus != ProcessStatus.Active)
        throw new IllegalStateException("Cannot start token to process in state: " + procStatus);

      log.debug("Sart Token: " + token);
      MutableToken mutableToken = (MutableToken)token;
      mutableToken.setTokenStatus(TokenStatus.Started);

      RunnableToken rtToken = runnableTokens.get(token.getTokenID());
      executor.submit(rtToken);
    }
  }

  public void move(Token token, SequenceFlow flow)
  {
    synchronized (runnableTokens)
    {
      if (flow == null)
        throw new IllegalArgumentException("Flow cannot be null");

      MutableToken mutableToken = (MutableToken)token;
      mutableToken.setFlow(flow);
    }
  }

  public void stop(Token token)
  {
    synchronized (runnableTokens)
    {
      log.debug("Stop Token: " + token);
      MutableToken mutableToken = (MutableToken)token;
      mutableToken.setTokenStatus(TokenStatus.Stoped);
    }
  }

  public void destroy(Token token)
  {
    synchronized (runnableTokens)
    {
      log.debug("Destroy Token: " + token);
      MutableToken mutableToken = (MutableToken)token;
      mutableToken.setTokenStatus(TokenStatus.Destroyed);
      runnableTokens.remove(token.getTokenID());
    }
  }

  public String suspend(Token token)
  {
    synchronized (runnableTokens)
    {
      log.debug("Suspend Token: " + token);
      MutableToken mutableToken = (MutableToken)token;
      mutableToken.setTokenStatus(TokenStatus.Suspended);
      return token.getTokenID();
    }
  }

  public Token activate(String tokenID)
  {
    synchronized (runnableTokens)
    {
      RunnableToken rtToken = runnableTokens.get(tokenID);
      if (rtToken == null)
        throw new IllegalStateException("Not a runnable token: " + tokenID);

      Token token = rtToken.getToken();
      if (token.getTokenStatus() != TokenStatus.Suspended)
        throw new IllegalStateException("Activate token in state: " + token.getTokenStatus());

      log.debug("Activate Token: " + token);
      MutableToken mutableToken = (MutableToken)token;
      mutableToken.setTokenStatus(TokenStatus.Started);

      executor.submit(rtToken);
      return token;
    }
  }

  private Process getProcess(Token token)
  {
    return token.getFlow().getTargetRef().getProcess();
  }

  /****************************************************
   * The runnable Token
   */
  class RunnableToken implements Runnable
  {
    private RuntimeProcess rtProc;
    private TokenExecutor tokenExecutor;
    private MutableToken token;

    public RunnableToken(RuntimeProcess rtProc, MutableToken token)
    {
      this.tokenExecutor = rtProc.getTokenExecutor();
      this.rtProc = rtProc;
      this.token = token;
    }

    public Token getToken()
    {
      return token;
    }

    public void run()
    {
      Process proc = rtProc.getProcess();
      try
      {
        ConnectingObject flow = token.getFlow();
        if (flow == null)
          throw new IllegalStateException("Cannot obtain initial flow");

        TokenStatus tokStatus = token.getTokenStatus();
        ProcessStatus procStatus = proc.getProcessStatus();
        while (procStatus == ProcessStatus.Active && tokStatus == TokenStatus.Started)
        {
          // Get the target and its handlers
          FlowObject flowObject = token.getFlow().getTargetRef();
          FlowObjectImpl flowObjectImpl = (FlowObjectImpl)flowObject;
          SignalHandler sigHandler = getSignalHandler(flowObject);

          // Synchronize execution on the target FlowObject
          synchronized (flowObject)
          {
            // Create a Token that includes properties from the current Activity
            DelegatingToken tokCopy = new DelegatingToken(token);

            // Throw the Enter Signal
            sigHandler.throwEnterSignal(tokCopy);

            // Execute the target FlowObject
            flowObjectImpl.execute(tokCopy);

            // Transfer the token to the FlowHandler
            flowObjectImpl.executeFlowHandler(tokenExecutor, tokCopy);

            // Throw the Exit Signal
            sigHandler.throwExitSignal(tokCopy);

            tokStatus = token.getTokenStatus();
            procStatus = proc.getProcessStatus();
          }
        }

        // Notify Process on token termination
        terminateToken(proc);
      }
      catch (RuntimeException rte)
      {
        log.error("Process aborted: " + proc, rte);
        ((ProcessImpl)proc).setRuntimeException(rte);

        log.debug("Terminate all suspended tokens");
        Set<String> keySet = new HashSet<String>(runnableTokens.keySet());
        for (String tokID : keySet)
        {
          RunnableToken rtTok = runnableTokens.get(tokID);
          Token auxToken = rtTok.getToken();
          if (auxToken.getTokenStatus() == TokenStatus.Suspended)
            tokenExecutor.destroy(auxToken);
        }

        // Notify Process on token termination
        terminateToken(proc);
      }
    }

    private void terminateToken(Process proc)
    {
      // Destroy the token if not already done
      synchronized (runnableTokens)
      {
        TokenStatus status = token.getTokenStatus();
        if (status != TokenStatus.Suspended && status != TokenStatus.Destroyed)
          tokenExecutor.destroy(token);
      }

      // Notify the runtime process
      synchronized (rtProc)
      {
        rtProc.notifyAll();
      }
    }

    private SignalHandler getSignalHandler(FlowObject target)
    {
      HandlerSupport handlerSupport = getHandlerSupport(target);
      SignalHandler handler = handlerSupport.getSignalHandler();
      if (handler == null)
        throw new IllegalStateException("Cannot obtain signal handler from: " + target);

      return handler;
    }

    private HandlerSupport getHandlerSupport(FlowObject fo)
    {
      if (fo instanceof HandlerSupport == false)
        throw new IllegalStateException("Flow object does not implement handler support: " + fo);
      return (HandlerSupport)fo;
    }

    @Override
    public String toString()
    {
      return token.toString();
    }
  }
}