/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: StartEventImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.management.ObjectName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.client.ExecutionManager;
import org.jboss.bpm.client.SignalListener;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.Constants;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.ObjectNameFactory;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.SignalEventDetail;
import org.jboss.bpm.model.StartEvent;
import org.jboss.bpm.model.Process.ProcessStatus;
import org.jboss.bpm.model.Signal.SignalType;
import org.jboss.bpm.runtime.SignalHandler;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;
import org.jboss.util.id.UID;

/**
 * A Start Event indicates where a particular Process will start. In terms of Sequence Flow, the Start Event starts the
 * Flow of the Process, and thus, will not have any incoming Sequence Flow. A Start Event can have a Trigger that
 * indicates how the Process starts: Message, Timer, Rule, Link, or Multiple.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public class StartEventImpl extends EventImpl implements StartEvent, SingleOutFlowSetterSupport
{
  // provide logging
  private static final Log log = LogFactory.getLog(StartEventImpl.class);

  private SequenceFlow outFlow;
  private List<EventDetail> triggers = new ArrayList<EventDetail>();
  private SignalListener startListener;

  public StartEventImpl(String name)
  {
    super(name == null ? "Start" : name);
  }

  @Override
  public ObjectName getID()
  {
    if (id == null)
    {
      StringBuilder str = new StringBuilder(Constants.ID_DOMAIN + ":");
      str.append("type=StartEvent,name=" + getName() + ",id=" + new UID());
      id = ObjectNameFactory.create(str.toString());
    }
    return id;
  }
  
  public List<EventDetail> getTrigger()
  {
    return Collections.unmodifiableList(triggers);
  }

  public void addTrigger(EventDetail eventDetail)
  {
    triggers.add(eventDetail);
  }

  public SequenceFlow getOutFlow()
  {
    return outFlow;
  }

  public void setOutFlow(SequenceFlow flow)
  {
    this.outFlow = flow;
  }

  @Override
  protected void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    tokenExecutor.move(token, getOutFlow());
  }

  public SignalHandler getSignalHandler()
  {
    SignalHandler handler = super.getSignalHandler();
    if (handler == null)
    {
      handler = new SignalHandler()
      {
        SignalManager signalManager = SignalManager.locateSignalManager();

        public void throwEnterSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_START_EVENT_ENTER);
          signalManager.throwSignal(signal);
        }

        public void throwExitSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_START_EVENT_EXIT);
          signalManager.throwSignal(signal);
        }
      };
    }
    return handler;
  }

  @Override
  protected void register(final Process proc)
  {
    super.register(proc);

    for (EventDetail eventDetail : getTrigger())
    {
      EventDetailImpl eventDetailImpl = (EventDetailImpl)eventDetail;
      eventDetailImpl.initialize(this);

      // Register the start signal listener
      if (eventDetail instanceof SignalEventDetail)
      {
        if (startListener == null)
        {
          SignalEventDetail signalEventDetail = (SignalEventDetail)eventDetail;
          final Signal startSignal = signalEventDetail.getSignalRef();
          final StartEvent start = this;
          startListener = new SignalListener()
          {
            public boolean acceptSignal(Signal signal)
            {
              SignalType expType = startSignal.getSignalType();
              String expMsg = startSignal.getMessage();
              
              String sigMsg = signal.getMessage();
              SignalType sigType = signal.getSignalType();
              
              boolean accept = (expType == sigType);
              if (accept && expMsg != null)
                accept = accept && expMsg.equals(sigMsg);
              
              return accept;
            }

            public void catchSignal(Signal signal)
            {
              log.debug("catchSignal: " + signal);
              ProcessStatus procStatus = proc.getProcessStatus();
              if (procStatus == ProcessStatus.Ready || procStatus == ProcessStatus.Active)
              {
                log.debug("Start process from signal: " + signal);
                ExecutionManager exManager = ExecutionManager.locateExecutionManager();
                exManager.startProcess(start, null);
              }
              else
              {
                log.debug("Ignore start signal for process: " + proc);
              }
            }

            public String toString()
            {
              return start + "." + startSignal;
            }
          };
          SignalManager signalManager = SignalManager.locateSignalManager();
          signalManager.addSignalListener(startListener);
        }
      }
    }
  }

  @Override
  protected void unregister(final Process proc)
  {
    // Unregister the start signal listener
    if (startListener != null)
    {
      SignalManager signalManager = SignalManager.locateSignalManager();
      signalManager.removeSignalListener(startListener);
      startListener = null;
    }
    super.unregister(proc);
  }

  public String toString()
  {
    return "StartEvent[" + getName() + "]";
  }
}