/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: ProcessBuilderImpl.java 1993 2008-08-25 10:45:32Z thomas.diesler@jboss.com $

import org.jboss.bpm.NotImplementedException;
import org.jboss.bpm.model.EventBuilder;
import org.jboss.bpm.model.FlowObject;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.GatewayBuilder;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.MessageBuilder;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.ProcessBuilder;
import org.jboss.bpm.model.Property;
import org.jboss.bpm.model.TaskBuilder;
import org.jboss.bpm.model.Assignment.AssignTime;
import org.jboss.bpm.model.Expression.ExpressionLanguage;
import org.jboss.bpm.model.Gateway.GatewayType;
import org.jboss.bpm.model.Task.TaskType;
import org.jboss.bpm.runtime.ExecutionHandler;
import org.jboss.bpm.runtime.FlowHandler;
import org.jboss.bpm.runtime.SignalHandler;

/**
 * The ProcessBuilder can be used to dynamically build a {@link Process}.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
public class ProcessBuilderImpl implements ProcessBuilder
{
  protected ProcessImpl proc;
  protected FlowObjectImpl flowObject;

  public ProcessBuilderImpl()
  {
  }

  protected ProcessBuilderImpl(ProcessImpl proc, FlowObjectImpl flowObject)
  {
    this.proc = proc;
    this.flowObject = flowObject;
  }

  public ProcessBuilder addProcess(String procName)
  {
    proc = new ProcessImpl(procName);
    return this;
  }

  public ProcessBuilder addProcess(Process addProc)
  {
    ProcessImpl procImpl = (ProcessImpl)addProc;
    for(Message msg : procImpl.getMessages())
    {
      getProcessInternal().addMessage(msg);
    }
    for(FlowObject fo : procImpl.getFlowObjects())
    {
      getProcessInternal().addFlowObject(fo);
    }
    return this;
  }

  public Process getProcess()
  {
    ProcessImpl internalProc = getProcessInternal();
    internalProc.create(internalProc);
    return internalProc;
  }

  public Process getProcessForInclude()
  {
    return getProcessInternal();
  }

  public ProcessBuilder addSequenceFlow(String targetName)
  {
    if (flowObject instanceof SingleOutFlowSetterSupport)
    {
      SingleOutFlowSetterSupport outFlow = (SingleOutFlowSetterSupport)flowObject;
      outFlow.setOutFlow(new SequenceFlowImpl(targetName));
    }
    else if (flowObject instanceof MultipleOutFlowSetterSupport)
    {
      MultipleOutFlowSetterSupport outFlow = (MultipleOutFlowSetterSupport)flowObject;
      outFlow.addOutFlow(new SequenceFlowImpl(targetName));
    }
    else if (flowObject instanceof Gateway)
    {
      GatewayBuilder gwBuilder = new GatewayBuilderImpl(getProcessInternal(), flowObject);
      gwBuilder.addGate(targetName);
    }
    else
    {
      throw new IllegalStateException("Cannot add a sequence flow to: " + flowObject);
    }
    return this;
  }

  public ProcessBuilder addMessageFlow(String targetName)
  {
    throw new NotImplementedException("JBPM-1382", "Message Flow");
  }

  public EventBuilder addStartEvent(String name)
  {
    flowObject = addFlowObject(new StartEventImpl(name));
    return new EventBuilderImpl(getProcessInternal(), flowObject);
  }

  public EventBuilder addEvent(String name)
  {
    flowObject = (FlowObjectImpl)getProcessInternal().getFlowObject(name);
    if (flowObject == null)
      flowObject = addFlowObject(new IntermediateEventImpl(name));
    return new EventBuilderImpl(getProcessInternal(), flowObject);
  }

  public EventBuilder addEndEvent(String name)
  {
    flowObject = (FlowObjectImpl)getProcessInternal().getFlowObject(name);
    if (flowObject == null)
      flowObject = addFlowObject(new EndEventImpl(name));
    return new EventBuilderImpl(getProcessInternal(), flowObject);
  }

  public TaskBuilder addTask(String name)
  {
    return addTask(name, TaskType.None);
  }
  
  public TaskBuilder addTask(String name, TaskType type)
  {
    flowObject = (FlowObjectImpl)getProcessInternal().getFlowObject(name);
    if (flowObject == null)
    {
      if (type == TaskType.None || type == null)
      {
        flowObject = new NoneTaskImpl(name);
      }
      else if (type == TaskType.Service)
      {
        throw new NotImplementedException("JBPM-1652", "Task Type Service");
      }
      else if (type == TaskType.Receive)
      {
        flowObject = new ReceiveTaskImpl(name);
      }
      else if (type == TaskType.Send)
      {
        flowObject = new SendTaskImpl(name);
      }
      else if (type == TaskType.User)
      {
        throw new NotImplementedException("JBPM-1653", "Task Type User");
      }
      else if (type == TaskType.Script)
      {
        throw new NotImplementedException("JBPM-1654", "Task Type Script");
      }
      else if (type == TaskType.Manual)
      {
        throw new NotImplementedException("JBPM-1655", "Task Type Manual");
      }
      else if (type == TaskType.Reference)
      {
        throw new NotImplementedException("JBPM-1656", "Task Type Reference");
      }
      else
      {
        throw new IllegalStateException("Task type: " + type);
      }
      addFlowObject(flowObject);
    }
    return new TaskBuilderImpl(getProcessInternal(), flowObject);
  }

  public GatewayBuilder addGateway(String name, GatewayType type)
  {
    flowObject = (FlowObjectImpl)getProcessInternal().getFlowObject(name);
    if (flowObject == null)
    {
      if (GatewayType.Exclusive == type)
      {
        flowObject = new ExclusiveGatewayDataBasedImpl(name);
      }
      else if (GatewayType.Inclusive == type)
      {
        flowObject = new InclusiveGatewayImpl(name);
      }
      else if (GatewayType.Parallel == type)
      {
        flowObject = new ParallelGatewayImpl(name);
      }
      else if (GatewayType.Complex == type)
      {
        flowObject = new ComplexGatewayImpl(name);
      }
      addFlowObject(flowObject);
    }
    return new GatewayBuilderImpl(getProcessInternal(), flowObject);
  }
  
  public MessageBuilder addMessage(String name)
  {
    MessageBuilder msgBuilder = new MessageBuilderImpl().newMessage(name);
    getProcessInternal().addMessage(msgBuilder.getMessage());
    return msgBuilder;
  }

  public ProcessBuilder addProperty(String name, String value)
  {
    Property prop = new PropertyImpl(name, new ExpressionImpl(value));
    getProcessInternal().addProperty(prop);
    return this;
  }
  
  public ProcessBuilder addAssignment(AssignTime time, ExpressionLanguage lang, String fromExpr, String toProp)
  {
    ExpressionImpl from = new ExpressionImpl(lang, fromExpr);
    PropertyImpl to = new PropertyImpl(toProp, null);
    AssignmentImpl assignment = new AssignmentImpl(time, from, to);
    if (flowObject != null)
      flowObject.addAssignment(assignment);
    else
      getProcessInternal().addAssignment(assignment);
    return this;
  }

  public ProcessBuilder addExecutionHandler(Class<?> clazz)
  {
    return addHandler(clazz);
  }

  public ProcessBuilder addFlowHandler(Class<?> clazz)
  {
    return addHandler(clazz);
  }

  public ProcessBuilder addSignalHandler(Class<?> clazz)
  {
    return addHandler(clazz);
  }

  private ProcessBuilder addHandler(Class<?> handlerClass)
  {
    if (flowObject instanceof HandlerSetterSupport == false)
      throw new IllegalStateException("Current flow object does not support handlers: " + flowObject);

    if (handlerClass != null)
    {
      HandlerSetterSupport hs = (HandlerSetterSupport)flowObject;
      try
      {
        Object handler = handlerClass.newInstance();
        if (handler instanceof ExecutionHandler)
          hs.setExecutionHandler((ExecutionHandler)handler);
        else if (handler instanceof FlowHandler)
          hs.setFlowHandler((FlowHandler)handler);
        else if (handler instanceof SignalHandler)
          hs.setSignalHandler((SignalHandler)handler);
        else
          throw new IllegalArgumentException("Given class is not a supported handler");
      }
      catch (RuntimeException rte)
      {
        throw rte;
      }
      catch (Exception ex)
      {
        throw new IllegalArgumentException("Cannot instanciate handler", ex);
      }
    }
    return this;
  }
  
  private FlowObjectImpl addFlowObject(FlowObjectImpl fo)
  {
    getProcessInternal().addFlowObject(fo);
    return fo;
  }
  
  private ProcessImpl getProcessInternal()
  {
    if (proc == null)
      throw new IllegalStateException("No process available");
    
    return proc;
  }
  
}